package csbase.server.services.schedulerservice;

import csbase.logic.CommandInfo;
import csbase.logic.SGAInfo;
import csbase.logic.SGASet;
import csbase.server.services.schedulerservice.filter.SGAFilterByCommandRequirements;

import java.rmi.RemoteException;
import java.util.*;

/**
 * Poltica de escalonamento que privilegia os SGAs com maior porcentagem de CPU
 * disponvel.
 *
 * @author valeria
 *
 */
public class PercCPUPolicy implements SchedulerPolicyInterface {
	@Override
	public Map<CommandInfo, SGASet> chooseServer(List<CommandInfo> commands,
			List<SGASet> servers) {

		HashMap<String, SGASet> originalServers = new HashMap<String, SGASet>();

		List<SGASet> updatedServers = new ArrayList<SGASet>(servers);

		BalancedCommandDistribution bdc = new BalancedCommandDistribution(
				servers);

		Map<CommandInfo, SGASet> allocation = new LinkedHashMap<CommandInfo, SGASet>();

		for (SGASet sga : servers) {
			originalServers.put(sga.getName(), sga);
		}

		List<CommandInfo> greedyCommands = getCommandsOrderedByGreediness(commands);

		for (CommandInfo cmd : greedyCommands) {
			List<SGASet> sgas;
			List<SGASet> filteresSGAs = SGAFilterByCommandRequirements.filter(cmd,
					updatedServers);
			if (filteresSGAs == null || filteresSGAs.isEmpty()) {
				continue;
			}
			sgas = getSortedSGAByCPU(filteresSGAs);

			SGASet choosedServer = bdc.chooseLightest(sgas);
			allocation.put(cmd, originalServers.get(choosedServer.getName()));

			SGASet choosedServerCopy = new SGASet(updateSGAInfo(cmd,
					choosedServer.getMainInfo()), choosedServer.getName(),
					choosedServer.getEnabled(), choosedServer.getAlive(),
					choosedServer.hasDiskAccess(),
					choosedServer.isBackoffExpired(),
					choosedServer.getJobsInfo(), choosedServer.isCSFSEnabled(),
					null, System.currentTimeMillis());

			updatedServers.remove(choosedServer);
			updatedServers.add(choosedServerCopy);

		}
		return allocation;
	}

	private SGAInfo[] updateSGAInfo(CommandInfo cmd, SGAInfo info) {
		int ramInfo = 0;
		try {
			double sgaFreeRam = info.getRAMFreeMemoryMb();
			double commandUseRam = cmd.getConfigurator().getMemoryAmount();
			ramInfo = (int) (sgaFreeRam - commandUseRam);
		} catch (RemoteException e) {
		}
		SGAInfo newInfo = new SGAInfo(info.getHostName(), info.getPlatformId(),
				info.getNumProcessors(), ramInfo, 0, info.getClockSpeedMHz(),
				info.getFileSeparator(), info.getProjectRootDirectory(),
				info.getAlgorithmRootDirectory(),
				info.getSandboxRootDirectory(), null);
		newInfo.setAlive(true);
		newInfo.setRAMFreeMemory(ramInfo);
		Set<String> requirements = info.getRequirements();
		for (String req : requirements)
			newInfo.addRequirement(req);
		SGAInfo[] infos = new SGAInfo[] { newInfo };
		return infos;
	}

	/**
	 * Ordena os SGAs de acordo com as suas porcentagens de CPU livre.
	 *
	 * @param serversNames
	 *            Lista de servidores disponveis
	 *
	 * @return Lista de SGAs ordenados de forma decrescente pela quantidade de
	 *         CPU livre
	 */
	private List<SGASet> getSortedSGAByCPU(List<SGASet> servers) {
		List<SGASet> sgasSets = new ArrayList<SGASet>();
		sgasSets.addAll(servers);
		Collections.sort(sgasSets, new SGASetCPUComparator());
		return sgasSets;
	}

	private List<CommandInfo> getCommandsOrderedByGreediness(
			List<CommandInfo> commands) {
		List<CommandInfo> orderedCopy = new ArrayList<CommandInfo>();
		for (CommandInfo cmd : commands) {
			orderedCopy.add(cmd);
		}
		Collections.sort(orderedCopy,
				Collections.reverseOrder(new CommandGreedinessComparator()));
		return orderedCopy;
	}

}


