package csbase.servlet;

import java.io.BufferedInputStream;
import java.io.FileInputStream;
import java.io.FileNotFoundException;
import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;
import java.io.PrintWriter;
import java.net.URL;
import java.net.URLDecoder;
import java.rmi.ConnectException;
import java.rmi.Naming;
import java.rmi.NotBoundException;

import javax.servlet.ServletConfig;
import javax.servlet.ServletException;
import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;

import csbase.remote.ClientRemoteLocator;
import csbase.remote.HttpServiceInterface;
import csbase.remote.ServerEntryPoint;

/**
 * Servlet de download.
 * 
 * @author Tecgraf/PUC-Rio
 */
public class DownloadServlet extends AbstractServlet {
  /**
   * MIME-type para arquivos CSV (FIXME:  isso mesmo?)
   */
  private static final String CSV_CONTENT_TYPE = "application/zip";
  /**
   * MIME-type para arquivos texto.
   */
  private static final String TEXT_CONTENT_TYPE = "text/plain";

  /**
   * {@inheritDoc}
   */
  @Override
  public void init(ServletConfig config) throws ServletException {
    super.init(config);

    // para que o jasperreports no tente abrir o relatrio em uma aplicao X
    System.setProperty("java.awt.headless", "true");
  }

  /**
   * O Servlet espera uma URL do tipo:
   * http://servidor:8080/documentservice/download/numero/download
   * 
   * @param request requisio
   * @param response resposta
   * 
   * @throws ServletException em caso de falha do servlet
   */
  @Override
  protected void doGet(HttpServletRequest request, HttpServletResponse response)
    throws ServletException {
    if (!hasPropertiesFile()) {
      if (!loadPropertiesFile()) {
        messageError(response,
          "Arquivo de propriedades System.properties no foi encontrado.", null);
        return;
      }
    }
    try {
      /* Obtem os parametros */
      String reqParameters = request.getPathInfo();
      String[] params = reqParameters.split("/", 3);
      if (params == null) {
        return;
      }
      String accessCode = params[1];

      /* Decodifica o path */
      String filePath = null;
      try {
        String[] path = params[2].split("/");
        if ((path != null) && (path.length >= 3)) {
          String pathDecoded = "";
          filePath = "";
          for (int i = 1; i < path.length - 1; i++) {
            pathDecoded = URLDecoder.decode(path[i], "UTF-8");
            filePath = filePath + pathDecoded + "/";
          }
          pathDecoded = URLDecoder.decode(path[path.length - 1], "UTF-8");
          filePath = filePath + pathDecoded;
        }
      }
      catch (Exception e) {
        filePath = null;
      }

      /* Obtem a comunicacao com o servidor rmi */
      ClientRemoteLocator.server =
        (ServerEntryPoint) Naming
          .lookup(getRMIpath() + ServerEntryPoint.LOOKUP);
      ClientRemoteLocator.httpService =
        ClientRemoteLocator.server.fetchHttpService();
      final HttpServiceInterface httpService = ClientRemoteLocator.httpService;
      /* Testa a comunicacao */
      if (httpService == null) {
        messageError(response, "Problema na comunicacao com o servidor", null);
        return;
      }
      /* Verifica se a url e' valida */
      String absolutePath = httpService.getFilePath(accessCode, filePath);
      if (absolutePath == null) {
        messageError(response, "Acesso Negado", null);
        return;
      }
      switch (httpService.getDownloadType(accessCode)) {
        case HttpServiceInterface.URL_DOWNLOAD_TYPE:

          // caminho do arquivo vindo como URL
          sendFile(httpService.getFileType(accessCode, filePath), response,
            new URL(absolutePath).openStream());
          break;
        case HttpServiceInterface.FILE_DOWNLOAD_TYPE:

          // caminho do arquivo normal (UNIX)
          sendFile(httpService.getFileType(accessCode, filePath), response,
            new FileInputStream(absolutePath));
          break;
        case HttpServiceInterface.TEXT_DOWNLOAD_TYPE:

          // texto em geral
          response.setContentType(TEXT_CONTENT_TYPE);
          PrintWriter pwtxt = response.getWriter();
          pwtxt.println(httpService.getText(accessCode));
          break;
        case HttpServiceInterface.CSV_DOWNLOAD_TYPE:

          // texto csv(campos separados por vrgula)
          response.setContentType(CSV_CONTENT_TYPE);
          PrintWriter pwcsv = response.getWriter();
          pwcsv.println(httpService.getText(accessCode));
          break;
        case HttpServiceInterface.UNKNOWN_DOWNLOAD_TYPE:

          // tipo deconhecido
          break;
        default:
          break;
      }
    }
    catch (ConnectException e) {
      messageError(response, "Conexo recusada pelo servidor RMI", e);
    }
    catch (FileNotFoundException e) {
      messageError(response, "Erro na leitura do arquivo", e);
    }
    catch (NotBoundException e) {
      messageError(response, "Problema na comunicacao com o servidor", e);
    }
    catch (Exception e) {
      messageError(response, "Ocorreu uma exceo.", e);
    }
  }

  /**
   * Monta uma mensagem de erro no formato html.
   * 
   * @param response objeto resposta que ser enviado ao cliente.
   * @param message mensagem a ser encapsulada no objeto resposta.
   * @param ex exceo
   */
  private void messageError(HttpServletResponse response, String message,
    Exception ex) {
    try {
      String localMessage = message;
      if (ex != null) {
        localMessage += ": " + ex.getLocalizedMessage();
      }
      response.sendError(HttpServletResponse.SC_INTERNAL_SERVER_ERROR,
        localMessage);
      if (ex != null) {
        ex.printStackTrace();
      }
    }
    catch (IOException e) {
      e.printStackTrace();
    }
  }

  /**
   * Envia o arquivo para o browser,
   * 
   * @param mimeType mime type
   * @param response resposta
   * @param fin stream de entrada
   * 
   * @throws IOException em caso de erro de I/O.
   */
  private void sendFile(String mimeType, HttpServletResponse response,
    InputStream fin) throws IOException {

    String mType = mimeType;
    if (mType == null) {
      mType = "application/octet-stream";
    }

    response.setContentType(mType);
    OutputStream out = response.getOutputStream();
    InputStream in = new BufferedInputStream(fin);
    int i;
    byte[] b = new byte[1024];
    while ((i = in.read(b)) != -1) {
      out.write(b, 0, i);
    }
    fin.close();
    out.flush();
    out.close();
  }
}
