package csbase.test.unit.service;

import org.junit.Before;

import csbase.server.Server;
import csbase.server.ServerException;
import csbase.server.ServiceManager;
import io.jsonwebtoken.lang.Assert;
import junit.framework.Test;

/**
 * A classe <code>CSBaseTestSetup</code> que  responsvel por:
 * 
 * <ul>
 * <li> Iniciar uma instncia de testes do CSBase Server antes da execuo dos
 * mtodos de teste dos servios;
 * <li> Carregar as configuraes iniciais atravs do mtodo
 * <code>loadTestData</code>
 * <li> E realizar um shutdown partial ao final da execuo de todos os mtodos
 *  de teste do servidor.
 * </ul>
 * 
 * Esta classe consiste em um "Wrapper" para todos os casos de testes definidos
 * para os servios do CSBase. Esta classe foi necessria pois antes da execuo
 * dos testes de um servio uma instncia do servidor CSBase deve estar no ar.
 */
public abstract class CSBaseTestSetup {

  /** Servidor do CSBase */
  protected Server server;

  public CSBaseTestSetup(Test test) {

  }

  public CSBaseTestSetup() {
    this(null);
  }

  /**
   * Este mtodo ser chamado uma unica vez antes da execucao de um conjunto de
   * metodos de testes contidos em um testcase ou test suite. Os passos seguidos
   * pelo mtodo sao: carrega as propriedades do servidor e cria os dados
   * iniciais de teste, como usurio e permisso.
   */
  @Before
  protected void setUp() throws Exception {
    server = createServer();
    initServer();
    loadTestData();
  }

  /**
   * Cria o servidor para teste. Deve ser implementado em cada um dos
   * sistema, pois no podemos iniciar servidor no CSBase.
   * 
   * @throws ServerException
   */
  protected abstract Server createServer() throws ServerException;

  /**
   * Inicia o servidor de testes.
   */
  protected void initServer() {
    boolean success = server.start();
    Assert.isTrue(success, "No foi possvel iniciar o servidor."
      + "\nPor favor consulte o log para maiores detalhes.");

    ServiceManager srvManager = ServiceManager.getInstance();
    Assert.notNull(srvManager, "Gerente de Servios fora do ar");
  }

  /**
   * Desliga o servidor de testes.
   */
  protected void shutDown() {
    server.shutdown();
  }

  /**
   * Aqui podemos criar alguns dados para teste. Por exemplo, usurios,
   * permisso, projeto que sero necessrios para testes de outros servios.
   */
  public void loadTestData() {
  }

}
