package csbase.util.proxy;

import java.lang.reflect.Constructor;
import java.lang.reflect.InvocationHandler;
import java.lang.reflect.InvocationTargetException;
import java.lang.reflect.Proxy;
import java.rmi.NoSuchObjectException;
import java.rmi.Remote;
import java.rmi.RemoteException;
import java.rmi.server.UnicastRemoteObject;
import java.util.ArrayList;
import java.util.List;

import tecgraf.javautils.core.lng.LNG;
import Acme.Serializable;

/**
 * Facilita a criao de proxies dinmicos.
 * 
 * @author Tecgraf
 */
public class ProxyUtils {
	
  /**
   * Nome da classe (utilizado para internacionalizao a priori).
   */
  final static String className = ProxyUtils.class.getSimpleName();

  /**
   * Cria um proxy que repassa todas as chamadas aos mtodos de um objeto, obj,
   * para o {@link InvocationHandler handler}. O proxy ir implementar todas as
   * interfaces implementadas pelo objeto, obj.
   * 
   * @param obj Objeto que ser substitudo pelo proxy.
   * @param handler Responsvel por tratar as chamadas aos mtodos do objeto.
   * @param <T> tipo do proxy.
   * 
   * @return Um proxy para o objeto.
   */
  @SuppressWarnings("unchecked")
  public static <T> T newProxyInstance(Object obj, InvocationHandler handler) {
    Class<?> objClass = obj.getClass();
    ClassLoader cl = objClass.getClassLoader();
    Class<?>[] interfaces = objClass.getInterfaces();
    if (interfaces.length == 0) {
      throw new RuntimeException(LNG.get(className + ".error.runtime.noimplementation"));
    }

    return (T) Proxy.newProxyInstance(cl, interfaces, handler);
  }

  /**
   * Cria e exporta um proxy remoto que implementa todas as interfaces remotas
   * do objeto, obj.
   *
   *
   * @param obj Objeto que ser substitudo pelo proxy. <br>
   *        Ele no precisa implementar {@link Serializable} ou estender
   *        {@link UnicastRemoteObject}. Precisa apenas implementar alguma
   *        interface que estenda {@link Remote}.
   * @param handler Responsvel por tratar as chamadas aos mtodos do objeto.
   * @param port Porta na qual o proxy ser exportado.
   * @param <T> tipo do proxy.
   * 
   * @return Um proxy remoto j exportado.
   *
   * @throws RemoteException em casos de falha de comunicao com servidor.
   */
  @SuppressWarnings("unchecked")
  public static <T> T newRemoteProxyInstance(Object obj,
    InvocationHandler handler, int port) throws RemoteException {
    Class<?> objClass = obj.getClass();
    ClassLoader cl = objClass.getClassLoader();

    // Obtm as interfaces remotas da classe.
    List<Class<?>> remoteInterfaceList = new ArrayList<Class<?>>();
    for (Class<?> anInterface : objClass.getInterfaces()) {
      if (Remote.class.isAssignableFrom(anInterface)) {
        remoteInterfaceList.add(anInterface);
      }
    }
    if (remoteInterfaceList.size() == 0) {
      throw new IllegalArgumentException(LNG.get(className + ".illegalarg.noimplementation.remote"));
    }
    Class<?>[] remoteInterfaces = remoteInterfaceList.toArray(new Class[0]);

    RemoteInvocationHandler remoteHandler =
      new RemoteInvocationHandler(obj, handler);
    remoteHandler.export(port);

    return (T) Proxy.newProxyInstance(cl, remoteInterfaces, remoteHandler);
  }

  /**
   * Remove o proxy remoto, proxy, da execuo do RMI.<br>
   * O proxy j no poder mais aceitar chamadas RMI.
   * 
   * @param proxy Proxy que no poder mais receber chamadas RMI.
   * @throws NoSuchObjectException Se o objeto remoto no estiver exportado.
   */
  public static void unexportRemoteProxy(Object proxy)
    throws NoSuchObjectException {
    Object handler = Proxy.getInvocationHandler(proxy);
    if (handler instanceof RemoteInvocationHandler) {
      ((RemoteInvocationHandler) handler).unexport();
    }
    else {
      throw new IllegalArgumentException(LNG.get(className + ".ilegalarg.proxy.notremote"));
    }
  }

  /**
   * Caso a classe tenha sido anotada com
   * {@link csbase.util.proxy.InvocationHandlers}, ser criado o
   * {@link InvocationHandler} indicado no valor da anotao, sendo passado o
   * objeto como parmetro do construtor. Caso contrrio, ser retornado um
   * {@link Invoker} que ir invocar os mtodos do objeto fazendo
   * {@code method.invoke(obj, args);}.
   * 
   * @param obj Objeto para o qual ser criado o {@link InvocationHandler}.
   * 
   * @return Um {@link InvocationHandler} para tratar as chamadas aos mtodos de
   *         objetos de um determinado tipo.
   * 
   * @throws InstantiationException caso no tenha sido possvel instanciar o
   *         {@link InvocationHandler} definido na anotao.
   * @throws IllegalAccessException caso no se tenha acesos ao construtor
   *         padro do {@link InvocationHandler} definido na anotao.
   * @throws NoSuchMethodException caso o {@link InvocationHandler} no tenha um
   *         construtor que receba um objeto como parmetro.
   * @throws SecurityException caso ocorra um erro de segurana ao tentar
   *         executar o construtor do {@link InvocationHandler}.
   * @throws InvocationTargetException caso ocorra um erro ao invocar o
   *         construtor do {@link InvocationHandler} passando o objeto como
   *         argumento.
   * @throws IllegalArgumentException caso ocorra um erro ao invocar o
   *         construtor do {@link InvocationHandler} passando o objeto como
   *         argumento.
   * 
   * @see csbase.util.proxy.InvocationHandlers
   */
  public static InvocationHandler createInvocationHandler(Object obj)
    throws InstantiationException, IllegalAccessException, SecurityException,
    NoSuchMethodException, IllegalArgumentException, InvocationTargetException {

    InvocationHandler handler = new Invoker(obj);

    Class<?> objType = obj.getClass();

    /*
     * Caso a classe do objeto esteja anotada com {@link InvocationHandlers},
     * instancia os {@link InvocationHandler handlers}, de trs pra frente,
     * passando obj e o ltimo {@link InvocationHandler handler} criado, at
     * chegar ao primeiro da lista.
     */
    if (objType.isAnnotationPresent(InvocationHandlers.class)) {
      InvocationHandlers annotation =
        objType.getAnnotation(InvocationHandlers.class);

      Class<? extends InvocationHandler>[] handlersClasses = annotation.value();
      // Cria os anteriores passando sempre o ltimo como entrada pro anterior.
      for (int inx = handlersClasses.length - 1; inx >= 0; inx--) {
        // Pega a classe do anterior.
        Class<? extends InvocationHandler> handlerClass = handlersClasses[inx];
        // Cria o anterior passando o ltimo criado como segundo parmetro.
        Constructor<? extends InvocationHandler> cons =
          handlerClass.getConstructor(Object.class, InvocationHandler.class);
        handler = cons.newInstance(obj, handler);
      }
    }
    return handler;
  }
}
