package validations;

import java.io.BufferedReader;
import java.io.BufferedWriter;
import java.io.File;
import java.io.FileReader;
import java.io.FileWriter;
import java.io.IOException;
import java.util.List;

import tecgraf.javautils.core.io.FileUtils;
import validations.util.ValidatorUtils;
import csbase.server.services.serverservice.ServerService;

/**
 * Altera o arquivo de propriedades e o arquivo de repositrio do
 * {@link ServerService servio de servidores}.
 * 
 * @author Tecgraf / PUC-Rio
 */
public class TrustedServer2ServerInfoValidation extends AbstractValidation {

  /** Diretrio de persistncia, a partir do qual o repositrio ser salvo. */
  private static final String PERSISTENCY_DIR_PROPERTY =
    "Persistency.newDirectory";
  /**
   * Nome da propriedade da validao que indica o caminho, relativo ao
   * repositrio, para o novo repositrio.
   */
  private static final String NEW_SERVER_REPOSITORY_PROPERTY =
    "ServerService.serverRepositoryName";
  /**
   * Nome da propriedade da validao que indica o caminho, relativo ao
   * repositrio, para o antigo repositrio.
   */
  private static final String OLD_SERVER_REPOSITORY_PROPERTY =
    "ServerService.oldServerRepositoryName";
  /**
   * Possvel valor da propriedade que indica o caminho do repositrio. <br>
   * Este valor indica que o servio no est utilizando o repositrio. Apesar
   * disso a checagem do uso do servio ser feita pela existncia do arquivo de
   * propriedades do servio.
   */
  private static final String NULL_PROPERTY_VALUE = "NULL";

  /**
   * Ao a ser executada pelo validador. <br>
   * A ao  determinada durante a inicializao do validador.
   */
  private enum Action {
    /** O validador deve remover o antigo repositrio. */
    REMOVE_OLD_REPOSITORY,
    /** A validao deve criar um novo repositrio baseado nos dados do antigo. */
    CREATE_NEW_REPOSITORY,
    /** Indica que o sistema j est validado. */
    VALIDATE_OK,
    /**
     * Termina a execuo do validador sem dar erro, mas sem marcar o validador
     * como executado.
     */
    TERMINATE
  }

  /** Diretrio de backup. */
  private File backupDir;
  /** Caminho para o diretrio raiz do repositrio. */
  private String persistencyDirPath;

  /**
   * ANTIGO arquivo de REPOSITRIO do servio.<br>
   * Pode ser {@code null} caso o repositrio no esteja sendo utilizado.
   */
  private File oldServersRepositoryFile;
  /**
   * NOVO arquivo de REPOSITRIO do servio.<br>
   * Pode ser {@code null} caso o repositrio no esteja sendo utilizado.
   */
  private File newServersRepositoryFile;

  /** Ao a ser executada pelo validador. */
  private Action action;
  /**
   * Mensagem a ser retornada para o usurio no mtodo
   * {@link #getSuccessMessage(Status)} quando actio == {@link Action#TERMINATE}
   * .
   */
  private String warning;

  /**
   * {@inheritDoc}
   */
  @Override
  protected boolean runsOnlyOnce() {
    return action != Action.TERMINATE;
  }

  /**
   * Inicializa as referncias ao:
   * <ul>
   * <li>diretrio de backup,</li>
   * <li>antigo arquivo de propriedades do servio,</li>
   * <li>novo arquivo de propriedades do servio,</li>
   * <li>antigo arquivo de repositrio do servio,</li>
   * <li>novo arquivo de repositrio do servio.</li>
   * </ul>
   * Os dois ltimos podem ser nulos caso o servio no esteja utilizando o
   * repositrio. {@inheritDoc}
   */
  @Override
  protected boolean init() {
    // Obtm o caminho para o diretrio de persistncia.
    persistencyDirPath = getMandatorySystemProperty(PERSISTENCY_DIR_PROPERTY);
    if (persistencyDirPath == null) {
      return false;
    }

    /*
     * Obtm o caminho para o antigo repositrio. Esse caminho  relativo ao
     * diretrio de persistncia.
     */
    String oldRepPath = getSystemProperty(OLD_SERVER_REPOSITORY_PROPERTY);
    if (null == oldRepPath) {
      /*
       * Se o caminho no for encontrado, considera que o servio de servidores
       * no est sendo utilizado e por isso a validao no dever seguir em
       * frente.
       * 
       * O usurio deve ser informado do fato pois o servio pode estar sendo
       * utilizado e ele simplesmente esqueceu de atribuir valor a propriedade
       * {@value OLD_SERVER_REPOSITORY_PROPERTY}. Levando isso em conta,
       * atribumos a ao {@link Action#TERMINATE} indicando que esse validador
       * no deve ser marcado como executado, permitindo assim que ele rode de
       * novo no futuro.
       */
      action = Action.TERMINATE;
      warning =
        String
          .format(
            "CONSIDEROU-SE, DEVIDO A AUSNCIA DA PROPRIEDADE %s, QUE O SERVIO DE SERVIDORES - %s - NO EST OU NO ESTAVA SENDO UTILIZADO.",
            OLD_SERVER_REPOSITORY_PROPERTY, ServerService.class.getSimpleName());
      return true;
    }
    oldRepPath = oldRepPath.trim();
    if (oldRepPath.equalsIgnoreCase(NULL_PROPERTY_VALUE)) {
      /*
       * Se o caminho para o antigo repositrio for {@value
       * #NULL_PROPERTY_VALUE} ser considerado que no existia um repositrio
       * anterior e o usurio sabe disto. Sendo assim, a validao no precisar
       * seguir em frente e o usurio no precisar ser informado.
       */
      action = Action.VALIDATE_OK;
      return true;
    }
    // OBTM referncia para o que ser o ANTIGO arquivo do REPOSITRIO.
    oldServersRepositoryFile =
      new File(persistencyDirPath + File.separator + oldRepPath);

    // Obtm o diretrio 
    String newRepPath =
      getMandatorySystemProperty(NEW_SERVER_REPOSITORY_PROPERTY);
    if (null == newRepPath) {
      /*
       * Uma vez que o usurio deu valor a propriedade {@value
       * OLD_SERVER_REPOSITORY_PROPERTY}, a propriedade {@value
       * NEW_SERVER_REPOSITORY_PROPERTY} passa a ser obrigatria.
       * 
       * Caso o usurio queira indicar que no ir mais usar o servio de
       * servidores, ele deve atribuir o valor {@value
       * OLD_SERVER_REPOSITORY_PROPERTY} a esta propriedade.
       */
      return false;
    }
    newRepPath = newRepPath.trim();
    /*
     * Se o caminho para o antigo repositrio for {@value #NULL_PROPERTY_VALUE}
     * ser considerado que o usurio no ir mais utilizar o servio de
     * servidores. Sendo assim, o validador deve remover o antigo repositrio.
     */
    if (newRepPath.equalsIgnoreCase(NULL_PROPERTY_VALUE)) {
      action = Action.REMOVE_OLD_REPOSITORY;
    }
    else {
      // OBTM referncia para o que ser o NOVO arquivo do REPOSITRIO.
      newServersRepositoryFile =
        new File(persistencyDirPath + File.separator + newRepPath);

      action = Action.CREATE_NEW_REPOSITORY;
    }

    // Obtm referncia para o diretrio de backup.
    this.backupDir =
      new File(getTempDirPath() + File.separator
        + this.getClass().getSimpleName());

    return true;
  }

  /**
   * {@inheritDoc}
   * 
   * @return <tt>true</tt> se o novo arquivo de propriedades existir, indicando
   *         que nenhuma alterao precisa ser feita; ou <tt>false<tt/>, caso
   *         o contrrio.
   * @throws ValidationException caso o nmero de arquivos de propriedades do
   *         servio seja diferente de um.
   */
  @Override
  protected boolean validate() throws ValidationException {
    if (action == Action.VALIDATE_OK || action == Action.TERMINATE) {
      return true;
    }

    if (!oldServersRepositoryFile.exists()) {
      String errmsg =
        "Erro obtendo o antigo repositrio do servio de servidores.\nO arquivo '%s' no foi encontrado.";
      throw new ValidationException(String.format(errmsg,
        this.oldServersRepositoryFile.getAbsolutePath()));
    }
    return false;
  }

  /**
   * {@inheritDoc}
   */
  @Override
  protected String getStartMessage() {
    StringBuilder sb = new StringBuilder();
    sb.append(
      "Criando um novo repositrio de servidores apontado pela propriedade ")
      .append(NEW_SERVER_REPOSITORY_PROPERTY)
      .append(
        ". O contedo deste repositrio ser copiado do repositrio apontado pela propriedade ")
      .append(OLD_SERVER_REPOSITORY_PROPERTY)
      .append(
        ", porm com o elemento <servidores_confiaveis> renomeado para <servidores>");
    return sb.toString();
  }

  /**
   * {@inheritDoc}
   */
  @Override
  protected void getSpecificFailureMessage(Status status, List<String> errors) {
    switch (status) {
      case VALIDATION_FAILED:
        errors.add("*** O REPOSITRIO DE SERVIDORES AINDA NO FOI ALTERADO.");
        break;

      case PATCH_FAILED:
        errors
          .add("*** OCORREU UM ERRO AO TENTAR ALTERAR O REPOSITRIO DE SERVIDORES.");
        break;

      case INIT_FAILED:
        errors.add("*** FALHA NA INICIALIZAO");
        break;

      default:
        errors.add("ESTADO INVLIDO: " + status.toString());
    }
  }

  /**
   * {@inheritDoc}
   */
  @Override
  protected String getSuccessMessage(Status status) {
    final String className = this.getClass().getSimpleName();
    switch (status) {
      case VALIDATION_OK:
        switch (action) {
          case TERMINATE:
            if (warning != null) {
              final String fmt = "O VALIDADOR %s FOI INTERROMPIDO. %s";
              return String.format(fmt, className, warning);
            }
            final String fmt = "O VALIDADOR %s FOI INTERROMPIDO (SEM MOTIVO).";
            return String.format(fmt, className);

          case VALIDATE_OK:
            return "O REPOSITRIO DE SERVIDORES J EST ALTERADO.";

          default:
            return "O SISTEMA FOI CONSIDERADO VLIDO, "
              + "PORM O VALIDADOR EST EM UM ESTADO INCONSISTENTE "
              + action.name() + ".";
        }
      case PATCH_OK:
        return "*** O REPOSITRIO DE SERVIDORES FOI ALTERADO COM SUCESSO.";

      default:
        return "ESTADO INVLIDO: " + status.toString();
    }
  }

  /**
   * {@inheritDoc}
   */
  @Override
  protected boolean backupData() {
    if (!backupDir.exists() && !backupDir.mkdirs()) {
      logger.severe("Erro criando diretrio de backup '"
        + backupDir.getAbsolutePath() + "'");
      return false;
    }

    if (oldServersRepositoryFile != null && oldServersRepositoryFile.exists()) {
      File bkpRepository =
        new File(backupDir, oldServersRepositoryFile.getName());
      if (!ValidatorUtils.copyFile(oldServersRepositoryFile, bkpRepository,
        logger, true)) {
        logger.severe(String.format(
          "Erro salvando o backup do arquivo '%s' no diretrio '%s'.",
          oldServersRepositoryFile.getAbsolutePath(),
          backupDir.getAbsolutePath()));
        return false;
      }
    }

    return true;
  }

  /**
   * {@inheritDoc}
   */
  @Override
  protected boolean restoreBackup() {
    if (!backupDir.exists()) {
      logger.severe("Erro, o diretrio de backup '"
        + backupDir.getAbsolutePath() + "' no foi encontrado.");
      return false;
    }

    if (oldServersRepositoryFile != null && oldServersRepositoryFile.exists()) {
      File bkpRepository =
        new File(backupDir, oldServersRepositoryFile.getName());
      if (!ValidatorUtils.copyFile(bkpRepository, oldServersRepositoryFile,
        logger, true)) {
        logger
          .severe(String.format(
            "Erro restaurando arquivo '%s' para o diretrio '%s'",
            bkpRepository.getAbsolutePath(),
            oldServersRepositoryFile.getParent()));
        return false;
      }
    }

    /*
     * Tendo os arquivos restaurados com sucesso, removemos o diretrio de
     * backup.
     */
    FileUtils.delete(backupDir);
    return true;
  }

  /**
   * {@inheritDoc}
   */
  @Override
  protected boolean applyPatch() {
    switch (action) {
      case CREATE_NEW_REPOSITORY:
        return createNewRepositoryFile();
      case REMOVE_OLD_REPOSITORY:
        return oldServersRepositoryFile.delete();
      default:
        logger.severe(String.format("Erro, ao %s no implementada.",
          action.name()));
        return false;
    }
  }

  /**
   * Cria o novo arquivo de repositrio baseado no antigo.
   * 
   * @return <tt>true</tt> se o arquivo foi criado com sucesso ou no existia
   *         arquivo antigo para ser alterado. Caso contrrio, retorna
   *         <tt>false</tt>.
   */
  private boolean createNewRepositoryFile() {
    boolean sameFile =
      oldServersRepositoryFile.getPath().equals(
        newServersRepositoryFile.getPath());
    /*
     * Se o arquivo antigo e novo repositrio de servidores so o mesmo arquivo,
     * cria-se um arquivo temporrio como novo repositrio para fazer a cpia e
     * depois renome-lo de volta.
     */
    if (sameFile) {
      String tmpPrefix = oldServersRepositoryFile.getName() + ".";
      String tmpSuffix = ".tmp";
      File tmpDir = oldServersRepositoryFile.getParentFile();
      try {
        newServersRepositoryFile =
          File.createTempFile(tmpPrefix, tmpSuffix, tmpDir);
      }
      catch (IOException e) {
        e.printStackTrace();
        return false;
      }
      /*
       * Marca para ser apagado ao fim da execuo, para caso haja algum
       * problema durante a cpia, o arquivo temporrio no seja mantido.
       */
      newServersRepositoryFile.deleteOnExit();
    }

    // Copia os dados, alterando eles, do antigo para o novo repositrio.  
    BufferedReader in = null;
    BufferedWriter out = null;
    try {
      in = new BufferedReader(new FileReader(oldServersRepositoryFile));
      out = new BufferedWriter(new FileWriter(newServersRepositoryFile));
      for (String line = in.readLine(); line != null; line = in.readLine()) {
        out.write(line.replaceAll("(</?)servidores_confiaveis>",
          "$1servidores>"));
        out.newLine();
      }
      out.flush();
    }
    catch (Exception e) {
      e.printStackTrace();
      return false;
    }
    finally {
      if (in != null) {
        try {
          in.close();
        }
        catch (Exception e) {
          logger.exception("Erro tentando fechar o arquivo '"
            + this.oldServersRepositoryFile.getAbsolutePath() + "'", e);
        }
      }

      if (out != null) {
        try {
          out.close();
        }
        catch (Exception e) {
          logger.exception("Erro tentando fechar o arquivo '"
            + this.newServersRepositoryFile.getAbsolutePath() + "'", e);
        }
      }
    }

    // Se o antigo e o novo repositrio eram os mesmos...
    if (sameFile) {
      // Renomeia o arquivo temporrio para o repositrio esperado.
      return newServersRepositoryFile.renameTo(oldServersRepositoryFile);
    }

    // Caso contrrio, remove o antigo repositrio.
    if (!oldServersRepositoryFile.delete()) {
      logger.warning(String.format(
        "Erro removendo o antigo repositrio representado pelo arquivo '%s'.",
        oldServersRepositoryFile.getAbsolutePath()));
    }
    return true;
  }

  /**
   * {@inheritDoc}
   */
  @Override
  protected void finish() {
  }
}
