/**
 * $Id$
 */
package validations.util;

import java.io.BufferedInputStream;
import java.io.BufferedOutputStream;
import java.io.Console;
import java.io.File;
import java.io.FileInputStream;
import java.io.FileOutputStream;
import java.io.IOException;
import java.util.HashMap;
import java.util.Map;
import java.util.logging.Level;
import java.util.logging.Logger;

import validations.AbstractValidation;
import validations.Validator;
import csbase.util.FileSystemUtils;

/**
 * Mtodos utilitrios usado pelas validaes.
 * 
 * @see Validator
 * @see AbstractValidation
 * 
 * @author Tecgraf
 */
public abstract class ValidatorUtils {

  /**
   * Enumerao para opes.
   * 
   * @see ValidatorUtils#readConfirmation(Console, Option)
   */
  public static enum Option {
    /**
     * Opo "sim".
     */
    YES("s"),
    /**
     * Opo "no".
     */
    NO("n");

    /**
     * String associada  opo.
     */
    private final String value;

    /**
     * Construtor.
     * 
     * @param value string associada  opo
     */
    Option(String value) {
      this.value = value;
    }
  }

  /**
   * Copia um arquivo para outro.
   * 
   * ref.:
   * <code>http://nadeausoftware.com/articles/2008/02/java_tip_how_read_files_quickly</code>
   * 
   * OBS.: no  a forma mais eficiente de se copiar um arquivo, mas tambm no
   *  a pior. Talvez possa ir pro JavaUtils...
   * 
   * @param inputFile arquivo de origem
   * @param outputFile arquivo de destino. Se for realmente um arquivo, o
   *        contedo da entrada  gravado sobre o contedo do mesmo; se for um
   *        diretrio, o contedo  copiado para um arquivo com mesmo nome que o
   *        de entrada. Em ambos os casos, o arquivo de sada  criado caso no
   *        exista
   * @param logger logger para registro de ocorrncias. Pode ser null.
   * @param skipExisting indica se devemos ignorar arquivos j existentes no
   *        destino. Se for <code>true</code> arquivos j existentes no sero
   *        sobrescritos, e a cpia indicar sucesso (retornara
   *        <code>true</code>)
   * @return true se a cpia foi bem-sucedida
   */
  public static boolean copyFile(File inputFile, File outputFile,
    Logger logger, boolean skipExisting) {
    if (!inputFile.exists()) {
      logger.severe(inputFile.getAbsolutePath() + " no existe");
      return false;
    }
    if (inputFile.isDirectory()) {
      logger.severe("arquivo de entrada " + inputFile.getAbsolutePath()
        + "  um diretrio");
      return false;
    }
    File outFile;
    if (outputFile.isDirectory()) {
      outFile =
        new File(outputFile.getAbsolutePath() + File.separatorChar
          + inputFile.getName());
    }
    else {
      outFile = outputFile;
    }
    /*
     * se no queremos sobrescrever e o arquivo de destino j existe, retornamos
     * sinalizando sucesso
     */
    if (outFile.exists()) {
      String path;
      try {
        path = outFile.getCanonicalPath();
      }
      catch (IOException e) {
        path = outFile.getAbsolutePath();
      }
      if (skipExisting) {
        logger.fine("Ignorando arquivo j existente " + path);
        return true;
      }
      logger.fine("Sobrescrevendo arquivo " + path);
    }

    try {
      /*
       * no verificamos o retorno da chamada abaixo, pois se o arquivo j
       * exisitir ser sobrescrito de qualquer forma
       */
      outFile.createNewFile();
    }
    catch (IOException e) {
      logger.log(Level.SEVERE, "Erro criando arquivo arquivo de sada"
        + outFile.getAbsolutePath(), e);
      return false;
    }

    /*
     * neste ponto  garantido que a entrada e a sada so arquivos que existem
     */
    BufferedInputStream in = null;
    BufferedOutputStream out = null;
    try {
      in = new BufferedInputStream(new FileInputStream(inputFile));
      out = new BufferedOutputStream(new FileOutputStream(outFile));
      byte[] buffer = new byte[8192];
      int nread = in.read(buffer, 0, buffer.length);
      while (nread != -1) {
        out.write(buffer, 0, nread);
        nread = in.read(buffer, 0, buffer.length);
      }
    }
    catch (Exception e) {
      logger.log(Level.SEVERE, e.getLocalizedMessage(), e);
      return false;
    }
    finally {
      try {
        if (in != null) {
          in.close();
        }
        if (out != null) {
          out.close();
        }
      }
      catch (IOException e) {
        logger.log(Level.SEVERE, e.getLocalizedMessage(), e);
        return false;
      }
    }
    return true;
  }

  /**
   * Copia vrios arquivos para um diretrio.
   * 
   * @param files array com os arquivos a serem copiados
   * @param destination diretrio de destino
   * @param logger logger para registro de ocorrncias. Pode ser null.
   * @param skipExisting indica se devemos ignorar arquivos j existentes no
   *        destino. Se for <code>true</code> arquivos j existentes no sero
   *        sobrescritos, e a cpia indicar sucesso (retornara
   *        <code>true</code>)
   * @return true se todos os arquivos foram copiados com sucesso. Caso alguma
   *         cpia tenha falhado o processo  interrompido
   * 
   * @throws IllegalArgumentException se o destino no for um diretrio
   */
  public static boolean copyFiles(File[] files, File destination,
    Logger logger, boolean skipExisting) {
    if (!destination.isDirectory()) {
      throw new IllegalArgumentException(destination.getAbsolutePath()
        + " no  um diretrio");
    }
    for (File file : files) {
      if (!copyFile(file, destination, logger, skipExisting)) {
        return false;
      }
    }
    return true;
  }

  /**
   * Copia um arquivo para outro. Se o arquivo j existe no destino, este ser
   * sobrescrito.
   * 
   * @param inputFile path para o arquivo de entrada
   * @param outputFile path para o arquivo de sada
   * @param logger logger para registro de ocorrncias. Pode ser null.
   * @return true se a cpia foi bem-sucedida
   * 
   * @see #copyFile(File, File, Logger, boolean)
   */
  public static boolean copyFile(String inputFile, String outputFile,
    Logger logger) {
    return ValidatorUtils.copyFile(new File(inputFile), new File(outputFile),
      logger, false);
  }

  /**
   * Cria um diretrio caso este ainda no exista.
   * <p>
   * Este mtodo encapsula {@link File#mkdir()} porque este retorna
   * <code>false</code> caso o diretrio j exista, e queremos retornar
   * <code>true</code> neste caso.
   * 
   * @param path path para o novo diretrio
   * @return <code>true</code> se o diretrio j existia ou, caso contrrio, se
   *         foi criado com sucesso
   */
  public static boolean mkDir(String path) {
    File dir = new File(path);
    try {
      if (dir.exists()) {
        return dir.isDirectory();
      }
      return dir.mkdir();
    }
    catch (SecurityException e) {
      return false;
    }
  }

  /**
   * Verifica se um diretrio existe e possui um subdiretrio especfico.
   * 
   * @param dir diretrio
   * @param subDirName nome do subdiretrio
   * @return <code>true</code> se o diretrio existe e possui o subdiretrio
   */
  public static boolean hasChildDir(File dir, String subDirName) {
    return FileSystemUtils.dirExists(dir)
      && FileSystemUtils.dirExists(dir.getAbsolutePath() + File.separatorChar
        + subDirName);
  }

  /**
   * Verifica se um arquivo existe no diretrio especificado.
   * 
   * @param dir diretrio
   * @param fileName nome do arquivo
   * @return <code>true</code> se o arquivo existe no diretrio
   */
  public static boolean hasChildFile(File dir, String fileName) {
    File child = new File(dir, fileName);
    return FileSystemUtils.dirExists(dir) && child.exists() && child.isFile();
  }

  /**
   * Obtm um mapa com os subdiretrios de um determinado diretrio, onde a
   * chave  o nome de cada subdiretrio.
   * 
   * @param dir diretrio de referncia
   * @return mapa com os subdiretrios do diretrio de referncia, onde a chave
   *          o nome de cada subdiretrio
   * @throws IllegalArgumentException se o diretrio de referncia no 
   *         realmente um diretrio
   */
  public static Map<String, File> getSubdirsMap(File dir) {
    Map<String, File> subDirsMap = new HashMap<String, File>();
    File[] subDirs = FileSystemUtils.getSubDirs(dir);
    if (subDirs == null || subDirs.length == 0) {
      return subDirsMap;
    }

    for (File subDir : subDirs) {
      final String subDirName = subDir.getName();
      subDirsMap.put(subDirName, subDir);
    }
    return subDirsMap;
  }

  /**
   * L uma linha do console.
   * 
   * @param console console
   * @param prompt texto exibido antes da captura da linha
   * @return linha lida interativamente, sem espaos no incio nem no fim
   */
  public static String readLine(Console console, String prompt) {
    if (console == null) {
      throw new IllegalArgumentException("console no pode ser null");
    }
    return console.readLine(prompt).trim();
  }

  /**
   * L do console uma confirmao do usurio para uma pergunta "sim ou no".
   * 
   * @param console console interativo
   * @param defaultOption opo default (i.e. a opo que ser considerada caso
   *        o usurio digite apenas ENTER)
   * @return true se o usurio respondeu "sim", seja digitando 's' ou aceitando
   *         o default quanto este  {@link Option#YES}. Qualquer resposta
   *         diferente de 's' ou 'n' equivale a 'n'.
   */
  public static boolean readConfirmation(Console console, Option defaultOption) {
    String option =
      readLine(console, String.format("  (s/n) [%s] ", defaultOption.value));
    if (option.isEmpty()) {
      return defaultOption == Option.YES;
    }
    return option.equals(Option.YES.value);
  }

  /**
   * Cpia de diretrios
   * 
   * @param srcDir origem
   * @param dstDir destino
   * @return indicativo de sucesso.
   */
  static public boolean copyDirectory(File srcDir, File dstDir) {
    return copyDirectory(srcDir, dstDir, null);
  }

  /**
   * Cpia de diretrios
   * 
   * @param srcDir origem
   * @param dstDir destino
   * @param logger o logger para resultados
   * @return indicativo de sucesso.
   */
  static public boolean copyDirectory(File srcDir, File dstDir, Logger logger) {
    if (srcDir.isDirectory()) {
      if (!dstDir.exists()) {
        if (!dstDir.mkdirs()) {
          final String dstPath = dstDir.getAbsolutePath();
          logger.severe("Impossvel criar novo diretrio: " + dstPath);
          return false;
        }
      }
      final String files[] = srcDir.list();
      for (int i = 0; i < files.length; i++) {
        final File srcFile = new File(srcDir, files[i]);
        final File dstFile = new File(dstDir, files[i]);
        final boolean copied = copyDirectory(srcFile, dstFile, logger);
        if (!copied) {
          final String dstPath = dstFile.getAbsolutePath();
          logger.severe("Falha na cpia de item: " + dstPath);
          return false;
        }
      }
    }
    else {
      // Ocorre somente no primero passo da recurso.
      if (!srcDir.exists()) {
        final String srcPath = srcDir.getAbsolutePath();
        logger.severe("Arquivo ou dir. no encontrado: " + srcPath);
        return false;
      }
      final boolean copied = copyFile(srcDir, dstDir, logger, false);
      if (!copied) {
        final String srcPath = srcDir.getAbsolutePath();
        final String dstPath = dstDir.getAbsolutePath();
        final String err = "Falha na cpia de " + srcPath + " -> " + dstPath;
        logger.severe(err);
        return false;
      }
    }
    return true;
  }
}
