------------------------------------------------------------------------
-- API para criao de um novo mdulo SGA.
------------------------------------------------------------------------
--
-- Esse arquivo possui o "esqueleto" das funes que so necessrias
-- para implementar um novo SGA. A configurao do n (em sgad-cnf.lua),
-- dever usar o atributo loadlib para indicar o nome do arquivo que
-- contm o cdigo desenvolvido. Esse atributo faz com que o 
-- sga-daemon carregue e execute as funes implementadas. Por exemplo,
-- supondo que o arquivo com a nova API possua o nome NovaAPI.lua, a
-- configurao do n seria da seguinte forma:
--
-- Node{
--   name = "nomeDaMaquina",
--   platform_id = "SGANovo",
--   loadlib = "NovaAPI",
-- }
-- 
-- A escolha de configurar o SGA como Node ou como Grid vai depender de
-- como voc deseja que este SGA aparea na interface do usurio. Se
-- o usurio deve ver o SGA Boinc como uma nica mquina, voc deve
-- configur-lo como Node. Porm, se o Boinc possui vrios ns e voc
-- deseja que o usurio possa monitor-los, voc deve configur-lo
-- como Grid. 
-- 
-- Grid {
--   name = "nomeDaMaquina",
--   loadlib = "NovaAPI",
--   default = {
--     platform_id = "SGANovo",
--   },
-- }
--
-- Voc encontra mais detalhes em:
-- https://jira.tecgraf.puc-rio.br/confluence/pages/viewpage.action?pageId=37814618
------------------------------------------------------------------------

------------------------------------------------------------------------

--
-- Especificao das funes pblicas
--

------------------------------------------------------------------------------
-- Incializa o mdulo para execuo. Essa funo deve ser chamada antes 
-- de qualquer outra funo desta biblioteca.
--
-- Retorna verdadeiro caso o mdulo seja inicializado com sucesso. Em caso 
-- de erro, retorna nil e a mensagem de erro.
--
-- ok, err = open()
------------------------------------------------------------------------------
open = function ()
  return true, nil
end

------------------------------------------------------------------------------
-- Termina a execuo do mdulo. Os processos disparados que ainda estejam
-- rodando no so afetados por essa chamada. Essa funo no retorna valores.
--
-- close()
------------------------------------------------------------------------------
close = function()
end

------------------------------------------------------------------------------
--
-- Funes de consulta  configurao de servidor.
--
-- Em SGAs que so clusters, o parmetro server indica o n sobre o
-- qual se deseja obter a informao.
------------------------------------------------------------------------------
------------------------------------------------------------------------------
-- Retorna o nmero de processadores no servidor especificado. Caso o servidor
-- seja nil, assume localhost. Em caso de erro, retorna nil e a mensagem de
-- erro.
--
-- cpus, err = getnumcpus(server)
------------------------------------------------------------------------------
getnumcpus = function (server)
   return 1, nil
end

------------------------------------------------------------------------------
-- Retorna a quantidade em bytes de memria no servidor especificado.
-- O retorno  uma tabela contendo o total de memria RAM e de Swap, 
-- no formato:
--    { ram = xxx, swap = yyy }
--
-- Caso o servidor seja nil, assume localhost.
--
-- Em caso de erro, retorna nil e a mensagem de erro.
--
-- mem, err = getmemory(server)
------------------------------------------------------------------------------
getmemory = function (server)
   return { ram = 0, swap = 0 }, nil
end

------------------------------------------------------------------------------
--
-- Funes de monitorao de servidor.
--
------------------------------------------------------------------------------
------------------------------------------------------------------------------
-- Retorna a taxa mdia de ocupao de CPU do ltimo minuto no servidor
-- especificado. Esse valor considera o nmero de processadores que o
-- servidor possui.
--
-- Por exemplo, caso a mtrica de ocupao seja o nmero de processos
-- na fila de prontos, este nmero estar dividido pela quantidade 
-- de processadores. Caso o servidor seja nil, assume localhost. 
-- 
-- O parmetro maxold indica o tempo mximo que a informao pode ter.
-- Caso maxold seja nil,  assumido zero e a informao ser buscada 
-- no servidor em questo.
-- 
-- Em caso de erro, retorna nil e a mensagem de erro.  
--
-- load, err = getcpuload(server, maxold)
------------------------------------------------------------------------------
getcpuload = function (server, maxold)
   return 0, nil
end

------------------------------------------------------------------------------
-- Retorna a mdia de ocupao de memria do ltimo minuto no servidor
-- especificado. O retorno  uma tabela contendo o total em bytes de
-- memria RAM e de Swap, no formato:
--   { ram = xxx, swap = yyy }
--
-- Caso o servidor seja nil, assume localhost.
--
-- O parmetro maxold indica o tempo mximo que a informao pode ter.
-- Caso maxold seja nil,  assumido zero e a informao ser buscada 
-- no servidor em questo. 
-- 
-- Em caso de erro, retorna nil e a mensagem de erro.
--
-- load, err = getmemoryload(server, maxold)
------------------------------------------------------------------------------
getmemoryload = function (server, maxold)
   return { ram=0, swap=0 }, nil
end


----------------------------------------------------------------------
-- Retorna o nmero de jobs que esto em execuo no servidor
-- especificado.
-- 
-- Caso o servidor seja nil, assume localhost.
--
-- O parmetro maxold indica o tempo mximo que a informao pode ter.
-- Caso maxold seja nil,  assumido zero e a informao ser buscada 
-- no servidor em questo.
-- 
-- Em caso de erro, retorna nil e a mensagem de erro.  
--
-- njobs, err = getnumberofjobs(server, maxold)
----------------------------------------------------------------------
getnumberofjobs = function(server, maxold)
   return 0, nil 
end


------------------------------------------------------------------------------
--
-- Funes de execuo, monitorao e controle de processos.
--
------------------------------------------------------------------------------
------------------------------------------------------------------------------
-- Executa um comando no servidor especificado e retorna um handle que
--  a referncia para o comando no contexto desta biblioteca. Pode ter
-- qualquer formato.
--
-- Parmetros:
-- . id - um identificador nico para este comando.
-- . command - o comando a ser executado.
-- . infile - nome de um arquivo de onde a entrada padro ser
--   direcionada. (opcional)
-- . outfile - nome de um arquivo para onde a sada padro ser
--   direcionada. (opcional)
-- . errorfile - nome de um arquivo para onde a sada de erro ser
--   direcionada. (opcional)
--
-- * A sada padro e de erro podem ser direcionadas para o mesmo arquivo.
--
-- Caso o servidor seja nil,assume localhost.
--
-- Em caso de erro, retorna nil e a mensagem de erro.
--
-- handle, err = executecommand(identifier, command, infile, outfile,
--   errorfile, server)
------------------------------------------------------------------------------
executecommand = function(id, command, infile, outfile, errorfile, server)
   local handle ={ id = id, command = command}
   return handle, nil
end

------------------------------------------------------------------------------
-- Interrompe um processo iniciado atravs da funo executecommand. Retorna
-- verdadeiro caso tenha sucesso. A definio de sucesso  que uma chamada a
-- getcommandstatus para o mesmo processo retornar o valor FINISHED. Retorna
-- falso caso o processo no termine.
-- 
-- Em caso de erro, retorna nil e a mensagem de erro.
-- 
-- ok, err = killcommand(handle)
------------------------------------------------------------------------------
killcommand = function (handle)
   return true, nil
end

------------------------------------------------------------------------------
-- Retorna uma tabela Lua com os dados do comando que precisam ser persistidos.
-- Estes dados devem ser suficientes para que um comando seja
-- 'recuperado' aps a reinicializao do SGA, atravs da chamada 
-- retrievecommandhandle.
--
-- Em caso de erro, retorna nil e a mensagem de erro.
--
-- pdata, err = getcommandpersistentdata(handle)
------------------------------------------------------------------------------
getcommandpersistentdata = function (handle)
  return handle, nil
end

------------------------------------------------------------------------------
-- A partir dos dados persistidos, retorna um handle que
--  a referncia para o comando no contexto desta biblioteca. Pode
-- ter qualquer formato.
--
-- Em caso de erro, retorna nil e a mensagem de erro.
--
-- handle, err = retrievecommandhandle(pdata)
------------------------------------------------------------------------------
retrievecommandhandle = function (pdata)
   return pdata, nil
end

------------------------------------------------------------------------------
-- Libera todos os recursos relacionados ao processo especificado. Esta funo
-- precisa ser chamada aps o trmino do processo para que eventuais recursos
-- alocados a ele possam ser liberados. Aps a chamada desta funo, o handle
-- do processo no poder mais ser utilizado.
--
-- Em caso de erro, retorna nil e a mensagem de erro.
--
-- ok, err = releasecommandinfo(handle)
------------------------------------------------------------------------------
releasecommandinfo = function (handle)
   return true, nil
end

------------------------------------------------------------------------------
-- Retorna o identificador do comando recebido na funo executecommand.
--
-- Em caso de erro, retorna nil e a mensagem de erro.
--
-- id, err = getcommandid(handle)
------------------------------------------------------------------------------
getcommandid = function (handle)
   return handle.id, nil
end

------------------------------------------------------------------------------
-- Retorna um identificador nico do processo iniciado atravs da funo
-- executecommand.
--
-- Em caso de erro, retorna nil e a mensagem de erro.
--
-- pid, err = getcommandpid(handle)
------------------------------------------------------------------------------
getcommandpid = function (handle)
   return 0, nil
end

------------------------------------------------------------------------------
-- Retorna o estado de um processo iniciado atravs da funo executecommand.
--
-- O parmetro maxold indica o tempo mximo que a informao pode ter.
-- Caso maxold seja nil,  assumido zero e a informao ser buscada no
-- servidor que estiver executando o comando.
--
-- O retorno pode ser RUNNING, NOT_RESPONDING, WAITING ou FINISHED. 
--
-- Em caso de erro, retorna nil e a mensagem de erro.
--
-- status, err = getcommandstatus(handle, maxold)
------------------------------------------------------------------------------
getcommandstatus = function (handle, maxold)
   return servermanager.FINISHED, nil
end

------------------------------------------------------------------------------
-- Retorna  a mquina que est executando o comando definido por handle.
-- O parmetro maxold indica o
-- tempo mximo que a informao pode ter. Caso maxold seja nil,  assumido
-- zero e a informao ser buscada no servidor que estiver executando o
-- comando. Em caso de erro, retorna nil e a mensagem de erro.
--
-- host, err = getcommandexechost(handle, maxold) 
------------------------------------------------------------------------------
getcommandexechost = function (handle, maxold)
   return "host unknown", nil
end

------------------------------------------------------------------------------
-- Retorna a taxa mdia de ocupao de CPU do ltimo minuto pelo processo
-- especificado. Esse valor considera o nmero de processadores que o 
-- servidor possui. Por exemplo, caso o processo execute em apenas um 
-- processador, este nmero estar dividido pela quantidade de processadores.
-- Caso o servidor seja nil, assume localhost.
--
-- O parmetro maxold indica o tempo mximo que a informao pode ter.
-- Caso maxold seja nil,  assumido zero e a informao ser buscada 
-- no servidor que estiver executando o comando.
--
-- Em caso de erro, retorna nil e a mensagem de erro.
--
-- load, err = getcommandcpuload(handle, maxold)
------------------------------------------------------------------------------
getcommandcpuload = function (handle, maxold)
   return 0, nil
end

------------------------------------------------------------------------------
-- Retorna a mdia de ocupao de memria do ltimo minuto pelo processo
-- especificado. O retorno  uma tabela contendo o total em bytes de
-- memria RAM e Swap, no formato:
--     { ram = xxx, swap = yyy }
--
-- O parmetro maxold indica o tempo mximo que a informao pode ter. Caso
-- maxold seja nil,  assumido zero e a informao ser buscada no servidor
-- que estiver executando o comando.
--
-- Em caso de erro, retorna nil e a mensagem de erro.
--
-- load, err = getcommandmemoryload(handle, maxold)
------------------------------------------------------------------------------
getcommandmemoryload = function (handle, maxold)
   return {ram=0, swap=0}, nil
end

------------------------------------------------------------------------------
-- Retorna os tempos de execuo do processo especificado. O retorno  uma
-- tabela contendo os tempos de usurio, sistema e total (tempo de parede),
-- no formato:
--               { user = xxx, system = yyy, elapsed = zzz }
--
-- O parmetro maxold indica o tempo mximo que a informao pode ter. Caso
-- maxold seja nil,  assumido zero e a informao ser buscada no servidor
-- que estiver executando o comando.
--
-- Em caso de erro, retorna nil e a mensagem de erro.
--
-- time, err = getcommandtimeusage(handle, maxold)
------------------------------------------------------------------------------
getcommandtimeusage = function (handle, maxold)
   return {
     user = 0,
     system = 0,
     elapsed = 0,
   }, nil
end

------------------------------------------------------------------------------
-- Retorna todas as informaes sobre um determinado comando. O retorno
--  uma tabela contendo os campos:
--
-- pid - identificador nico do processo iniciado atravs da funo executecommand.
-- command - o comando executado.
-- execHost - a mquina que est executando o comando.
-- state - o estado de um processo.
-- memoryRamSizeMb - o total de memria RAM usada pelo comando. (Em Mb)
-- memorySwapSizeMb - o total de memria SWAP usada pelo comando. (Em Mb)
-- CPUPerc - a taxa mdia de ocupao de CPU do ltimo minuto pelo comando.
-- CPUTimeSec - o tempo de CPU usado pelo comando.
-- wallTimeSec - o tempo total usado pelo comando.
-- userTimeSec - o tempo de usurio usado pelo comando.
-- systemTimeSec - o tempo de sistema usado pelo comando.
--
-- O parmetro maxold indica o tempo mximo que a informao pode ter.
-- Caso maxold seja nil,  assumido zero e a informao ser buscada 
-- no servidor que estiver executando o comando.
--
-- Em caso de erro, retorna nil e a mensagem de erro.
--
-- load, err = getcommandallinfo(handle, maxold)
------------------------------------------------------------------------------
getcommandallinfo = function (handle, maxold)
   local allinfo = {
       pid = 0,
       command = handle.command,
       execHost = "unknown",
       state = servermanager.FINISHED,
       memoryRamSizeMb = 0,
       memorySwapSizeMb = 0,
       CPUPerc = 0,
       CPUTimeSec = 0,
       wallTimeSec = 0,
       userTimeSec = 0,
       systemTimeSec = 0,
   }
   return allinfo, nil
end

------------------------------------------------------------------------------
--
-- Funes de consulta  clusters.
--
------------------------------------------------------------------------------
------------------------------------------------------------------------------
-- Obtm as informaes de todos os processos em execuo no SGA.
-- ** Em desenvolvimento **
------------------------------------------------------------------------------
getjobsinfo = function()
   return nil
end

------------------------------------------------------------------------
-- Retorna uma tabela com os ns do servidor especificado.
--
-- Caso o servidor seja nil, assume localhost.
--
-- Em caso de erro, retorna nil e a mensagem de erro.
--
-- nodes, err = getnodes(server)
------------------------------------------------------------------------
getnodes = function (server)
   return { "localhost" }, nil
end

------------------------------------------------------------------------------
--
-- Funes de log de histrico
--
------------------------------------------------------------------------------
------------------------------------------------------------------------------
-- Indica se o histrico deve ser ativado ou no para o SGA em questo.
--
-- setHistoric(historic)
------------------------------------------------------------------------------
setHistoric = function (historic)
  servermanager._enableHistoric = historic
end

------------------------------------------------------------------------------
-- Retorna true se o histrico deve ser ativado para o sga em questo,
-- ou false caso contrrio.
--
-- enableHistoric = getHistoric()
------------------------------------------------------------------------------
getHistoric = function ()
  return servermanager._enableHistoric
end

------------------------------------------------------------------------
------------------------------------------------------------------------
servermanager = {
  -------------
  -- private --
  -------------
  _enableHistoric = false,

  -----------------------
  -- Constantes do mdulo
  -----------------------
  RUNNING = 0,
  NOT_RESPONDING = 1,
  WAITING = 2,
  FINISHED = 3,

  ERROR_CODE = -1,

  -------------
  -- public --
  -------------
  now = now, -- Definida em C nos binrios para Unix.
  sleep = sleep, -- Definida em C nos binrios para Unix.

  -- Funes de controle do mdulo:
  open = open,
  close = close,

  -- Funes de consulta  configurao de servidor:
  getnumcpus = getnumcpus,
  getmemory = getmemory,

  -- Funes de monitorao de servidor:
  getcpuload = getcpuload,
  getmemoryload = getmemoryload,
  getnumberofjobs = getnumberofjobs,

  -- Funes de execuo, monitorao e controle de processos:
  executecommand = executecommand,
  killcommand = killcommand,
  getcommandpersistentdata = getcommandpersistentdata,
  retrievecommandhandle = retrievecommandhandle,
  releasecommandinfo = releasecommandinfo,
  getcommandid = getcommandid,
  getcommandpid = getcommandpid,
  getcommandstatus = getcommandstatus,
  getcommandexechost = getcommandexechost,
  getcommandcpuload = getcommandcpuload,
  getcommandmemoryload = getcommandmemoryload,
  getcommandtimeusage = getcommandtimeusage,
  getcommandallinfo = getcommandallinfo, -- OPCIONAL

  -- Funes para clusters:
  getjobsinfo = getjobsinfo,
  getnodes = getnodes, -- OPCIONAL
  
  -- Funes de log de histrico:
  setHistoric = setHistoric, -- OPCIONAL
  getHistoric = getHistoric, -- OPCIONAL
}
