package csdk.v1_0.api.core;

import java.awt.Window;
import java.nio.charset.Charset;
import java.util.Locale;

import csdk.v1_0.api.application.ApplicationException;
import csdk.v1_0.api.application.IMessage;

/**
 * Interface com o ambiente CSDK.
 */
public interface ICSDKEnvironment {

  /**
   * Inicia a execuo de outra aplicao.
   * 
   * @param appId identificador da aplicao.
   * @return o identificador da nova instncia da aplicao ou <code>null</code>
   *         caso no exista aplicao com o identificador definido.
   * 
   * @throws ApplicationException caso ocorra falha ao iniciar a nova aplicao.
   */
  public String runApplication(String appId) throws ApplicationException;

  /**
   * Envia uma mensagem a uma instncia de aplicao em execuo.
   * 
   * @param instanceId o identificador da instncia da aplicao.
   * @param message a mensagem a ser enviada.
   * 
   * @return <code>true</code>, se a instncia foi encontrada e recebeu a
   *         mensagem com sucesso ou <code>false</code>, caso contrrio.
   */
  public boolean sendMessage(String instanceId, IMessage message);

  /**
   * Requisita o encerramento da execuo da aplicao. Este mtodo deve ser
   * chamado caso a aplicao deseje ser terminada a qualquer hora antes da
   * finalizao pelo prprio ambiente. A aplicao precisa estar ativa para que
   * possa ser finalizada.
   */
  public void finishApplication();

  /**
   * Delega para o ambiente o tratamento de uma exceo.
   * 
   * @param error a exceo.
   * @param parent janela que pode ser usada como base para mostrar mensagens de
   *        erro, se necessrio. Aceita <code>null</code>.
   */
  public void handleException(Exception error, Window parent);

  /**
   * Obtm o locale do ambiente.
   * 
   * @return o locale.
   */
  public Locale getLocale();

  /**
   * Obtm o charset do servidor.
   * 
   * @return o charset.
   */
  public Charset getServerCharset();

  /**
   * Obtm o charset do ambiente.
   * 
   * @return o charset.
   */
  public Charset getCharset();

  /**
   * Obtm um contexto do ambiente a partir da sua interface. O contexto pode
   * no estar disponvel no ambiente e, nesse caso, o mtodo retorna
   * <code>null</code>.
   * 
   * @param contextClass a interface do contexto desejado.
   * @return uma instncia do contexto ou <code>null</code> caso este contexto
   *         no esteja disponvel no ambiente.
   */
  public <T extends IContext> T getContext(Class<T> contextClass);

}
