package csdk.v1_0.helper;

import java.awt.Window;

import csdk.v1_0.api.command.CommandException;
import csdk.v1_0.api.command.IAlgorithmTemplate;
import csdk.v1_0.api.command.ICommandContext;
import csdk.v1_0.api.command.ICommandObserver;
import csdk.v1_0.api.command.IFlowTemplate;
import csdk.v1_0.api.core.ICSDKEnvironment;
import csdk.v1_0.api.filesystem.FileSelectionType;
import csdk.v1_0.api.filesystem.IFile;
import csdk.v1_0.api.filesystem.local.ILocalFileSystemContext;
import csdk.v1_0.api.filesystem.local.LocalFileSystemException;
import csdk.v1_0.api.filesystem.project.IProjectContext;
import csdk.v1_0.api.filesystem.project.IProjectObserver;
import csdk.v1_0.api.filesystem.project.ProjectException;
import csdk.v1_0.helper.application.AbstractApplication;

/**
 * Classe abstrata para facilitar o desenvolvimento de aplicaes que acessam a
 * rea de projetos com CSDK.
 */
public abstract class AbstractCSDKApplication extends AbstractApplication
  implements IProjectContext, ILocalFileSystemContext, ICommandContext {

  /**
   * Contexto de acesso  rea de projetos.
   */
  private IProjectContext projectContext;

  /**
   * Contexto de acesso ao sistema de arquivos local ao cliente.
   */
  private ILocalFileSystemContext localFileSystemContext;

  /**
   * Contexto de acesso  execuo de comandos.
   */
  private ICommandContext commandContext;

  /**
   * Construtor.
   * 
   * @param csdkInterface interface padro para o ambiente CSDK.
   */
  public AbstractCSDKApplication(ICSDKEnvironment csdkInterface) {
    super(csdkInterface);
    this.projectContext = getContext(IProjectContext.class);
    this.localFileSystemContext = getContext(ILocalFileSystemContext.class);
    this.commandContext = getContext(ICommandContext.class);
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public final IFile browseProjectFileInOpenMode(String[] fileTypes,
    boolean allowAll, FileSelectionType selectionType, Window frame)
    throws ProjectException {
    checkProjectContext();
    return this.projectContext.browseProjectFileInOpenMode(fileTypes, allowAll,
      selectionType, frame);
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public IFile[] browseMultipleProjectFilesInOpenMode(String[] fileTypes,
    boolean allowAll, FileSelectionType selectionType, Window frame)
    throws ProjectException {
    checkProjectContext();
    return this.projectContext.browseMultipleProjectFilesInOpenMode(fileTypes,
      allowAll, selectionType, frame);
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public final IFile browseProjectFileInSaveMode(String[] fileTypes,
    boolean allowAll, String defaultName, Window frame) throws ProjectException {
    checkProjectContext();
    return this.projectContext.browseProjectFileInSaveMode(fileTypes, allowAll,
      defaultName, frame);
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public final IFile getProjectFile(String[] path, Window window)
    throws ProjectException {
    checkProjectContext();
    return this.projectContext.getProjectFile(path, window);
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public final IFile createProjectFile(String[] path, String name,
    String fileType, Window window) throws ProjectException {
    checkProjectContext();
    return this.projectContext.createProjectFile(path, name, fileType, window);
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public final String getCurrentProjectId() {
    checkProjectContext();
    return this.projectContext.getCurrentProjectId();
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public void addProjectObserver(IProjectObserver observer) {
    checkProjectContext();
    this.projectContext.addProjectObserver(observer);
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public void removeProjectObserver(IProjectObserver observer) {
    checkProjectContext();
    this.projectContext.removeProjectObserver(observer);
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public boolean removeProjectFile(String[] path, Window window)
    throws ProjectException {
    checkProjectContext();
    return this.projectContext.removeProjectFile(path, window);
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public final IFile browseLocalFileInOpenMode(String[] fileTypes,
    boolean allowAll, IFile currentDir, FileSelectionType fileSelectionType,
    Window frame) throws LocalFileSystemException {
    checkFileSystemContext();
    return this.localFileSystemContext.browseLocalFileInOpenMode(fileTypes,
      allowAll, currentDir, fileSelectionType, frame);
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public IFile[] browseMultipleLocalFilesInOpenMode(String[] fileTypes,
    boolean allowAll, IFile currentDir, FileSelectionType fileSelectionType,
    Window frame) throws LocalFileSystemException {
    checkFileSystemContext();
    return this.localFileSystemContext.browseMultipleLocalFilesInOpenMode(
      fileTypes, allowAll, currentDir, fileSelectionType, frame);
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public final IFile browseLocalFileInSaveMode(String[] fileTypes,
    boolean allowAll, String defaultName, IFile currentDir, Window frame)
    throws LocalFileSystemException {
    checkFileSystemContext();
    return this.localFileSystemContext.browseLocalFileInSaveMode(fileTypes,
      allowAll, defaultName, currentDir, frame);
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public IFile getLocalFile(String[] path) throws LocalFileSystemException {
    checkFileSystemContext();
    return this.localFileSystemContext.getLocalFile(path);
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public final IFile createLocalFile(String[] path, String name, boolean isDir)
    throws LocalFileSystemException {
    checkFileSystemContext();
    return this.localFileSystemContext.createLocalFile(path, name, isDir);
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public boolean removeLocalFile(String[] path) throws LocalFileSystemException {
    checkFileSystemContext();
    return this.localFileSystemContext.removeLocalFile(path);
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public final String executeAlgorithm(IAlgorithmTemplate info,
    String description, String sgaServerName, Window owner,
    ICommandObserver observer) throws CommandException {
    checkCommandContext();
    return this.commandContext.executeAlgorithm(info, description,
      sgaServerName, owner, observer);
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public final String executeFlow(IFlowTemplate info, String description,
    String sgaServerName, Window owner, ICommandObserver observer)
    throws CommandException {
    checkCommandContext();
    return this.commandContext.executeFlow(info, description, sgaServerName,
      owner, observer);
  }

  /**
   * Indica se o contexto de projetos est disponvel.
   * 
   * @return <code>true</code> se o contexto estiver disponvel ou
   *         <code>false</code> caso contrrio.
   */
  public boolean isProjectContextAvailable() {
    return this.projectContext != null && this.projectContext.isActive();
  }

  /**
   * Verifica se existe projeto corrente.
   * 
   * @return <code>true</code> caso exista um projeto aberto ou
   *         <code>false</code> caso contrrio.
   */
  public boolean isProjectOpen() {
    return isProjectContextAvailable() && getCurrentProjectId() != null;
  }

  /**
   * Indica se o contexto de sistema de arquivo est disponvel.
   * 
   * @return <code>true</code> se o contexto estiver disponvel ou
   *         <code>false</code> caso contrrio.
   */
  public boolean isFileSystemContextAvailable() {
    return this.localFileSystemContext != null
      && this.localFileSystemContext.isActive();
  }

  /**
   * Indica se o contexto de execuo de comandos est disponvel.
   * 
   * @return <code>true</code> se o contexto estiver disponvel ou
   *         <code>false</code> caso contrrio.
   */
  public boolean isCommandContextAvailable() {
    return this.commandContext != null && this.commandContext.isActive();
  }

  /**
   * Verifica se o contexto de projetos est disponvel, lanando uma exceo
   * {@link ContextNotAvailableException} caso negativo.
   */
  private void checkProjectContext() {
    if (!isProjectContextAvailable()) {
      throw new ContextNotAvailableException(IProjectContext.class);
    }
  }

  /**
   * Verifica se o contexto de sistema de arquivos est disponvel, lanando uma
   * exceo {@link ContextNotAvailableException} caso negativo.
   */
  private void checkFileSystemContext() {
    if (!isFileSystemContextAvailable()) {
      throw new ContextNotAvailableException(ILocalFileSystemContext.class);
    }
  }

  /**
   * Verifica se o contexto de execuo de comandos est disponvel, lanando
   * uma exceo {@link ContextNotAvailableException} caso negativo.
   */
  private void checkCommandContext() {
    if (!isCommandContextAvailable()) {
      throw new ContextNotAvailableException(ICommandContext.class);
    }
  }

  /**
   * {@inheritDoc}
   */
  @Override
  protected void disposeContexts() {
    super.disposeContexts();
    this.commandContext = null;
    this.localFileSystemContext = null;
    this.projectContext = null;
  }
}
