package csdk.v1_0.runner;

import java.nio.charset.Charset;
import java.util.Collection;
import java.util.HashMap;
import java.util.Hashtable;
import java.util.Iterator;
import java.util.Locale;
import java.util.Map;
import java.util.Properties;
import java.util.Set;

import tecgraf.javautils.core.lng.LNG;
import csdk.v1_0.api.application.IMessage;
import csdk.v1_0.api.application.IMessageSender;
import csdk.v1_0.runner.application.DefaultContextFactory;
import csdk.v1_0.runner.application.RunnerApplication;

/**
 * Classe que representa o gerenciador de aplicaes.
 * 
 * Essa classe *no* deve ser usada por desenvolvedores CSDK em suas aplicaes.
 * Ela  de uso exclusivo do ambiente simulado do {@link Runner}.
 */
public class ApplicationManager {

  /**
   * Instncia do runner.
   */
  static private ApplicationManager instance;

  /**
   * Lista de aplicaes em execuo.
   */
  final private Map<String, RunnerApplication> activeApplications =
    new HashMap<String, RunnerApplication>();

  /**
   * Lista de tipos de aplicaes registradas.
   */
  final private Hashtable<String, ApplicationRegistry> registries =
    new Hashtable<String, ApplicationRegistry>();

  /**
   * Fbrica de contextos do Runner.
   */
  private IContextFactory contextFactory;

  /**
   * Propriedades dos contextos.
   */
  private final Properties contextProperties;

  /**
   * Construtor
   */
  private ApplicationManager() {
    this.contextFactory = new DefaultContextFactory();
    this.contextProperties = new Properties();
  }

  /**
   * Retorna o singleton do <code>Runner</code>.
   * 
   * @return a instncia
   */
  final public static ApplicationManager getInstance() {
    if (instance != null) {
      return instance;
    }
    instance = new ApplicationManager();
    return instance;
  }

  /**
   * Termina a execuo de todas as aplicaes em execuo.
   */
  public void finishAllApplications() {
    Collection<RunnerApplication> values = activeApplications.values();
    Iterator<RunnerApplication> iterator = values.iterator();
    while (iterator.hasNext()) {
      RunnerApplication app = iterator.next();
      if (app.finishApplication()) {
        iterator.remove();
      }
    }
  }

  /**
   * Adio na lista de aplicaes ativas
   * 
   * @param application aplicao
   */
  private void activate(RunnerApplication application) {
    logSevere("Application " + application.getInstanceId() + " activated.");
    activeApplications.put(application.getInstanceId(), application);
  }

  /**
   * Adio de um registry.
   * 
   * @param reg registro da aplicao.
   */
  public void addApplicationRegistry(ApplicationRegistry reg) {
    String applicationId = reg.getApplicationId();
    if (registries.containsKey(applicationId)) {
      throw new IllegalArgumentException("double registered id");
    }
    registries.put(applicationId, reg);
  }

  /**
   * Atribui a fbrica de contextos do Runner.
   * 
   * @param contextFactory a fbrica de contextos.
   */
  public void setContextFactory(IContextFactory contextFactory) {
    this.contextFactory = contextFactory;
  }

  /**
   * Obtm a fbrica de contextos do Runner.
   * 
   * @return a fbrica de contextos.
   */
  public IContextFactory getContextFactory() {
    return contextFactory;
  }

  /**
   * @return o conjunto de aplicaes.
   */
  public Set<String> getAllAplicationsIds() {
    return registries.keySet();
  }

  /**
   * Tomada de um id de aplicao com base em um identificador.
   * 
   * @param id id
   * @return o registro.
   */
  public ApplicationRegistry getApplicationRegistry(String id) {
    if (id == null) {
      throw new IllegalArgumentException("invalid null id");
    }
    if (registries.containsKey(id)) {
      return registries.get(id);
    }
    return null;
  }

  /**
   * Verifica se existe uma aplicao com o identificador especificado.
   * 
   * @param id o identificador.
   * @return <code>true</code> se existir aplicao com o identificador ou
   *         <code>false</code> caso contrrio.
   */
  public boolean hasApplicationRegistry(String id) {
    if (id == null) {
      return false;
    }
    return registries.containsKey(id);
  }

  /**
   * Consulta ao locale.
   * 
   * @return o locale
   */
  public Locale getLocale() {
    return LNG.getLocale();
  }

  /**
   * Obtm a instncia da aplicao a partir do seu identificador.
   * 
   * @param id o identificador.
   * @return a instncia da aplicao.
   */
  public RunnerApplication getRunningApp(String id) {
    return activeApplications.get(id);
  }

  /**
   * Obtm a instncia da aplicao a partir do seu identificador.
   * 
   * @param id o identificador do tipo de aplicao.
   * @return a instncia da aplicao.
   */
  public RunnerApplication getRunningApplicationType(String id) {
    for (RunnerApplication app : activeApplications.values()) {
      if (app.getApplicationId().equals(id)) {
        return app;
      }
    }
    return null;
  }

  /**
   * @return indicativo
   */
  public boolean hasActiveApplications() {
    return activeApplications.size() != 0;
  }

  /**
   * Obtm as propriedades dos contextos.
   * 
   * @return as propriedades.
   */
  public Properties getRunnerProperties() {
    return contextProperties;
  }

  /**
   * Escreve uma mensagem de log de nvel grave.
   * 
   * @param text mensagem.
   */
  private void logSevere(String text) {
    CSDKLogger logger = CSDKLogger.getInstance();
    logger.logSevere(text);
  }

  /**
   * Remoo da lista de aplicaes ativas
   * 
   * @param application a aplicao
   */
  public void deactivate(RunnerApplication application) {
    logSevere("Application " + application.getInstanceId() + " deactivated.");
    activeApplications.remove(application.getInstanceId());
  }

  /**
   * Lana uma nova instncia da aplicao com o identificador definido.
   * 
   * @param id identificador da aplicao.
   * @param charset o charset do ambiente.
   * @return a nova instncia da aplicao.
   */
  public RunnerApplication runApplication(String id, Charset charset) {
    ApplicationRegistry reg = getApplicationRegistry(id);
    if (reg == null) {
      logSevere("No application " + id + " found!");
      return null;
    }
    try {
      logSevere("Launching new application " + id + "...");
      RunnerApplication runningApp = getRunningApplicationType(id);
      if (reg.isSingleton() && runningApp != null) {
        return runningApp;
      }
      RunnerApplication wrapperApp = new RunnerApplication(reg, charset);
      wrapperApp.onApplicationStart();
      activate(wrapperApp);
      return wrapperApp;
    }
    catch (Exception e) {
      String msg = e.getMessage();
      if (msg != null) {
        logSevere("Exception detected: " + msg);
      }
      Throwable cause = e.getCause();
      if (cause != null) {
        logSevere("Cause of exception inside application: "
          + cause.getMessage());
        cause.printStackTrace();
      }
      else {
        e.printStackTrace();
      }
      return null;
    }
  }

  /**
   * Envia uma mensagem a uma instncia de aplicao.
   * 
   * @param id o identificador da instncia da aplicao.
   * @param message a mensagem.
   * @param sender identifica a aplicao que enviou a mensagem.
   * @return verdadeiro, se a instncia foi encontrada e recebeu a mensagem com
   *         sucesso ou falso, caso contrrio.
   */
  public boolean sendMessage(String id, IMessage message, IMessageSender sender) {
    RunnerApplication app = getRunningApp(id);
    if (app == null) {
      return false;
    }
    app.onMessageReceived(message, sender);
    return true;
  }

  /**
   * Atribui as propriedades dos contextos.
   * 
   * @param props as propriedades.
   */
  public void addContextProperties(Properties props) {
    this.contextProperties.putAll(props);
  }

}
