package snakegame.session;

import java.util.ArrayList;
import java.util.List;

import snakegame.algorithm.IMoveAlgorithm;
import snakegame.session.listener.MapSizeChangedListener;
import snakegame.session.listener.ModeChangedListener;

/**
 * Classe que representa a sesso do usurio.
 * 
 * @author Tecgraf/PUC-Rio
 */
public class Session {

  /** Tamanho do mapa. */
  private MapSize mapSize;

  /** Estado do jogo. */
  private Mode mode;

  /** Nvel de dificuldade. */
  private Level level;

  /** Algoritmo. */
  private IMoveAlgorithm algorithm;

  /** Ouvintes de mudana de estado do jogo. */
  private List<ModeChangedListener> modeChangedListeners;

  /** Ouvintes de mudana de tamanho do mapa. */
  private List<MapSizeChangedListener> mapSizeChangedListeners;

  /**
   * Construtor.
   * 
   * @param mapSize tamanho do mapa.
   * @param level nvel de dificuldade.
   */
  public Session(MapSize mapSize, Level level) {
    if (mapSize == null) {
      throw new IllegalArgumentException("mapSize no pode ser nulo.");
    }
    if (level == null) {
      throw new IllegalArgumentException("level no pode ser nulo.");
    }
    this.mapSize = mapSize;
    this.level = level;
    this.mode = Mode.OFF;

    modeChangedListeners = new ArrayList<ModeChangedListener>();
    mapSizeChangedListeners = new ArrayList<MapSizeChangedListener>();
  }

  /**
   * Tamanho do mapa.
   * 
   * @return tamanho.
   */
  public MapSize getMapSize() {
    return mapSize;
  }

  /**
   * Estado do jogo.
   * 
   * @return estado do jogo.
   */
  public Mode getMode() {
    return mode;
  }

  /**
   * Nvel de dificuldade.
   * 
   * @return nvel de dificuldade.
   */
  public Level getLevel() {
    return level;
  }

  /**
   * Algoritmo.
   * 
   * @return algoritmo.
   */
  public IMoveAlgorithm getAlgorithm() {
    return algorithm;
  }

  /**
   * Tamanho do mapa.
   * 
   * @param mapSize tamanho.
   */
  public void setMapSize(MapSize mapSize) {
    if (mapSize == null) {
      throw new IllegalArgumentException("mapSize no pode ser nulo.");
    }

    this.mapSize = mapSize;

    for (MapSizeChangedListener listener : mapSizeChangedListeners) {
      listener.perform(mapSize);
    }
  }

  /**
   * Define o estado do jogo.
   * 
   * @param mode estado do jogo.
   */
  public void setMode(Mode mode) {
    if (mode == null) {
      throw new IllegalArgumentException("mode no pode ser nulo.");
    }

    this.mode = mode;

    for (ModeChangedListener listener : modeChangedListeners) {
      switch (mode) {
        case ON:
          listener.modeOn(algorithm);
          break;
        case OFF:
          listener.modeOff();
          break;
      }
    }
  }

  /**
   * Define o nvel de dificuldade.
   * 
   * @param level nvel de dificuldade.
   */
  public void setLevel(Level level) {
    if (level == null) {
      throw new IllegalArgumentException("level no pode ser nulo.");
    }
    this.level = level;
  }

  /**
   * Define o algoritmo.
   * 
   * @param algorithm algoritmo.
   */
  public void setAlgorithm(IMoveAlgorithm algorithm) {
    this.algorithm = algorithm;
  }

  /**
   * Adiciona ouvinte.
   * 
   * @param listener ouvinte.
   */
  public void addMapSizeChangedListener(MapSizeChangedListener listener) {
    mapSizeChangedListeners.add(listener);
  }

  /**
   * Adiciona ouvinte.
   * 
   * @param listener ouvinte.
   */
  public void addModeChangedListener(ModeChangedListener listener) {
    modeChangedListeners.add(listener);
  }
}
