package tecgraf.javautils.gui.print;

import java.awt.BorderLayout;
import java.awt.Color;
import java.awt.Component;
import java.awt.Container;
import java.awt.Dimension;
import java.awt.Graphics;
import java.awt.Graphics2D;
import java.awt.Paint;
import java.awt.event.ActionEvent;
import java.awt.event.ActionListener;
import java.awt.print.PageFormat;
import java.awt.print.Paper;
import java.awt.print.Printable;
import java.awt.print.PrinterException;

import javax.swing.JButton;
import javax.swing.JFrame;
import javax.swing.JOptionPane;
import javax.swing.JPanel;
import javax.swing.JScrollPane;

/**
 * Simulador de impresso. Baseado na premissa de que o Java utiliza a API
 * Java2D para renderizar os componentes de impresso, simula a impresso em uma
 * janela do Swing, o que  menos custoso do que imprimir no papel e mais rpido
 * do que gerar arquivos PDF.
 * 
 * @see PrintSimulatorSample
 * 
 * @author Tecgraf
 */
public class PrintSimulator {

  /** Painel que simula uma pgina impressa */
  private PagePanel pagePanel;
  /** Boto de navegao para a prxima pgina */
  private JButton nextButton;

  /**
   * Construtor simples.
   */
  public PrintSimulator() {
    pagePanel = new PagePanel(this);
  }

  /**
   * Construtor que j permite visualizar a impresso.
   * 
   * @param printable item a ser visualizado.
   */
  public PrintSimulator(Printable printable) {
    this();
    setPrintable(printable);
    print();
  }

  /**
   * Atribui o item a ser visualizado.
   * 
   * @param printable item a ser visualizado.
   */
  public void setPrintable(Printable printable) {
    if (printable == null) {
      throw new IllegalArgumentException("printable == null");
    }
    pagePanel.setPrintable(printable);
  }

  /**
   * Simula a impresso, visualizando o item passado em uma janela Swing.
   */
  public void print() {
    JFrame window = new JFrame("Pgina de teste de impresso");
    window.setDefaultCloseOperation(JFrame.EXIT_ON_CLOSE);
    window.setSize(650, 700);
    window.setLocationRelativeTo(null);
    window.add(new JScrollPane(pagePanel), BorderLayout.CENTER);
    window.add(getButtonBar(), BorderLayout.SOUTH);
    window.setVisible(true);
  }

  /**
   * Retorna a barra de botes.
   * 
   * @return barra de botes.
   */
  private Component getButtonBar() {
    JPanel panel = new JPanel();
    nextButton = new JButton("Prxima pgina >>");
    nextButton.addActionListener(new ActionListener() {
      @Override
      public void actionPerformed(ActionEvent e) {
        pagePanel.nextPage();
      }
    });
    panel.add(nextButton);
    return panel;
  }

  /**
   * Callback a ser chamada quando no houver mais pginas a serem visualizadas.
   * Desabilita o boto de navegao para a prxima pgina.
   */
  public void noSuchPageCallback() {
    nextButton.setEnabled(false);
  }
}

/**
 * Classe representando uma pgina a ser visualizada.
 * 
 * @author Tecgraf
 */
class PagePanel extends JPanel {
  /** Altura do papel */
  private static final double PAPER_HEIGHT = 792.0;
  /** Largura do papel */
  private static final double PAPER_WIDTH = 612.0;
  /** Altura da rea que pode ser desenhada */
  private static final double IMAGEABLE_AREA_HEIGHT = 706.96;
  /** Largura da rea que pode ser desenhada */
  private static final double IMAGEABLE_AREA_WIDTH = 470.27;
  /** Origem horizontal da rea que pode ser desenhada */
  private static final double IMAGEABLE_AREA_X = 70.87;
  /** Origem vertical da rea que pode ser desenhada */
  private static final double IMAGEABLE_AREA_Y = 42.52;

  /** Elemento a ser visualizado */
  private Printable printable;
  /** Formato da pgina */
  private PageFormat pageFormat;
  /** ndice da pgina sendo visualizada */
  private int pageIndex;
  /** Referncia para o simulador de impresso */
  private PrintSimulator simulator;

  /**
   * Construtor.
   * 
   * @param simulator referncia para o simulador de impresso.
   */
  public PagePanel(PrintSimulator simulator) {
    super();
    this.simulator = simulator;
    Paper paper = new Paper();
    paper.setSize(PAPER_WIDTH, PAPER_HEIGHT);
    paper.setImageableArea(IMAGEABLE_AREA_X, IMAGEABLE_AREA_Y,
      IMAGEABLE_AREA_WIDTH, IMAGEABLE_AREA_HEIGHT);
    pageFormat = new PageFormat();
    pageFormat.setPaper(paper);
    pageFormat.setOrientation(PageFormat.PORTRAIT);
    setMaximumSize(new Dimension((int) PAPER_WIDTH, (int) PAPER_HEIGHT));
    setPreferredSize(new Dimension((int) PAPER_WIDTH, (int) PAPER_HEIGHT));
    setMinimumSize(new Dimension((int) PAPER_WIDTH, (int) PAPER_HEIGHT));
    setBackground(Color.BLACK);
    setPrintable(printable);
    pageIndex = 0;
  }

  /**
   * Atribui o item a ser visualizado.
   * 
   * @param printable item a ser visualizado.
   */
  public void setPrintable(Printable printable) {
    this.printable = printable;
  }

  /**
   * 
   * {@inheritDoc}
   */
  @Override
  public void paint(Graphics g) {
    super.paint(g);
    Graphics2D g2d = (Graphics2D) g;
    Paint colorBackup = g2d.getPaint();
    g2d.setPaint(Color.WHITE);
    g2d.fillRect(0, 0, (int) PAPER_WIDTH, (int) PAPER_HEIGHT);
    g2d.setPaint(colorBackup);
    try {
      int result = printable.print(g, pageFormat, pageIndex);
      if (result == Printable.NO_SUCH_PAGE) {
        simulator.noSuchPageCallback();
      }
    }
    catch (PrinterException e) {
      Container parentFrame =
        getParent().getParent().getParent().getParent().getParent().getParent();
      JOptionPane.showMessageDialog(parentFrame, e.getMessage()
        + "\nConsulte a sada do console para ver a pilha.",
        "PrintSimulator - Erro ao visualizar impresso",
        JOptionPane.ERROR_MESSAGE);
      e.printStackTrace();
    }
  }

  /**
   * Navega para a prxima pgina do relatrio.
   */
  public void nextPage() {
    pageIndex++;
    repaint();
  }
}