/*
 * Created on Apr 15, 2005
 */
package tecgraf.javautils.gui.table;

import java.awt.Component;
import java.awt.Dimension;
import java.awt.GridBagConstraints;
import java.awt.GridBagLayout;
import java.awt.Insets;
import java.awt.Rectangle;
import java.awt.event.ActionEvent;
import java.awt.event.ActionListener;
import java.awt.event.KeyAdapter;
import java.awt.event.KeyEvent;
import java.util.Arrays;
import java.util.Comparator;
import java.util.Enumeration;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.regex.Pattern;

import javax.swing.BorderFactory;
import javax.swing.JButton;
import javax.swing.JCheckBox;
import javax.swing.JLabel;
import javax.swing.JPanel;
import javax.swing.JTable;
import javax.swing.JTextField;
import javax.swing.RowFilter;
import javax.swing.RowSorter;
import javax.swing.SortOrder;
import javax.swing.event.EventListenerList;
import javax.swing.event.RowSorterEvent;
import javax.swing.table.TableCellRenderer;
import javax.swing.table.TableColumn;
import javax.swing.table.TableColumnModel;
import javax.swing.table.TableModel;

import tecgraf.javautils.gui.GUIResources;

/**
 * <p>
 * Esta classe representa uma tabela com linhas ordenadas, que pode ser
 * reordenada clicando-se no cabealho de cada uma de suas colunas.<br>
 * A tabela exibir um cone de seta, indicando o sentido da ordenao, no
 * cabealho da coluna que estiver servindo de base para a ordenao. Clicando
 * na mesma coluna, a ordenao ser invertida. Clicando em outra, esta passar
 * a ser base para uma nova ordenao, em sentido crescente.
 * </p>
 * <p>
 * O mtodo {@link #setNoSortStateEnabled(boolean)} oferece uma terceira
 * possibilidade: se essa opo estiver habilitada, quando o usurio tentar
 * reordenar uma coluna atualmente em sentido decrescente, ao invs de passar
 * para o sentido crescente, est perder a ordenao, at a prxima solicitao
 * de ordenao. Isto , ao invs do ciclo binrio "crescente-decrescente",
 * acrescenta-se um terceiro estado, gerando um ciclo "crescente-decrescente-no
 * ordenado". Esta opo foi criada para casos em que o custo de ordenao era
 * considerado muito alto em razo do nmero de linhas a serem ordenadas,
 * prejudicando o desempenho.
 * </p>
 * <p>
 * Se o seu objetivo  apenas instanciar uma {@link SortableTable} para mostrar
 * objetos, d uma olhada na classe {@link ObjectTableBuilder} que foi feita
 * para facilitar este trabalho.
 * </p>
 * 
 * @see SortableTableSample
 */
public class SortableTable extends JTable implements SortableTableListener {

  /**
   * Indica se existe um ajuste de colunas pendente a ser feito. Este controle 
   * necessrio se o mtodo de ajuste tiver sido chamado logo aps a
   * instanciao da classe, pois o clculo da largura s pode ser realizado a
   * partir do momento em que tiver sido estabelecida a largura de exibio da
   * tabela.
   */
  private boolean pendingColumnsAdjustment = false;

  /**
   * ndices das colunas que iro repartir um possvel espao remanescente
   * durante um ajuste de colunas.
   */
  private int[] columnIndexesForRemainingSpace;

  /** Indica tooltip habilitado/desabilitado */
  private boolean toolTipEnabled = false;

  /**
   * Cpia dos renderers de clulas originais da tabela, os quais sero
   * sobrescritos se o mtodo {@link #setToolTipEnabled(boolean)} for utilizado.
   * Permite que os renderers originais possam ser repostos se o tooltip for
   * desabilitado.
   */
  private Map<Integer, TableCellRenderer> originalTableCellRenderers;

  /** Classe responsvel por realizar a ordenao das linhas da tabela. */
  private TecTableRowSorter<TableModel> rowSorter;

  /** Indica se o estado "no-ordenado"  ou no permitido. */
  private boolean noSortStateEnabled;

  /**
   * Indica se as clulas devem parecer desabilitadas quando a tabela estiver
   * desabilitada.
   */
  private boolean useDisabledLook = false;

  /**
   * <p>
   * Constri uma instncia da classe, sem especificar o modelo.
   * </p>
   */
  public SortableTable() {
    this(true);
  }

  /**
   * <p>
   * Constri uma instncia da classe, sem especificar o modelo.
   * 
   * @param enableSort indica se a ordenao est ativa (true) ou no (false).
   *        </p>
   */
  public SortableTable(boolean enableSort) {
    if (enableSort) {
      rowSorter = new TecTableRowSorter<TableModel>();
      setRowSorter(rowSorter);
    }
  }

  /**
   * Constri uma instncia da classe.
   * 
   * @param model modelo para essa tabela (se for passado <code>null</code>,
   *        ser criado um <code>DefaultTableModel</code>).
   */
  public SortableTable(TableModel model) {
    this(model, true);
  }

  /**
   * Constri uma instncia da classe.
   * 
   * @param model modelo para essa tabela (se for passado <code>null</code>,
   *        ser criado um <code>DefaultTableModel</code>).
   * @param enableSort indica se a ordenao est ativa (true) ou no (false).
   */
  public SortableTable(TableModel model, boolean enableSort) {
    this(model, enableSort, new TecTableRowSorter<TableModel>(model));
  }

  /**
   * Constri uma instncia da classe.
   * 
   * @param model modelo para essa tabela (se for passado <code>null</code>,
   *        ser criado um <code>DefaultTableModel</code>).
   * @param enableSort indica se a ordenao est ativa (true) ou no (false).
   * @param sorter ordenador. Permite realizar ordenaes diferentes da padro.
   * @param useDisabledLook - se igual a true, os componentes das clulas sero
   *        desabilitados quando a tabela estiver desabilitada, dando  mesma a
   *        real aparncia de desabilitada. Caso contrrio, a tabela ter o
   *        comportamento default para estes casos (as clulas tm aparncia
   *        normal mas no so interativas)
   */
  public SortableTable(TableModel model, boolean enableSort,
    TecTableRowSorter<TableModel> sorter, boolean useDisabledLook) {
    super(model);
    this.useDisabledLook = useDisabledLook;
    if (!enableSort) {
      return;
    }
    rowSorter = sorter;
    sorter.setModel(model);
    setRowSorter(rowSorter);
    if (model instanceof FooterModelWrapper) {
      ((TecTableRowSorter<? extends TableModel>) rowSorter).setNullComparison(
        true);
      for (int i = 0; i < this.getColumnCount(); i++) {
        Comparator<?> actualComp = this.getComparator(i);
        FooterComparatorWrapper<?> compWrapper = new FooterComparatorWrapper(
          this, actualComp);
        this.addSortableTableListener(compWrapper);
        this.setComparator(i, compWrapper);
      }
    }
    sort();
  }

  /**
   * Constri uma instncia da classe.
   * 
   * @param model modelo para essa tabela (se for passado <code>null</code>,
   *        ser criado um <code>DefaultTableModel</code>).
   * @param enableSort indica se a ordenao est ativa (true) ou no (false).
   * @param sorter ordenador. Permite realizar ordenaes diferentes da padro.
   */
  public SortableTable(TableModel model, boolean enableSort,
    TecTableRowSorter<TableModel> sorter) {
    this(model, enableSort, sorter, false);
  }

  /**
   * Substitui o modelo, reaplicando o ordenador.
   * 
   * @param dataModel novo modelo.
   */
  @Override
  public void setModel(TableModel dataModel) {
    super.setModel(dataModel);
    setRowSorter(rowSorter);
    if (rowSorter != null) {
      rowSorter.setModel(dataModel);
    }
  }

  /**
   * Associa um filtro de linha a tabela.
   * 
   * @param filter um critrio para filtrar as linhas da tabela
   */
  public void setRowFilter(
    RowFilter<? super TableModel, ? super Integer> filter) {
    rowSorter.setRowFilter(filter);
  }

  /**
   * Habilita/desabilita o tooltip para exibio de contedo das clulas, til
   * no caso de clulas que estejam em colunas muito estreitas.
   * 
   * @param enabled se true, habilita o tooltip; se false, desativa.
   */
  public final void setToolTipEnabled(boolean enabled) {
    toolTipEnabled = enabled;
    if (enabled) {
      setDefaultToolTipTableCellRenderer();
    }
    else {
      restoreOriginalTableCellRenderers();
    }
  }

  /**
   * <p>
   * Substitui os renderers de coluna da tabela, para que estes passem a exibir
   * um tooltip com o contedo da clula. Esta implementao tem a limitao de
   * sobrescrever quaisquer renderers previamente instalados nas colunas, e de
   * inibir os renderers default, quando no houver nenhum instalado (maiores
   * detalhes sobre o mecanismo de priorizao de renderers em
   * http://java.sun.com
   * /docs/books/tutorial/uiswing/components/table.html#editrender
   * </p>
   * <p>
   * NOTA DE IMPLEMENTAO: Para permitir que a operao seja desfeita, os
   * renderers originais so armazenados em uma estrutura que permite sua
   * posterior recuperao.
   * </p>
   * 
   * @see #setToolTipEnabled(boolean)
   */
  private void setDefaultToolTipTableCellRenderer() {
    Enumeration<TableColumn> it = this.getColumnModel().getColumns();
    originalTableCellRenderers = new HashMap<Integer, TableCellRenderer>();
    while (it.hasMoreElements()) {
      TableColumn column = it.nextElement();
      originalTableCellRenderers.put(column.getModelIndex(), column
        .getCellRenderer());
      column.setCellRenderer(new DefaultToolTipTableCellRenderer());
    }

  }

  /**
   * <p>
   * Restaura os renderers originais das clulas da tabela, sobrescritos pela
   * funcionalidade de "tooltip" (visualizao do contudo das clulas, til
   * para clulas inseridas em colunas muito estreitas).
   * </p>
   * <p>
   * NOTA DE IMPLEMENTAO: Se alguma coluna tiver sido acrescentada aps a
   * cpia dos renderers originais ter sido feita, um eventual renderer nela
   * presente ser removido (sobrescrito com "null").
   * </p>
   * 
   * @see #setToolTipEnabled(boolean)
   */
  private void restoreOriginalTableCellRenderers() {
    if (originalTableCellRenderers == null || originalTableCellRenderers
      .size() == 0) {
      return;
    }
    Enumeration<TableColumn> it = this.getColumnModel().getColumns();
    while (it.hasMoreElements()) {
      TableColumn column = it.nextElement();
      TableCellRenderer originalRenderer = originalTableCellRenderers.get(column
        .getModelIndex());
      column.setCellRenderer(originalRenderer);
    }

  }

  /**
   * Indica se o tooltip para exibio de contedo das clulas est ou no
   * habilitado.
   * 
   * @return true se o tooltip estiver habilitado, false caso contrrio.
   */
  public final boolean getToolTipEnabled() {
    return toolTipEnabled;
  }

  /**
   * Ajusta a altura de todas as linhas da tabela, buscando a menor altura
   * suficiente para exibio do contedo.
   */
  public void adjustRowHeight() {
    int rowCount = getRowCount();
    for (int row = 0; row < rowCount; row++) {
      rowHeight = 0;
      int prefCellHeight = getPreferredCellHeight(row);
      if (prefCellHeight > rowHeight) {
        rowHeight = prefCellHeight;
      }
      rowHeight += getRowMargin();
      setRowHeight(row, rowHeight);
    }
  }

  /**
   * Obtm a altura preferida para as clulas de uma linha, buscando a menor
   * largura suficiente para exibio do contedo de todas as linhas daquela
   * linha.
   * 
   * @param viewRowIndex ndice da linha na tabela.
   * 
   * @return altura em pixels mnima para essa linha, suficiente para exibio
   *         do seu contedo.
   */
  protected int getPreferredCellHeight(int viewRowIndex) {
    int prefRowHeight = 0;
    int colCount = getColumnCount();
    for (int c = 0; c < colCount; c++) {
      TableCellRenderer renderer = getCellRenderer(viewRowIndex, c);
      int modelRowIndex = convertRowIndexToModel(viewRowIndex);
      boolean isSelected = isCellSelected(viewRowIndex, c);
      boolean rowIsLead = (getSelectionModel()
        .getLeadSelectionIndex() == viewRowIndex);
      boolean colIsLead = (getColumnModel().getSelectionModel()
        .getLeadSelectionIndex() == c);
      boolean hasFocus = (rowIsLead && colIsLead) && isFocusOwner();
      Component comp = renderer.getTableCellRendererComponent(this, getModel()
        .getValueAt(modelRowIndex, c), isSelected, hasFocus, viewRowIndex, c);
      int prefCellHeight = comp.getPreferredSize().height;
      if (prefCellHeight > prefRowHeight) {
        prefRowHeight = prefCellHeight;
      }
    }
    return prefRowHeight;
  }

  /**
   * <p>
   * Ajusta a largura de todas as colunas da tabela, buscando a menor largura
   * que seja suficiente para exibio do contedo. Eventuais espaos restantes
   * sero distribudos igualmente entre todas as colunas.
   * </p>
   * <p>
   * A operao de ajuste pode no ser efetuada exatamente no momento da chamada
   * do mtodo, mas sim postergada para um momento mais adequado. Isto ocorre
   * devido  possibilidade do mtodo ser invocado antes da tabela ter
   * configurado suas dimenses finais, o que acontece somente aps ter sido
   * executado um <code>pack</code> ou <code>setVisible</code> em seu
   * <code>Container</code>.
   * </p>
   * <p>
   * Como o clculo da largura das colunas depende da largura da tabela, caso
   * este mtodo tenha sido invocado antes da tabela ter configurado a sua
   * largura final, a execuo  postergada at o momento em que o mtodo
   * {@link #doLayout()}  chamado, pois somente nesse ponto ter-se- certeza de
   * que a tabela j ter suas dimenses finais.
   * </p>
   * 
   * @see #doLayout()
   */
  public void adjustColumnWidth() {
    columnIndexesForRemainingSpace = getColumnIndexes();
    if (!isDisplayable()) {
      pendingColumnsAdjustment = true;
      return;
    }
    doAdjustColumnWidths();
  }

  /**
   * Obtm um array com os ndices de todas as colunas visveis da tabela.
   * 
   * @return array de ndices de todas as colunas visveis da tabela.
   */
  private int[] getColumnIndexes() {
    int[] indexes = new int[getColumnCount()];
    for (int i = 0; i < indexes.length; i++) {
      indexes[i] = i;
    }
    return indexes;
  }

  /**
   * <p>
   * Ajusta a largura de todas as colunas da tabela, buscando a menor largura
   * que seja suficiente para exibio do contedo. Eventuais espaos restantes
   * sero distribudos igualmente entre as colunas especificadas.
   * </p>
   * <p>
   * Esse mtodo pode no executar no momento exato em que  chamado. Maiores
   * detalhes, veja {@link #adjustColumnWidth()}.
   * </p>
   * 
   * @param columnIndexes ndices das colunas que repartiro o espao restante.
   */
  public void adjustColumnWidth(int[] columnIndexes) {
    if (columnIndexes == null || columnIndexes.length == 0) {
      throw new IllegalArgumentException("columnIndexes est vazia");
    }
    for (int i = 0; i < columnIndexes.length; i++) {
      if (columnIndexes[i] < 0 || columnIndexes[i] >= getColumnCount()) {
        throw new IllegalArgumentException("columnIndexes[" + i + "]: "
          + columnIndexes[i]);
      }
    }
    columnIndexesForRemainingSpace = columnIndexes;
    if (!isDisplayable()) {
      pendingColumnsAdjustment = true;
      return;
    }
    doAdjustColumnWidths();
  }

  /**
   * <p>
   * Sobrescreve o <code>doLayout</code> da <code>JTable</code> para executar
   * eventuais ajustes de largura das colunas que tenham ficado pendentes. Isto
   *  necessrio caso o mtodo de ajuste tenha sido chamado antes da largura
   * final da tabela ter sido calculada, o que invalidaria o resultado do
   * mtodo. Quando este mtodo  invocado, a tabela j tem suas dimenses
   * finais, o que justifica sua escolha como ponto de execuo adequado para o
   * mtodo de ajuste.
   * </p>
   * O redesenho da tabela  feito conforme descrito no mtodo
   * {@link JTable#doLayout()}.<br>
   * 
   */
  @Override
  public void doLayout() {
    if (pendingColumnsAdjustment) {
      doAdjustColumnWidths();
    }
    super.doLayout();
  }

  /**
   * <p>
   * Ajusta a largura das colunas da tabela, distribuindo um eventual espao
   * restante entre as colunas especificadas em
   * {@link #columnIndexesForRemainingSpace}.
   * </p>
   * <p>
   * Como o clculo da largura das colunas depende da largura da tabela, esse
   * mtodo s deve ser chamado quando se a tabela j estiver disponvel para
   * exibio, isto , j tenha configuradas suas dimenses finais.
   * </p>
   * 
   * @see #doLayout()
   */
  private void doAdjustColumnWidths() {
    setPreferredColumnWidthToFit();
    setRemainingSpaceToColumns(columnIndexesForRemainingSpace);
    pendingColumnsAdjustment = false;
  }

  /**
   * <p>
   * Ajusta a largura <u>preferencial</u> de todas as colunas da tabela,
   * buscando a menor largura que seja suficiente para exibio do contedo.
   * </p>
   */
  protected void setPreferredColumnWidthToFit() {
    TableColumnModel colModel = getColumnModel();
    int colCount = colModel.getColumnCount();
    for (int i = 0; i < colCount; i++) {
      TableColumn column = colModel.getColumn(i);
      int columnWidth = getPreferredHeaderWidth(column, i);
      int prefCellWidth = getPreferredCellWidth(i);
      if (prefCellWidth > columnWidth) {
        columnWidth = prefCellWidth;
      }
      columnWidth += colModel.getColumnMargin() * 2;
      column.setPreferredWidth(columnWidth);
    }
  }

  /**
   * Obtm a largura preferida para o cabealho da coluna especificada.
   * 
   * @param column referncia para a coluna.
   * @param viewColIndex ndice da coluna na tabela.
   * 
   * @return largura em pixels mnima para esse cabealho, suficiente para
   *         exibio do seu contedo.
   */
  protected int getPreferredHeaderWidth(TableColumn column, int viewColIndex) {
    TableCellRenderer renderer = column.getHeaderRenderer();
    if (renderer == null) {
      renderer = getTableHeader().getDefaultRenderer();
    }
    Component comp = renderer.getTableCellRendererComponent(this, column
      .getHeaderValue(), false, false, 0, viewColIndex);
    return comp.getPreferredSize().width;
  }

  /**
   * Obtm a largura preferida para as clulas de uma coluna, buscando a menor
   * largura suficiente para exibio do contedo de todas as linhas daquela
   * coluna.
   * 
   * @param viewColIndex ndice da coluna na tabela.
   * 
   * @return largura em pixels mnima para essa coluna, suficiente para exibio
   *         do seu contedo.
   */
  protected int getPreferredCellWidth(int viewColIndex) {
    int prefColWidth = 0;
    int rowCount = getRowCount();
    for (int r = 0; r < rowCount; r++) {
      TableCellRenderer renderer = getCellRenderer(r, viewColIndex);
      int modelColIndex = convertColumnIndexToModel(viewColIndex);
      boolean isSelected = isCellSelected(r, viewColIndex);
      boolean rowIsLead = (getSelectionModel().getLeadSelectionIndex() == r);
      boolean colIsLead = (getColumnModel().getSelectionModel()
        .getLeadSelectionIndex() == viewColIndex);
      boolean hasFocus = (rowIsLead && colIsLead) && isFocusOwner();
      Component comp = renderer.getTableCellRendererComponent(this, getModel()
        .getValueAt(r, modelColIndex), isSelected, hasFocus, r, viewColIndex);
      int prefCellWidth = comp.getPreferredSize().width;
      if (prefCellWidth > prefColWidth) {
        prefColWidth = prefCellWidth;
      }
    }
    return prefColWidth + getIntercellSpacing().width * 2;
  }

  /**
   * <p>
   * Ajusta a largura das colunas especificadas para preencher o espao restante
   * na largura da tabela, se houver, durante um ajuste de colunas. O espao
   * restante ser distribudo igualmente entre as colunas especificadas (na
   * prtica, isso no  sempre possvel, devido ao fato de estar-se lidando com
   * valores inteiros. Sobras da diviso so redistribudas entre as primeiras
   * colunas).
   * </p>
   * Se ocorrer a condio oposta, isto , se o espao da tabela no for
   * suficiente para a exibio de todas as colunas, o comportamento padro da
   * <code>JTable</code>, de distribuio do delta negativo entre TODAS as
   * colunas, ser mantido.
   * 
   * @param columnIndexes ndices das colunas que recebero o espao adicional
   *        (se houver).
   */
  protected void setRemainingSpaceToColumns(int[] columnIndexes) {
    TableColumnModel colModel = getColumnModel();
    int colCount = colModel.getColumnCount();
    if (colCount == 0) {
      return;
    }
    // Calcula a largura total PREFERENCIAL das colunas. No  possvel usar o 
    // mtodo TableColumnModel.getTotalColumnWidth pois ele no utiliza o 
    // preferredWidth das colunas, mas sim o width das mesmas.
    int totalPrefWidth = 0;
    Enumeration<TableColumn> columns = colModel.getColumns();
    while (columns.hasMoreElements()) {
      TableColumn col = columns.nextElement();
      totalPrefWidth += col.getPreferredWidth();
    }
    int remaining = getWidth() - totalPrefWidth;
    if (remaining > 0) {
      int indexesCount = columnIndexes.length;
      int quotient = remaining / indexesCount;
      int modulus = remaining % indexesCount;
      for (int i = 0; i < indexesCount; i++) {
        int delta = (modulus-- > 0) ? quotient + 1 : quotient;
        TableColumn column = colModel.getColumn(columnIndexes[i]);
        column.setPreferredWidth(column.getPreferredWidth() + delta);
      }
    }
  }

  /**
   * Ajusta as dimenses da tabela, buscando a menor altura para cada linha e a
   * menor largura para cada coluna suficiente para exibio do seu contedo.
   */
  public void adjustSize() {
    adjustColumnWidth();
    adjustRowHeight();
  }

  /**
   * Ordena a tabela pela primeira coluna, em sentido crescente.
   */
  public void sort() {
    sort(0, SortOrder.ASCENDING);
  }

  /**
   * Ordena a tabela por determinada coluna, em determinado sentido.
   * 
   * @param initialBaseColumnIndex coluna a servir de base para a ordenao
   *        (0-based).
   * @param initialSortOrder sentido da ordenao.
   */
  public void sort(int initialBaseColumnIndex, SortOrder initialSortOrder) {
    sort(new RowSorter.SortKey(initialBaseColumnIndex, initialSortOrder));
  }

  /**
   * Ordena a tabela por quaisquer colunas e sentidos, na sequencia determinada
   * pelos parmetros. Os parmetros consistem em uma lista de qualquer tamanho
   * de {@link RowSorter.SortKey}, classe que representa um par {@code (<ndice
   * da coluna>,<sentido da ordenao>)}, iniciando da coluna 0
   * (0-based). Se o primeiro par (coluna,sentido) for (2,ASCENDING) e o segundo
   * (0, DESCENDING), a tabela ser primeiro ordenada pela terceira coluna em
   * sentido crescente, e a seguir pela primeira coluna em sentido decrescente.
   *
   * @param sortKeys lista de pares {@code (<ndice da coluna>,<sentido da
   * ordenao>)}.
   */
  public void sort(RowSorter.SortKey... sortKeys) {
    if (getRowSorter() == null) {
      throw new IllegalStateException("rowSorter == null");
    }
    getRowSorter().setSortKeys(Arrays.asList(sortKeys));
  }

  /**
   * Retorna o comparador para a coluna especificada (se houver). Caso
   * contrrio, retorna nulo.
   * 
   * @param colIndex ndice da coluna para o qual se deseja o comparador.
   * 
   * @return comparador.
   */
  public Comparator<?> getComparator(int colIndex) {
    TecTableRowSorter<? extends TableModel> sorter =
      (TecTableRowSorter<? extends TableModel>) getRowSorter();
    if (sorter == null) {
      return null;
    }
    return sorter.getComparator(colIndex);
  }

  /**
   * Permite que determinada coluna seja ordenada de forma personalizada.
   * ATENO: Este mtodo no dispara uma ordenao. Se for desejada uma
   * reordenao, deve ser invocado posteriormente o mtodo {@link #sort()} ou
   * suas variantes.
   * 
   * @param colIndex ndice da coluna no modelo.
   * @param comparator comparador a ser usado para ordenao da coluna.
   */
  public void setComparator(int colIndex, Comparator<?> comparator) {
    TecTableRowSorter<? extends TableModel> sorter =
      (TecTableRowSorter<? extends TableModel>) getRowSorter();
    if (sorter != null) {
      sorter.setComparator(colIndex, comparator);
    }
  }

  /**
   * Permite que todas as colunas sejam ordenadas de forma personalizada.
   * ATENO: Este mtodo no dispara uma ordenao. Se for desejada uma
   * reordenao, deve ser invocado posteriormente o mtodo {@link #sort()} ou
   * suas variantes.
   * 
   * @param comparators array de comparadores, a serem atribudos na mesma ordem
   *        s colunas do modelo.
   */
  public void setComparators(Comparator<?>[] comparators) {
    if (comparators == null) {
      throw new IllegalArgumentException("comparators == null");
    }
    TecTableRowSorter<? extends TableModel> sorter =
      (TecTableRowSorter<? extends TableModel>) getRowSorter();
    if (sorter != null) {
      for (int colIndex = 0; colIndex < comparators.length; colIndex++) {
        sorter.setComparator(colIndex, comparators[colIndex]);
      }
    }
  }

  /**
   * Obtm o ndice da coluna (na viso) que est atualmente servindo de base
   * para a ordenao.
   * 
   * @return ndice da coluna-base para a ordenao ou -1 se a tabela no
   *         estiver ordenada.
   */
  public int getSortedColIndexView() {
    if (this.getRowSorter() == null) {
      return -1;
    }
    List<? extends RowSorter.SortKey> sortKeys = this.getRowSorter()
      .getSortKeys();
    if (sortKeys == null || sortKeys.size() < 1) {
      return -1;
    }
    return convertColumnIndexToView(sortKeys.get(0).getColumn());
  }

  /**
   * Retorna o sentido atual de ordenao.
   * 
   * @return sentido atual de ordenao.
   * 
   * @see SortOrder
   */
  public SortOrder getCurrentSortOrder() {
    if (this.getRowSorter() == null) {
      return SortOrder.UNSORTED;
    }
    List<? extends RowSorter.SortKey> sortKeys = this.getRowSorter()
      .getSortKeys();
    if (sortKeys == null || sortKeys.size() < 1) {
      return SortOrder.UNSORTED;
    }
    return sortKeys.get(0).getSortOrder();
  }

  /**
   * Retorna o renderizador adequado para a clula especificada. Garante que, no
   * caso do modelo ser um {@link FooterModelWrapper}, os renderizadores
   * originais para cada tipo de clula sero "decorados" por um
   * {@link FooterRendererWrapper}, o qual tem o papel de extrair o valor das
   * clulas "marcadas" (encapsuladas por uma classe indicando tratar-se de
   * clula de totalizao) para permitir sua correta exibio.
   * 
   * @param row ndice da linha da clula
   * @param column ndice da coluna da clula
   * 
   * @return renderizador adequado para a clula.
   */
  @Override
  public TableCellRenderer getCellRenderer(int row, int column) {
    TableCellRenderer renderer = super.getCellRenderer(row, column);
    if (getModel() instanceof FooterModelWrapper) {
      // TODO Desempenho: verificar se pode ser singleton
      renderer = new FooterRendererWrapper(renderer, useDisabledLook);
    }
    else if (useDisabledLook) {
      /*
       * este teste a rigor no  necessrio, poderamos simplesmente retornar
       * RespectEnabledStateCellRenderer(renderer, useDisabledLook); no o
       * fizemos apenas para no criar um wrapper inutilmente quando
       * useDisabledLook == false
       */
      renderer = new RespectEnabledStateCellRenderer(renderer);
    }
    return renderer;
  }

  /**
   * Indica se o estado "no-ordenado" est ou no habilitado.
   * 
   * @return <code>true</code> se o estado "no-ordenado" for permitido,
   *         <code>false</code> caso contrrio.
   */
  public boolean getNoSortStateEnabled() {
    return this.noSortStateEnabled;
  }

  /**
   * Habilita / desabilita o estado "no-ordenado" da tabela.
   * 
   * @param enable se <code>true</code>, habilita; <code>false</code> caso
   *        contrrio.
   */
  public void setNoSortStateEnabled(boolean enable) {
    this.noSortStateEnabled = enable;
    RowSorter<? extends TableModel> sorter = getRowSorter();
    if (sorter != null && sorter instanceof TecTableRowSorter) {
      ((TecTableRowSorter<? extends TableModel>) sorter).setNoSortEnabled(
        noSortStateEnabled);
    }
  }

  /**
   * Substitui o ordenador/filtrador atual. Habilita / desabilita no novo
   * filtrador o estado "no-ordenado", dependendo do estado atual deste objeto.
   */
  @Override
  public void setRowSorter(RowSorter<? extends TableModel> sorter) {
    super.setRowSorter(sorter);
    if (sorter instanceof TecDefaultRowSorter) {
      ((TecTableRowSorter<? extends TableModel>) sorter).setNoSortEnabled(
        noSortStateEnabled);
    }
  }

  /**
   * Obtm o ndice da linha no modelo a partir do ndice da linha da viso
   * (tabela).
   * 
   * @param viewRowIndex ndice de linha na viso (tabela).
   * 
   * @return ndice da linha no modelo. Retorna -1 se a linha especificada
   *         estiver fora dos limites do modelo, ou se o modelo atual for nulo.
   */
  @Override
  public int convertRowIndexToModel(int viewRowIndex) {
    if (viewRowIndex < 0) {
      return -1;
    }
    if (getModel() == null) {
      return -1;
    }
    if (viewRowIndex >= getModel().getRowCount()) {
      return -1;
    }
    return super.convertRowIndexToModel(viewRowIndex);
  }

  /**
   * Obtm o ndice da linha na viso a partir do ndice da linha do modelo
   * (tabela).
   * 
   * @param modelRowIndex ndice de linha no modelo (tabela).
   * 
   * @return ndice da linha na viso. Retorna -1 se a linha especificada
   *         estiver fora dos limites do modelo, ou se o modelo atual for nulo.
   */
  @Override
  public int convertRowIndexToView(int modelRowIndex) {
    if (modelRowIndex < 0) {
      return -1;
    }
    if (getModel() == null) {
      return -1;
    }
    if (modelRowIndex >= getModel().getRowCount()) {
      return -1;
    }
    return super.convertRowIndexToView(modelRowIndex);
  }

  /**
   * Define se determinada coluna do modelo  ou no ordenvel. Todas as colunas
   * do modelo so ordenveis por default, se a tabela for ordenvel. Este
   * mtodo permite desabilitar a ordenao em uma coluna especfica.
   * 
   * @param columnIndex ndice da coluna no modelo.
   * @param sortable indica se a coluna deve ser ordenvel (<code>true</code>)
   *        ou no (<code>false</code>).
   * 
   * @throws IndexOutOfBoundsException se a coluna estiver fora dos limites do
   *         modelo.
   */
  public void setSortable(int columnIndex, boolean sortable) {
    if (rowSorter != null) {
      rowSorter.setSortable(columnIndex, sortable);
    }
  }

  /**
   * Define como deve ser a aparncia da tabela desabilitada.
   * 
   * @param useDisabledLook - se igual a true, os componentes das clulas sero
   *        desabilitados quando a tabela estiver desabilitada, dando  mesma a
   *        real aparncia de desabilitada. Caso contrrio, a tabela ter o
   *        comportamento default para estes casos (as clulas tm aparncia
   *        normal mas no so interativas)
   */
  public void setUseDisabledLook(boolean useDisabledLook) {
    this.useDisabledLook = useDisabledLook;
  }

  /**
   * Adiciona um objeto  lista de listeners, sendo notificado sempre que um
   * evento para o qual ele est registrado ocorrer.
   * 
   * @param l o listener (ouvinte).
   */
  public void addSortableTableListener(SortableTableListener l) {
    listenerList.add(SortableTableListener.class, l);
  }

  /**
   * Remove um objeto da lista de listeners.
   * 
   * @param l o listener (ouvinte).
   */
  public void removeSortableTableListener(SortableTableListener l) {
    listenerList.remove(SortableTableListener.class, l);
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public void sorterChanged(RowSorterEvent e) {
    super.sorterChanged(e);
    if (e.getType().equals(RowSorterEvent.Type.SORTED)) {
      tableSorted(this);
    }
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public void tableSorted(Object source) {
    fireTableSorted();
  }

  /**
   * Dispara uma notificao de tabela (re)ordenada para todos os
   * <code>SortableTableListeners</code> que se registraram como ouvintes desta
   * tabela.
   * 
   * @see #addSortableTableListener(SortableTableListener)
   * @see EventListenerList
   */
  private void fireTableSorted() {
    Object[] listeners = listenerList.getListenerList(); // No-nulo garantido
    for (int i = listeners.length - 2; i >= 0; i -= 2) {
      if (listeners[i] == SortableTableListener.class) {
        ((SortableTableListener) listeners[i + 1]).tableSorted(this);
      }
    }
  }

  /**
   * Cria um painel de busca por palavra em uma coluna da tabela.
   * 
   * @param title titulo do painel.
   * @param ignoreCaseLabelText texto do checkbox usado para ignorar maisculas
   *        e minsculas.
   * @param previousToolTipText texto do tootip do buto de prvio.
   * @param nextToolTipText texto do tootip do buto de prximo.
   * @param paddingSize espaamento entre os componentes.
   * @param inLine true para a criao de um painel sem borda e com o ttulo
   *        alinhado com a caixa de texto e os botes.
   * 
   * @return um painel com uma caixa de texto onde ser digitada a palavra a ser
   *         buscada, dois butes para navegar entre os resultados e um checkbox
   *         que habilita ou desabilita se a busca ser sensvel a letras
   *         maisculas e minsculas.
   */
  public JPanel createSearchPanel(String title, String ignoreCaseLabelText,
    String previousToolTipText, String nextToolTipText, int paddingSize,
    boolean inLine) {
    return new SearchPanel(title, ignoreCaseLabelText, previousToolTipText,
      nextToolTipText, paddingSize, inLine);
  }

  /**
   * Cria um painel de busca por palavra em uma coluna da tabela.
   * 
   * @param title titulo do painel.
   * @param ignoreCaseLabelText texto do checkbox usado para ignorar maisculas
   *        e minsculas.
   * @param previousToolTipText texto do tootip do buto de prvio.
   * @param nextToolTipText texto do tootip do buto de prximo.
   * @param paddingSize espaamento entre os componentes.
   * 
   * @return um painel com uma caixa de texto onde ser digitada a palavra a ser
   *         buscada, dois butes para navegar entre os resultados e um checkbox
   *         que habilita ou desabilita se a busca ser sensvel a letras
   *         maisculas e minsculas.
   */
  public JPanel createSearchPanel(String title, String ignoreCaseLabelText,
    String previousToolTipText, String nextToolTipText, int paddingSize) {
    return this.createSearchPanel(title, ignoreCaseLabelText,
      previousToolTipText, nextToolTipText, paddingSize, false);
  }

  /**
   * Painel de busca por palavra em uma coluna da tabela.
   */
  private class SearchPanel extends JPanel {
    /** Campo de texto para insero do critrio de busca. */
    private JTextField searchField;
    /**
     * Checkbox indicando se devem ser diferenciadas maisculas de minsculas.
     * Pode ser null, indicando que ignoreCse estar sempre 'ligado'.
     */
    private JCheckBox ignoreCaseCheck;
    /**
     * Label para o Checkbox que indica se devem ser diferenciadas maisculas de
     * minsculas. Pode ser null, para indicar que queremos omitir o check.
     */
    private String ignoreCaseLabelText;
    /** Texto de tooltip para o boto "Anterior" */
    private String previousToolTipText;
    /** Texto de tooltip para o boto "Prximo" */
    private String nextToolTipText;
    /**
     * ndice de linha que corresponde ao critrio da ltima busca realizada.
     * Este ndice reverte ao valor inicial (-1) sempre que muda o critrio de
     * busca. Se for realizada nova busca com o mesmo critrio anterior,
     * procura-se algum ndice na seqncia de linhas da tabela que satisfaa
     * novamente ao critrio, at que no haja mais linhas satisfazendo ao
     * critrio.
     */
    private int matchingRowIndex = -1;
    /**
     * Coluna que serviu de base para a ltima busca realizada.  necessrio
     * guardar esse valor, pois, no caso de vrias buscas sucessivas (botes
     * "prximo" e "anterior", se o usurio selecionar outra coluna, a busca
     * deve recomear do incio.
     */
    private int lastSearchedColIndexView = -1;
    /**
     * Sentido de escolha da prxima linha que atende o critrio de busca.
     * Verifica de trs para frente, isto , da ltima para a primeira linha.
     */
    private static final int SEARCH_TYPE_BACK = -1;
    /**
     * Sentido de escolha da prxima linha que atende o critrio de busca.
     * Verifica de frente para trs, isto , da primeira para a ltima linha.
     */
    private static final int SEARCH_TYPE_FORWARD = 1;

    /**
     * Constri um painel, que permite a busca de uma palavra em uma coluna da
     * tabela. No exibe o check-box "Ignorar capitalizao".
     * 
     * @param title #TODO
     * @param previousToolTipText texto para o tooltip do boto "Anterior".
     * @param nextToolTipText texto para o tooltip do boto "Prximo".
     * @param paddingSize #TODO
     * @param inLine #TODO
     */
    private SearchPanel(String title, String previousToolTipText,
      String nextToolTipText, int paddingSize, boolean inLine) {
      this(title, null, previousToolTipText, nextToolTipText, paddingSize,
        inLine);
    }

    /**
     * Constri um painel, que permite a busca de uma palavra em uma coluna da
     * tabela.
     * 
     * @param title #TODO
     * @param ignoreCaseLabelText texto para o checkbox "Ignorar capitalizao".
     *        Pode ser null.
     * @param previousToolTipText texto para o tooltip do boto "Anterior".
     * @param nextToolTipText texto para o tooltip do boto "Prximo".
     * @param paddingSize #TODO
     * @param inLine #TODO
     */
    private SearchPanel(String title, String ignoreCaseLabelText,
      String previousToolTipText, String nextToolTipText, int paddingSize,
      boolean inLine) {
      this.ignoreCaseLabelText = ignoreCaseLabelText;
      this.previousToolTipText = previousToolTipText;
      this.nextToolTipText = nextToolTipText;
      this.searchField = createSearchField();
      this.ignoreCaseCheck = createIgnoreCaseCheck();
      //--
      this.setLayout(new GridBagLayout());
      GridBagConstraints c = new GridBagConstraints();
      c.insets = new Insets(0, 0, 0, 0);
      c.fill = GridBagConstraints.HORIZONTAL;
      c.anchor = GridBagConstraints.CENTER;
      c.insets.set(paddingSize, paddingSize, 0, 0);
      c.gridx = 0;
      c.gridy = 0;
      c.gridwidth = 1;
      c.gridheight = 1;
      if (inLine) {
        c.weightx = 0;
        c.weighty = 0;
        this.add(new JLabel(title), c);
        c.gridx = 1;
      }
      c.weightx = 3;
      c.weighty = 3;
      this.add(this.searchField, c);
      c.insets.set(paddingSize, paddingSize, 0, 0);
      c.fill = GridBagConstraints.NONE;
      c.gridx++;
      c.gridy = 0;
      c.gridwidth = 1;
      c.gridheight = 1;
      c.weightx = 0;
      c.weighty = 0;
      this.add(this.createStepBackButton(), c);
      c.insets.set(paddingSize, paddingSize, 0, paddingSize);
      c.fill = GridBagConstraints.NONE;
      c.gridx++;
      c.gridy = 0;
      c.gridwidth = 1;
      c.gridheight = 1;
      c.weightx = 0;
      c.weighty = 0;
      this.add(this.createStepForwardButton(), c);
      // se existe check "ignoreCase", vamos exibir
      if (ignoreCaseCheck != null) {
        c.insets.set(paddingSize, paddingSize, paddingSize, paddingSize);
        c.fill = GridBagConstraints.BOTH;
        c.gridx = 0;
        c.gridy = 1;
        c.gridwidth = 3;
        c.gridheight = 1;
        c.weightx = 1;
        c.weighty = 1;
        this.add(ignoreCaseCheck, c);
      }
      if (!inLine) {
        this.setBorder(BorderFactory.createTitledBorder(title));
      }
    }

    /**
     * Cria o campo de busca.
     * 
     * @return campo de busca.
     */
    private JTextField createSearchField() {
      JTextField textField = new JTextField(10);
      textField.addKeyListener(new KeyAdapter() {
        @Override
        public void keyReleased(KeyEvent e) {
          simpleSearch();
        }
      });
      return textField;
    }

    /**
     * Cria uma checkbox que indica ao campo de busca se este deve ou no
     * diferenciar letras maisculas de minsculas no texto a ser buscado.
     * 
     * @return checkbox que indica ao campo de busca se este deve ou no
     *         diferenciar letras maisculas de minsculas. Pode retornar null,
     *         se check no for exibido e ignoreCase deva estar sempre ligado.
     */
    private JCheckBox createIgnoreCaseCheck() {
      if (ignoreCaseLabelText == null) {
        return null;
      }
      JCheckBox check = new JCheckBox(ignoreCaseLabelText);
      check.addActionListener(new ActionListener() {
        @Override
        public void actionPerformed(ActionEvent e) {
          simpleSearch();
        }
      });
      return check;
    }

    /**
     * Cria o boto que volta na busca de uma palavra, na coluna selecionada.
     * 
     * @return boto com a ao de voltar definida.
     */
    private JButton createStepBackButton() {
      JButton stepBackButton = new JButton(GUIResources.BACK_ICON);
      stepBackButton.setPreferredSize(new Dimension(19, 19));
      stepBackButton.setToolTipText(previousToolTipText);
      stepBackButton.addActionListener(new ActionListener() {
        @Override
        public void actionPerformed(ActionEvent e) {
          searchsWithNavigation(SearchPanel.SEARCH_TYPE_BACK);
        }
      });
      return stepBackButton;
    }

    /**
     * Cria o boto que avana na busca de uma palavra, na coluna selecionada.
     * 
     * @return boto com a ao de avanar definida.
     */
    private JButton createStepForwardButton() {
      JButton stepForwardButton = new JButton(GUIResources.FORWARD_ICON);
      stepForwardButton.setPreferredSize(new Dimension(19, 19));
      stepForwardButton.setToolTipText(nextToolTipText);
      stepForwardButton.addActionListener(new ActionListener() {
        @Override
        public void actionPerformed(ActionEvent e) {
          searchsWithNavigation(SearchPanel.SEARCH_TYPE_FORWARD);
        }
      });
      return stepForwardButton;
    }

    /**
     * Busca e seleciona a primeira linha que contm o texto digitado no campo
     * de busca. A busca  sempre feita da primeira  ltima linha.
     */
    private void simpleSearch() {
      boolean hasText = false;
      int rowIndex = 0;
      SortableTable table = SortableTable.this;
      lastSearchedColIndexView = table.getSortedColIndexView();
      if (lastSearchedColIndexView == -1) {
        SortableTable.this.sort();
        lastSearchedColIndexView = table.getSortedColIndexView();
      }
      for (; rowIndex < table.getRowCount(); rowIndex++) {
        //#TODO Precisa do convertRowIndexToModel?
        if (hasSearchTextInCurrentCellValue(table.getValueAt(rowIndex,
          lastSearchedColIndexView), this.searchField.getText(),
          ignoreCaseIsSelected())) {
          hasText = true;
          break;
        }
      }
      if (hasText) {
        selectMatchingRow(rowIndex);
      }
      else {
        clearMatchingRow();
      }
    }

    /**
     * Remove a seleo de linhas atendendo ao critrio de busca. Chamado sempre
     * que o critrio de busca passado no for encontrado na tabela.
     */
    private void clearMatchingRow() {
      SortableTable.this.clearSelection();
      this.matchingRowIndex = -1;
    }

    /**
     * Busca nas prximas linhas da tabela o texto digitado no campo de busca,
     * partindo da linha que atendeu ao critrio buscado anteriormente.
     * 
     * @param searchType indica se a busca ocorre antes ou depois da linha
     *        corrente.
     */
    private void searchsWithNavigation(int searchType) {
      int baseColIndex = SortableTable.this.getSortedColIndexView();
      // Reinicia a busca, caso ocorra alguma modificao na tabela.
      if (this.lastSearchedColIndexView == -1
        || this.lastSearchedColIndexView != SortableTable.this
          .getSortedColIndexView()) {
        this.matchingRowIndex = -1;
        this.lastSearchedColIndexView = (baseColIndex < 0 ? 0 : baseColIndex);
      }
      int searchPointer = this.matchingRowIndex;
      searchPointer = nextRow(searchType, searchPointer);
      /*
       * O objetivo do loop abaixo  evitar que o mtodo fique verificando
       * indefinidamente as linhas da tabela. Se no for encontrado o critrio
       * de busca, o loop garante que cada linha s ser inspecionada uma vez.
       */
      for (int i = 0; i < SortableTable.this.getRowCount(); i++) {
        //#TODO Precisa do convertRowIndexToModel?
        if (hasSearchTextInCurrentCellValue(SortableTable.this.getValueAt(
          searchPointer, this.lastSearchedColIndexView), this.searchField
            .getText(), ignoreCaseIsSelected())) {
          selectMatchingRow(searchPointer);
          break;
        }
        searchPointer = nextRow(searchType, searchPointer);
      }
    }

    /**
     * @return TRUE se opo ignorar maisculas e minsculas estiver
     *         configurada/selecionada; FALSE se devemos fazer a distino de
     *         maisculas/minsculas
     */
    private boolean ignoreCaseIsSelected() {
      if (ignoreCaseCheck == null) {
        return true;
      }
      return ignoreCaseCheck.isSelected();
    }

    /**
     * Obtm a prxima linha a ser comparada com o critrio de busca.
     * 
     * @param searchType indica a direo da busca.
     * @param rowIndex ndice da linha verificada na busca anterior.
     * 
     * @return o ndice da prxima linha a ser verificada.
     */
    private int nextRow(int searchType, int rowIndex) {
      int nextRow = rowIndex + searchType;
      if (searchType == SearchPanel.SEARCH_TYPE_FORWARD
        && nextRow >= SortableTable.this.getRowCount()) {
        nextRow = 0;
      }
      else if (searchType == SearchPanel.SEARCH_TYPE_BACK && nextRow < 0) {
        nextRow = SortableTable.this.getRowCount() - 1;
      }
      return nextRow;
    }

    /**
     * Seleciona a linha da tabela cujo contedo corresponde ao critrio de
     * busca.
     * 
     * @param rowIndex ndice da linha a ser selecionada.
     */
    private void selectMatchingRow(int rowIndex) {
      // Seleciona a linha.  
      SortableTable.this.setRowSelectionInterval(rowIndex, rowIndex);
      // Rola o scroll da tabela.
      Rectangle rect = SortableTable.this.getCellRect(rowIndex, 0, true);
      SortableTable.this.scrollRectToVisible(rect);
      this.matchingRowIndex = rowIndex;
    }

    /**
     * Executa uma expresso regular que verifica se o valor de uma clula,
     * contm a palavra que est sendo procurada.
     * 
     * @param cellValue clula verificada.
     * @param searchText palavra que est sendo procurado.
     * @param ignoreCase indica se a busca ir ignorar letras maisculas e
     *        minsculas.
     * 
     * @return true caso a palavra procurada exista na clula verificada ou
     *         false caso contrrio.
     */
    private boolean hasSearchTextInCurrentCellValue(Object cellValue,
      String searchText, boolean ignoreCase) {
      Pattern pattern;
      if (cellValue != null && (searchText != null && searchText.trim()
        .length() > 0) && !hasSpecialChar(searchText)) {
        if (ignoreCase) {
          pattern = Pattern.compile(".*" + searchText + ".*",
            Pattern.CASE_INSENSITIVE);
        }
        else {
          pattern = Pattern.compile(".*" + searchText + ".*");
        }
        return (pattern.matcher(cellValue.toString())).matches();
      }
      return false;
    }

    /**
     * Verifica se uma string contm caracteres que no so letras do alfabeto.
     * 
     * @param str string que ser verificada.
     * 
     * @return true caso a string verifica tenha algum caractere que no seja
     *         uma letra do alfabeto e false caso contrrio.
     */
    private boolean hasSpecialChar(String str) {
      String[] specialChar = { "'", "!", "@", "#", "$", "%", "", "&", "*", "(",
          ")", "+", "=", "`", "'", "{", "[", "}", "]", "^", "~", "|", "\\", "<",
          ",", ">", ".", ":", ";", "?", "//", "*", "+" };
      if (str != null && str.trim().length() > 0) {
        for (int i = 0; i < specialChar.length; i++) {
          if (str.contains(specialChar[i])) {
            return true;
          }
        }
        return false;
      }
      return true;
    }
  }
}
