/*
 * $Id: ColumnGroup.java 140991 2013-05-15 15:36:56Z pasti $
 */
package tecgraf.javautils.gui.table;

import java.util.ArrayList;
import java.util.List;

import javax.swing.table.TableCellRenderer;
import javax.swing.table.TableColumn;

/**
 * Agrupamento de colunas de uma tabela. Utilizado para criar um header com um
 * ttulo para vrias colunas. Um agrupamento pode conter colunas ou
 * agrupamentos de colunas.
 */
public class ColumnGroup {
  /** Colunas ou grupamentos contidas por este agrupamento */
  private List<Object> columns;
  /** Ttulo deste agrupamento */
  private String text;
  /** Renderizador do cabealho deste grupamento. */
  private TableCellRenderer renderer;

  /**
   * Cria um agrupamento com um ttulo. O apresentador deste agrupamento utiliza
   * o mesmo padro do apresentador j existente na tabela.
   * 
   * @param text ttulo da coluna.
   */
  public ColumnGroup(String text) {
    this.text = text;
    columns = new ArrayList<Object>();
  }

  /**
   * Adiciona uma coluna ou grupamento ao agrupamento.
   * 
   * @param col coluna ou grupamento a ser adicionada.
   */
  public void add(Object col) {
    if (col == null) {
      throw new IllegalArgumentException(
        "O argumento col no pode ter valor null.");
    }
    columns.add(col);
  }

  /**
   * Informa o ttulo do agrupamento.
   * 
   * @return ttulo do agrupamento.
   */
  public String getHeaderValue() {
    return text;
  }

  /**
   * Altera o ttulo do agrupamento.
   * 
   * @param text ttulo do agrupamento.
   */
  public void setHeaderValue(String text) {
    this.text = text;
  }

  /**
   * Recupera as colunas ou grupamentos contidos neste agrupamento.
   * 
   * @return as colunas ou grupamentos contidos neste agrupamento.
   */
  public List<Object> getAllColumns() {
    return new ArrayList<Object>(columns);
  }
  
  /**
   * Obtm o renderizador do cabealho deste grupamento.
   * 
   * @return o renderizador do cabealho deste grupamento.
   */
  public TableCellRenderer getHeaderRenderer() {
    return renderer;
  }

  /**
   * Define o renderizador do cabealho deste grupamento.
   * 
   * @param renderer o renderizador do cabealho deste grupamento.
   */
  public void setHeaderRenderer(TableCellRenderer renderer) {
    this.renderer = renderer;
  }
  
  /**
   * Obtm a largura do agrupamento.
   * 
   * @return largura do agrupamento.
   */
  public int getWidth() {
    int width = 0;
    List<Object> columns = getAllColumns();
    for (Object obj : columns) {
      if (obj instanceof TableColumn) {
        TableColumn aColumn = (TableColumn) obj;
        width += aColumn.getWidth();
      }
      else {
        width += ColumnGroup.class.cast(obj).getWidth();
      }
    }
    return width;
  }
}
