package tecgraf.javautils.gui.table;

import java.awt.Component;
import java.awt.Dimension;
import java.text.DateFormat;
import java.text.NumberFormat;
import java.text.SimpleDateFormat;
import java.util.ArrayList;
import java.util.Date;
import java.util.List;
import java.util.Locale;

import javax.swing.JFrame;
import javax.swing.JScrollPane;
import javax.swing.JTable;
import javax.swing.table.DefaultTableCellRenderer;
import javax.swing.table.TableModel;

/**
 * Exemplo de uso da classe {@link ObjectTableModel}.
 *
 * @author Tecgraf/PUC-Rio
 */
public class ObjectTableModelSample {
  
  /**
   * Mtodo para teste da classe.
   * 
   * @param args (no usado)
   */
  public static void main(String[] args) {
    JFrame frame = new JFrame("teste");
    frame.setDefaultCloseOperation(JFrame.EXIT_ON_CLOSE);
    List<TestUser> rows = new ArrayList<TestUser>();
    rows.add(new TestUser("Joo", new Date(124123412L), new Double(1.80)));
    rows.add(new TestUser("Elisa", new Date(565124123412L), new Double(1.55)));
    rows
      .add(new TestUser("Flvia", new Date(775124123412L), new Double(1.675)));
    ObjectTableProvider provider = new DefaultObjectTableProvider() {
      public String[] getColumnNames() {
        String[] colNames = { "Nome", "Data Nasc.", "Altura" };
        return colNames;
      }

      public Class<?>[] getColumnClasses() {
        Class<?>[] colClasses = { String.class, Date.class, Double.class };
        return colClasses;
      }

      @Override
      public Object[] getCellValues(Object row) {
        if (row == null) {
          return new Object[] { "???" };
        }
        TestUser user = (TestUser) row;
        return new Object[] { user.getName(), user.getBirthdate(),
            user.getHeight() };
      }
    };
    TableModel m = new ObjectTableModel<TestUser>(rows, provider);
    SortableTable table = new SortableTable(m);

    // Exemplo de renderizador por classe
    table.setDefaultRenderer(Date.class, new DateRenderer());

    // Exemplo de renderizador por coluna
    table.getColumnModel().getColumn(2).setCellRenderer(new HeightRenderer());
    table.setPreferredScrollableViewportSize(new Dimension(200, 200));
    table.adjustColumnWidth();
    frame.getContentPane().add(new JScrollPane(table));
    frame.pack();
    frame.setVisible(true);
  }
}

/**
 * Classe auxiliar para teste da classe <code>ObjectTableModel</code>.
 */
class TestUser {
  /** Data de nascimento */
  private Date birthdate;

  /** Altura */
  private Double height;

  /** Nome */
  private String name;

  /**
   * Cria um usurio para teste.
   * 
   * @param name nome.
   * @param birthdate data de nascimento.
   * @param height altura.
   */
  public TestUser(String name, Date birthdate, Double height) {
    this.name = name;
    this.birthdate = birthdate;
    this.height = height;
  }

  /**
   * Obtm o nome.
   * 
   * @return nome.
   */
  public String getName() {
    return name;
  }

  /**
   * Obtm a altura.
   * 
   * @return altura.
   */
  public Double getHeight() {
    return height;
  }

  /**
   * Obtm a data de nascimento.
   * 
   * @return data de nascimento.
   */
  public Date getBirthdate() {
    return birthdate;
  }
}

/**
 * Classe auxiliar de teste para formatao da data. Possui um formatador padro
 * no formato "dd/MM/yyyy - HH:mm:ss".
 */
class DateRenderer extends DefaultTableCellRenderer {
  /** Formato para as datas */
  private static final String DEFAULT_DATE_PATTERN = "dd/MM/yyyy";

  /** Formatador encarregado de formatar a data */
  private DateFormat formatter;

  /**
   * Cria o renderizador.
   */
  public DateRenderer() {
    this.formatter = new SimpleDateFormat(DEFAULT_DATE_PATTERN);
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public Component getTableCellRendererComponent(JTable table, Object value,
    boolean isSelected, boolean hasFocus, int row, int column) {
    Component comp =
      super.getTableCellRendererComponent(table, value, isSelected, hasFocus,
        row, column);
    if (value instanceof Date) {
      Date date = (Date) value;
      String strDate = formatter.format(date);
      ((DefaultTableCellRenderer) comp).setText(strDate);
    }
    return comp;
  }
}

/**
 * Classe auxiliar de teste para formatao da altura. Formata no padro
 * brasileiro, com 2 casas decimais e "m" no final (exemplo: 1,55m).
 */
class HeightRenderer extends DefaultTableCellRenderer {
  /** Formatador encarregado de formatar a data */
  private NumberFormat formatter;

  /**
   * Cria o renderizador.
   */
  public HeightRenderer() {
    formatter = NumberFormat.getInstance(new Locale("pt", "BR"));
    formatter.setMaximumFractionDigits(2);
    formatter.setMinimumFractionDigits(2);
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public Component getTableCellRendererComponent(JTable table, Object value,
    boolean isSelected, boolean hasFocus, int row, int column) {
    Component comp =
      super.getTableCellRendererComponent(table, value, isSelected, hasFocus,
        row, column);
    if (value instanceof Double) {
      Double height = (Double) value;
      String strHeight = formatter.format(height) + "m";
      ((DefaultTableCellRenderer) comp).setText(strHeight);
    }
    return comp;
  }
}
