package tecgraf.javautils.gui.calendar;

import java.awt.BorderLayout;
import java.awt.Color;
import java.awt.Component;
import java.awt.Dimension;
import java.awt.Graphics;
import java.awt.GridBagConstraints;
import java.awt.GridBagLayout;
import java.awt.GridLayout;
import java.awt.Insets;
import java.awt.Polygon;
import java.awt.event.ActionEvent;
import java.awt.event.ActionListener;
import java.awt.event.MouseAdapter;
import java.awt.event.MouseEvent;
import java.awt.event.WindowAdapter;
import java.awt.event.WindowEvent;
import java.text.DateFormatSymbols;
import java.util.ArrayList;
import java.util.Calendar;
import java.util.GregorianCalendar;
import java.util.Locale;

import javax.swing.BorderFactory;
import javax.swing.JButton;
import javax.swing.JDialog;
import javax.swing.JFrame;
import javax.swing.JLabel;
import javax.swing.JPanel;
import javax.swing.SwingConstants;
import javax.swing.border.Border;

import tecgraf.javautils.core.lng.LNG;
import tecgraf.javautils.gui.GUIUtils;

/**
 * Componente visual para calendrio.
 * Pode ser usado como um painel ou como um {@link JDialog}
 *
 */
public class CalendarPanel extends JPanel {

  private ArrayList<CalendarListener> listeners;
  private JPanel northPanel;
  private JLabel monthYearLabel;
  private JPanel bodyPanel;
  private Button nextYear;
  private Button previewYear;
  private Button nextMonth;
  private Button previewMonth;
  private ArrayList<DayOfMonth> days;
  private ArrayList<DayOfWeek> daysOfWeeks;
  private long selectedDate;
  private boolean currentMonthOnly;
  private Color titleBackground = new Color(53, 74, 124);
  private Color titleForeground = Color.WHITE;
  private Color actualMonthForeground = Color.BLACK;
  private Color otherMonthForeground = Color.LIGHT_GRAY;
  private Color calendarBackground = Color.WHITE;
  private JDialog dialog;
  private Long dialogValue;

  /**
   * Construtor
   *
   */
  public CalendarPanel() {
    this(null);
  }

  /**
   * Construtor
   * @param locale Locale para o idioma
   */
  public CalendarPanel(Locale locale) {
    super(new BorderLayout());
    listeners = new ArrayList<CalendarListener>();
    if (locale == null) {
      locale = LNG.getLocale();
    }
    setLocale(locale);

    currentMonthOnly = false;
    days = new ArrayList<DayOfMonth>();
    daysOfWeeks = new ArrayList<DayOfWeek>();
    updateBorder(titleBackground);
    add(getNorthPanel(), BorderLayout.NORTH);
    add(getBodyPanel(), BorderLayout.CENTER);

    DateFormatSymbols dfs = new DateFormatSymbols(getLocale());
    String[] wds = dfs.getShortWeekdays();

    DayOfWeek dow = new DayOfWeek(wds[Calendar.SUNDAY]);
    daysOfWeeks.add(dow);
    dow = new DayOfWeek(wds[Calendar.MONDAY]);
    daysOfWeeks.add(dow);
    dow = new DayOfWeek(wds[Calendar.TUESDAY]);
    daysOfWeeks.add(dow);
    dow = new DayOfWeek(wds[Calendar.WEDNESDAY]);
    daysOfWeeks.add(dow);
    dow = new DayOfWeek(wds[Calendar.THURSDAY]);
    daysOfWeeks.add(dow);
    dow = new DayOfWeek(wds[Calendar.FRIDAY]);
    daysOfWeeks.add(dow);
    dow = new DayOfWeek(wds[Calendar.SATURDAY]);
    daysOfWeeks.add(dow);

    for (DayOfWeek d : daysOfWeeks) {
      getBodyPanel().add(d);
    }

    GregorianCalendar gc = new GregorianCalendar();
    gc.setTimeInMillis(System.currentTimeMillis());
    for (int i = 0; i < 6 * 7; i++) {
      DayOfMonth day = new DayOfMonth();
      days.add(day);
      getBodyPanel().add(day);
    }
    setSelectedDate(System.currentTimeMillis());
  }

  /**
   * Atualiza a cor da borda do calendrio
   * @param color {@link Color}
   */
  private void updateBorder(Color color) {
    setBorder(BorderFactory.createMatteBorder(1, 1, 1, 1, color));
  }

  /**
   * Obtm o painel com os botes, ms e ano
   * @return {@link JPanel}
   */
  private JPanel getNorthPanel() {
    if (northPanel == null) {
      northPanel = new JPanel();
      updateNorthPanelBorder(titleForeground);
      northPanel.setBackground(titleBackground);
      northPanel.setLayout(new GridBagLayout());
      GridBagConstraints gbc = new GridBagConstraints();
      gbc.gridy = 0;

      gbc.gridx = 0;
      gbc.weightx = 1;
      gbc.insets = new Insets(3, 3, 3, 3);
      gbc.anchor = GridBagConstraints.WEST;
      northPanel.add(getPreviewYear(), gbc);

      gbc.gridx = 1;
      gbc.weightx = 1;
      northPanel.add(getPreviewMonth(), gbc);

      gbc.weightx = 1000;
      gbc.fill = GridBagConstraints.HORIZONTAL;
      gbc.anchor = GridBagConstraints.CENTER;
      gbc.gridx = 2;
      northPanel.add(getMonthYearLabel(), gbc);

      gbc.fill = GridBagConstraints.NONE;
      gbc.weightx = 1;
      gbc.anchor = GridBagConstraints.EAST;
      gbc.gridx = 3;
      northPanel.add(getNextMonth(), gbc);

      gbc.gridx = 4;
      gbc.weightx = 1;
      northPanel.add(getNextYear(), gbc);
    }
    return northPanel;
  }

  /**
   * Atualiza a borda do painel de botes, ms e ano
   * @param color {@link Color}
   */
  private void updateNorthPanelBorder(Color color) {
    Border out = BorderFactory.createMatteBorder(0, 0, 1, 0, color);
    Border in = BorderFactory.createEmptyBorder(3, 3, 2, 3);
    northPanel.setBorder(BorderFactory.createCompoundBorder(out, in));
  }

  /**
   * Obtm o {@link JLabel} para ms e ano
   * @return {@link JLabel}
   */
  private JLabel getMonthYearLabel() {
    if (monthYearLabel == null) {
      monthYearLabel = new JLabel();
      monthYearLabel.setForeground(titleForeground);
      monthYearLabel.setHorizontalAlignment(SwingConstants.CENTER);
    }
    return monthYearLabel;
  }

  private JPanel getBodyPanel() {
    if (bodyPanel == null) {
      bodyPanel = new JPanel();
      bodyPanel.setLayout(new GridLayout(7, 7));
    }
    return bodyPanel;
  }

  private Button getNextMonth() {
    if (nextMonth == null) {
      nextMonth = new Button(ButtonsType.NEXTMONTH);
      nextMonth.addMouseListener(new MouseAdapter() {
        @Override
        public void mousePressed(MouseEvent arg0) {
          nextMonth();
        }
      });
    }
    return nextMonth;
  }

  private Button getNextYear() {
    if (nextYear == null) {
      nextYear = new Button(ButtonsType.NEXTYEAR);
      nextYear.addMouseListener(new MouseAdapter() {
        @Override
        public void mousePressed(MouseEvent arg0) {
          nextYear();
        }
      });
    }
    return nextYear;
  }

  private Button getPreviewMonth() {
    if (previewMonth == null) {

      previewMonth = new Button(ButtonsType.PREVIEWMONTH);
      previewMonth.addMouseListener(new MouseAdapter() {
        @Override
        public void mousePressed(MouseEvent arg0) {
          previewMonth();
        }
      });
    }
    return previewMonth;
  }

  private Button getPreviewYear() {
    if (previewYear == null) {
      previewYear = new Button(ButtonsType.PREVIEWYEAR);
      previewYear.addMouseListener(new MouseAdapter() {
        @Override
        public void mousePressed(MouseEvent arg0) {
          previewYear();
        }
      });
    }
    return previewYear;
  }

  /**
   * Ao para retroceder um ano.
   *
   */
  public void previewYear() {
    GregorianCalendar gc = new GregorianCalendar();
    gc.setTimeInMillis(selectedDate);
    gc.add(GregorianCalendar.YEAR, -1);
    setSelectedDate(gc.getTimeInMillis());
  }

  /**
   * Ao para avanar um ano.
   *
   */
  public void nextYear() {
    GregorianCalendar gc = new GregorianCalendar();
    gc.setTimeInMillis(selectedDate);
    gc.add(GregorianCalendar.YEAR, 1);
    setSelectedDate(gc.getTimeInMillis());
  }

  /**
   * Ao para avanar um ms.
   *
   */
  public void nextMonth() {
    GregorianCalendar gc = new GregorianCalendar();
    gc.setTimeInMillis(selectedDate);
    gc.add(GregorianCalendar.MONTH, 1);
    setSelectedDate(gc.getTimeInMillis());
  }

  /**
   * Ao para retroceder um ms.
   *
   */
  public void previewMonth() {
    GregorianCalendar gc = new GregorianCalendar();
    gc.setTimeInMillis(selectedDate);
    gc.add(GregorianCalendar.MONTH, -1);
    setSelectedDate(gc.getTimeInMillis());
  }

  /**
   * Seta a visibilidade dos botes.
   * @param visible boolean
   */
  public void setButtonsVisible(boolean visible) {
    getPreviewMonth().setVisible(visible);
    getPreviewYear().setVisible(visible);
    getNextMonth().setVisible(visible);
    getNextYear().setVisible(visible);
  }

  /**
   * Indica se os botes esto visveis.
   * @return boolean
   */
  public boolean isButtonsVisible() {
    return getPreviewMonth().isVisible();
  }

  /**
   * Verifica se apenas os dias do ms corrente esto sendo exibidos
   * @return boolean
   */
  public boolean isCurrentMonthOnly() {
    return currentMonthOnly;
  }

  /**
   * Obtm a data selecionada
   * @return Data em milesegundos
   */
  public long getSelectedDate() {
    return selectedDate;
  }

  /**
   * Atualiza as cores do ano, ms e botes de retroceder e avanar
   *
   */
  private void repaintTitle() {
    getNorthPanel().repaint();
    for (DayOfWeek dow : daysOfWeeks) {
      dow.update();
    }
  }

  /**
   * Seta se apenas os dias do ms corrente devem ser exibidos.
   * @param currentMonthOnly boolean
   */
  public void setCurrentMonthOnly(boolean currentMonthOnly) {
    this.currentMonthOnly = currentMonthOnly;
    updateCalendar();
  }

  /**
   * Obtm a cor de fundo do texto para ms, ano e botes de retroceder e avanar
   * @return {@link Color}
   */
  public Color getTitleBackground() {
    return titleBackground;
  }

  /**
   * Seta a cor de fundo do texto para ms, ano e botes de retroceder e avanar
   * @param titleBackground {@link Color}
   */
  public void setTitleBackground(Color titleBackground) {
    this.titleBackground = titleBackground;
    getNorthPanel().setBackground(titleBackground);
    updateBorder(titleBackground);
    repaintTitle();
    updateCalendar();
  }

  /**
   * Obtm a cor do texto para ms, ano e botes de retroceder e avanar
   * @return {@link Color}
   */
  public Color getTitleForeground() {
    return titleForeground;
  }

  /**
   * Seta a cor do texto para ms, ano e botes de retroceder e avanar
   * @param titleForeground {@link Color}
   */
  public void setTitleForeground(Color titleForeground) {
    this.titleForeground = titleForeground;
    updateNorthPanelBorder(titleForeground);
    getMonthYearLabel().setForeground(titleForeground);
    repaintTitle();
  }

  /**
   * Obtm a cor de texto dos dias do ms corrente
   * @return {@link Color}
   */
  public Color getActualMonthForeground() {
    return actualMonthForeground;
  }

  /**
   * Seta a cor de texto dos dias do ms corrente
   * @param actualMonthForeground {@link Color}
   */
  public void setActualMonthForeground(Color actualMonthForeground) {
    this.actualMonthForeground = actualMonthForeground;
    updateCalendar();
  }

  /**
   * Obtm a cor de texto dos dias do ms no corrente
   * @return {@link Color}
   */
  public Color getOtherMonthForeground() {
    return otherMonthForeground;
  }

  /**
   * Seta a cor de texto dos dias do ms no corrente
   * @param otherMonthForeground {@link Color}
   */
  public void setOtherMonthForeground(Color otherMonthForeground) {
    this.otherMonthForeground = otherMonthForeground;
    updateCalendar();
  }

  /**
   * Obtm a cor de fundo do calendrio
   * @return {@link Color}
   */
  public Color getCalendarBackground() {
    return calendarBackground;
  }

  /**
   * Seta a cor de fundo do calendrio
   * @param calendarBackground {@link Color}
   */
  public void setCalendarBackground(Color calendarBackground) {
    this.calendarBackground = calendarBackground;
    updateCalendar();
  }

  /**
   * Seta uma data como selecionada
   * @param date Data em milisegundos
   */
  public void setSelectedDate(long date) {
    int actualMonth;
    int actualYear;
    GregorianCalendar gc = new GregorianCalendar();
    gc.setTimeInMillis(selectedDate);
    actualMonth = gc.get(GregorianCalendar.MONTH);
    actualYear = gc.get(GregorianCalendar.YEAR);
    gc.setTimeInMillis(date);
    boolean dateInSameMonth = (gc.get(GregorianCalendar.MONTH) == actualMonth);
    boolean dateInSameYear = (gc.get(GregorianCalendar.YEAR) == actualYear);
    selectedDate = date;
    updateCalendar();
    CalendarEvent event = new CalendarEvent(date, dateInSameMonth
      && dateInSameYear);
    fireCalendarListeners(event);
  }

  /**
   * Atualiza o calendrio
   */
  private void updateCalendar() {
    GregorianCalendar gc = new GregorianCalendar();
    gc.setTimeInMillis(selectedDate);
    gc.set(GregorianCalendar.HOUR_OF_DAY, 0);
    gc.set(GregorianCalendar.MINUTE, 0);
    gc.set(GregorianCalendar.SECOND, 0);
    gc.set(GregorianCalendar.MILLISECOND, 0);
    long time = gc.getTimeInMillis();
    selectedDate = gc.getTimeInMillis();

    gc.set(GregorianCalendar.DAY_OF_MONTH, 1);
    int dayofweek = gc.get(GregorianCalendar.DAY_OF_WEEK);

    int less = 0;
    switch (dayofweek) {
    case GregorianCalendar.SUNDAY:
      less = 0;
      break;
    case GregorianCalendar.MONDAY:
      less = 1;
      break;
    case GregorianCalendar.TUESDAY:
      less = 2;
      break;
    case GregorianCalendar.WEDNESDAY:
      less = 3;
      break;
    case GregorianCalendar.THURSDAY:
      less = 4;
      break;
    case GregorianCalendar.FRIDAY:
      less = 5;
      break;
    case GregorianCalendar.SATURDAY:
      less = 6;
      break;
    }

    int currentMonth = gc.get(GregorianCalendar.MONTH);
    DateFormatSymbols dfs = new DateFormatSymbols(getLocale());
    String monthYear = dfs.getMonths()[currentMonth];
    monthYear += " - " + gc.get(GregorianCalendar.YEAR);
    getMonthYearLabel().setText(monthYear);

    gc.add(GregorianCalendar.DAY_OF_MONTH, (-1) * less);

    for (int i = 0; i < 6 * 7; i++) {

      DayVisible dayVisible = DayVisible.ACTUALMONTH;
      if (currentMonthOnly) {
        if (gc.get(GregorianCalendar.MONTH) != currentMonth) {
          dayVisible = DayVisible.OFF;
        }
      }
      else {
        if (gc.get(GregorianCalendar.MONTH) != currentMonth) {
          dayVisible = DayVisible.OTHERMONTH;
        }
      }

      DayOfMonth day = days.get(i);
      day.setDate(gc.getTimeInMillis(), dayVisible);
      if (compare(time, gc.getTimeInMillis())) {
        day.setSelected(true);
      }
      else {
        day.setSelected(false);
      }
      gc.add(GregorianCalendar.DAY_OF_MONTH, 1);
    }
  }

  /**
   * Compara se duas datas em milisegundos so iguais
   * @param time1 Primeira data
   * @param time2 Segunda data
   * @return boolean
   */
  private boolean compare(long time1, long time2) {
    GregorianCalendar gc1 = new GregorianCalendar();
    GregorianCalendar gc2 = new GregorianCalendar();
    gc1.setTimeInMillis(time1);
    gc2.setTimeInMillis(time2);
    boolean day = (gc1.get(GregorianCalendar.DAY_OF_MONTH) == gc2
      .get(GregorianCalendar.DAY_OF_MONTH));
    boolean month = (gc1.get(GregorianCalendar.MONTH) == gc2
      .get(GregorianCalendar.MONTH));
    boolean year = (gc1.get(GregorianCalendar.YEAR) == gc2
      .get(GregorianCalendar.YEAR));
    return (day && month && year);
  }

  /**
   * Enum para os tipos de botes de retroceder e avanar
   *
   */
  enum ButtonsType {
    PREVIEWMONTH, PREVIEWYEAR, NEXTYEAR, NEXTMONTH;
  }

  /**
   * Classe para representar os botes de retroceder e avanar
   *
   */
  class Button extends JPanel {

    private ButtonsType buttonsType;

    /**
     * Construtor
     * @param buttonsType {@link ButtonsType}
     */
    public Button(ButtonsType buttonsType) {
      this.buttonsType = buttonsType;
    }

    @Override
    public Dimension getPreferredSize() {
      return new Dimension(20, 20);
    }

    @Override
    public void paint(Graphics g) {
      g.setColor(titleBackground);
      Dimension dim = getSize();
      g.fillRect(0, 0, dim.width, dim.height);
      g.setColor(titleForeground);
      int minHeight = 2 * dim.height / 10;
      int maxHieght = 8 * dim.height / 10;

      int width1 = 2 * dim.width / 16;
      int width2 = 8 * dim.width / 16;
      int width3 = 8 * dim.width / 16;
      int width4 = 14 * dim.width / 16;

      if (buttonsType == ButtonsType.PREVIEWMONTH
        || buttonsType == ButtonsType.PREVIEWYEAR) {
        Polygon pol = new Polygon();
        pol.addPoint(width1, dim.height / 2);
        pol.addPoint(width3, minHeight);
        pol.addPoint(width3, maxHieght);
        g.fillPolygon(pol);

      }
      if (buttonsType == ButtonsType.PREVIEWYEAR) {
        Polygon pol = new Polygon();
        pol.addPoint(width2, dim.height / 2);
        pol.addPoint(width4, minHeight);
        pol.addPoint(width4, maxHieght);
        g.fillPolygon(pol);
      }

      if (buttonsType == ButtonsType.NEXTMONTH
        || buttonsType == ButtonsType.NEXTYEAR) {

        Polygon pol = new Polygon();
        pol.addPoint(width3, dim.height / 2);
        pol.addPoint(width1, minHeight);
        pol.addPoint(width1, maxHieght);
        g.fillPolygon(pol);
      }
      if (buttonsType == ButtonsType.NEXTYEAR) {
        Polygon pol = new Polygon();
        pol.addPoint(width4, dim.height / 2);
        pol.addPoint(width2, minHeight);
        pol.addPoint(width2, maxHieght);
        g.fillPolygon(pol);
      }
    }

  }

  /**
   * Enum para representar a visibilidade dos dias
   *
   */
  enum DayVisible {
    ACTUALMONTH, OTHERMONTH, OFF;
  }

  /**
   * Classe para representar o dia da semana
   *
   */
  class DayOfWeek extends JPanel {

    private JLabel label;

    /**
     * Construtor
     * @param name Nome do dia da semana
     */
    public DayOfWeek(String name) {
      super(new BorderLayout());
      setBorder(BorderFactory.createEmptyBorder(3, 3, 3, 3));
      label = new JLabel();
      label.setText(name);
      label.setHorizontalAlignment(SwingConstants.CENTER);
      add(label, BorderLayout.CENTER);
      update();
    }

    /**
     * Atualiza as cores
     */
    public void update() {
      setBackground(titleBackground);
      label.setForeground(titleForeground);
    }
  }

  /**
   * Classe para representar os dias
   *
   */
  class DayOfMonth extends JPanel {

    private JLabel label;
    private long time;
    private boolean selected;
    private DayVisible dayVisible;

    /**
     * Construtor
     *
     */
    public DayOfMonth() {
      super(new BorderLayout());
      label = new JLabel();
      add(label);

      MouseAdapter a = new MouseAdapter() {
        @Override
        public void mousePressed(MouseEvent arg0) {
          if (DayOfMonth.this.dayVisible != DayVisible.OFF) {
            setSelectedDate(DayOfMonth.this.getTime());
          }
        }
      };
      label.addMouseListener(a);
      this.addMouseListener(a);
    }

    /**
     * Seta a data
     * @param time Data em milisegundos
     * @param dayVisible {@link DayVisible}
     */
    public void setDate(long time, DayVisible dayVisible) {
      this.time = time;
      this.dayVisible = dayVisible;
      update();
    }

    private void update() {
      setBackground(calendarBackground);
      GregorianCalendar gc = new GregorianCalendar();
      gc.setTimeInMillis(time);
      int day = gc.get(GregorianCalendar.DAY_OF_MONTH);
      label.setText(Integer.toString(day));
      label.setHorizontalAlignment(SwingConstants.CENTER);
      if (dayVisible == DayVisible.ACTUALMONTH) {
        label.setForeground(actualMonthForeground);
      }
      else if (dayVisible == DayVisible.OTHERMONTH) {
        label.setForeground(otherMonthForeground);
      }
      else {
        label.setForeground(calendarBackground);
      }

      Border border = null;
      if (selected) {
        Border out = BorderFactory.createEmptyBorder(2, 2, 2, 2);
        Border in = BorderFactory
          .createMatteBorder(2, 2, 2, 2, titleBackground);
        border = BorderFactory.createCompoundBorder(out, in);
        border = BorderFactory.createCompoundBorder(out, border);
      }
      else {
        border = BorderFactory.createEmptyBorder(6, 6, 6, 6);
      }
      setBorder(border);
    }

    /**
     * Obtm a data em milisegundos
     * @return long
     */
    public long getTime() {
      return time;
    }

    /**
     * Verifica se est selecionado
     * @return boolean
     */
    public boolean isSelected() {
      return selected;
    }

    /**
     * Seta como selecionado
     * @param selected boolean
     */
    public void setSelected(boolean selected) {
      this.selected = selected;
      update();
    }
  }

  /**
   * Exibe um {@link JDialog} com o calendrio.
   * @param owner {@link JFrame}
   * @param title Ttulo para o dialog
   * @param confirmButtonTitle Ttulo para o boto de confirmar
   * @param cancelButtonTitle Ttulo para o boto de cancelar ou
   *  <CODE>NULL</CODE> para no ter boto de cancelar.
   * @param comp {@link Component} usado para localizar o {@link JDialog} na tela ou
   *  <CODE>NULL</CODE> para ser centralizado na tela.
   * @return {@link Long} com a data em milisegundos ou <CODE>NULL</CODE> caso
   * o {@link JDialog} seja fechado sem escolher nenhum valor.
   */
  public Long showAsDialog(JFrame owner, String title,
                           String confirmButtonTitle, String cancelButtonTitle,
                           Component comp) {
    dialogValue = null;
    JPanel panel = new JPanel(new BorderLayout());
    panel.setBorder(BorderFactory.createEmptyBorder(10, 10, 5, 10));
    JPanel buttonPanel = new JPanel();
    JButton confirmButton = new JButton();
    confirmButton.setText(confirmButtonTitle);
    confirmButton.addActionListener(new ActionListener() {
      public void actionPerformed(ActionEvent e) {
        dialogValue = selectedDate;
        dialog.dispose();
      }
    });
    buttonPanel.add(confirmButton);
    JButton cancelButton = null;
    if (cancelButtonTitle != null) {
      cancelButton = new JButton();
      cancelButton.setText(cancelButtonTitle);
      cancelButton.addActionListener(new ActionListener() {
        public void actionPerformed(ActionEvent e) {
          dialog.dispose();
        }
      });
      buttonPanel.add(cancelButton);
    }

    if (cancelButton != null) {
      JButton[] buttons = { confirmButton, cancelButton };
      GUIUtils.matchPreferredSizes(buttons);
    }

    dialog = new JDialog(owner, title, true);
    dialog.setResizable(false);
    panel.add(this, BorderLayout.CENTER);
    panel.add(buttonPanel, BorderLayout.SOUTH);
    dialog.add(panel);
    dialog.pack();
    dialog.setLocationRelativeTo(comp);
    dialog.setVisible(true);
    return dialogValue;
  }

  /**
   * Exibe um {@link JDialog} com o calendrio. O clique num dia fechar o {@link JDialog} 
   * retornando a data escolhida.
   * @param owner {@link JFrame}
   * @param title Ttulo para o dialog  
   * @param comp {@link Component} usado para localizar o {@link JDialog} na tela ou
   *  <CODE>NULL</CODE> para ser centralizado na tela.
   * @return {@link Long} com a data em milisegundos ou <CODE>NULL</CODE> caso
   * o {@link JDialog} seja fechado sem escolher nenhum valor.
   */
  public Long showAsDialog(JFrame owner, String title, Component comp) {
    dialogValue = null;
    JPanel panel = new JPanel(new BorderLayout());
    panel.setBorder(BorderFactory.createEmptyBorder(10, 10, 5, 10));
    dialog = new JDialog(owner, title, true);
    dialog.setResizable(false);
    panel.add(this, BorderLayout.CENTER);
    dialog.add(panel);
    dialog.pack();
    dialog.setLocationRelativeTo(comp);
    dialog.addWindowListener(new WindowAdapter() {
      @Override
      public void windowClosed(WindowEvent e) {
        dialogValue = null;
      }
    });

    CalendarListener listener = new CalendarListener() {
      public void dateSelected(CalendarEvent calendarEvent) {
        if (calendarEvent.isMonthChanged()) {
          dialogValue = calendarEvent.getDate();
          dialog.dispose();
        }
      }
    };
    addCalendarListener(listener);

    dialog.setVisible(true);

    removeCalendarListener(listener);
    return dialogValue;
  }

  /**
   * Notifica os listeners
   * @param event {@link CalendarEvent}
   */
  private void fireCalendarListeners(CalendarEvent event) {
    for (CalendarListener listener : listeners) {
      listener.dateSelected(event);
    }
  }

  /**
   * Adiciona um listener
   * @param listener {@link CalendarListener}
   */
  public void addCalendarListener(CalendarListener listener) {
    listeners.add(listener);
  }

  /**
   * Remove um listener
   * @param listener {@link CalendarListener}
   */
  public void removeCalendarListener(CalendarListener listener) {
    listeners.remove(listener);
  }

}
