package tecgraf.javautils.gui.crud.gui.table;

import java.awt.BorderLayout;
import java.awt.Dimension;
import java.awt.event.ActionEvent;
import java.awt.event.ActionListener;
import java.util.List;

import javax.swing.AbstractAction;
import javax.swing.JButton;
import javax.swing.JPanel;
import javax.swing.JPopupMenu;
import javax.swing.JScrollPane;
import javax.swing.JTable;
import javax.swing.JToolBar;
import javax.swing.ListSelectionModel;
import javax.swing.SortOrder;
import javax.swing.SwingConstants;
import javax.swing.event.ListSelectionEvent;
import javax.swing.event.ListSelectionListener;
import javax.swing.table.AbstractTableModel;
import javax.swing.table.TableColumn;
import javax.swing.table.TableColumnModel;

import tecgraf.javautils.gui.crud.IRegistrationDescriptor;
import tecgraf.javautils.gui.crud.RegistrationImages;
import tecgraf.javautils.gui.crud.gui.actions.RegistrationAction;
import tecgraf.javautils.gui.crud.gui.main.RegistrationMainPanel;
import tecgraf.javautils.gui.crud.gui.main.RegistrationMainPanel.Mode;
import tecgraf.javautils.gui.crud.gui.main.RegistrationModeListener;
import tecgraf.javautils.gui.table.SortableTable;

/**
 * Painel que representa os objetos que podem ser selecionados para
 * consulta/edio.
 *
 * @author Tecgraf
 * @param <M> tipo do objeto do modelo.
 * @param <I> tipo do objeto da interface.
 */
public class RegistrationTablePanel<M, I> extends JPanel {

  /**
   * A tabela
   */
  final private SortableTable table = new SortableTable();

  /**
   * Scrollpane da tabela principal.
   */
  final private JScrollPane scrollPane;

  /**
   * Descritor
   */
  final private IRegistrationDescriptor<M, I> descriptor;

  /**
   * Boto: others
   */
  final private JButton othersButton = new JButton();

  /**
   * Painel mestre.
   */
  final private RegistrationMainPanel<M, I> mainPanel;

  /**
   * Inicializao da tabela com sua montagem em grupos, ajuste de adapters e
   * tamanhos de colunas.
   */
  final private void updateTableColumns() {
    final TableColumnModel columnModel = table.getColumnModel();
    final int numColumns = columnModel.getColumnCount();
    for (int c = 0; c < numColumns; c++) {
      final TableColumn column = columnModel.getColumn(c);
      final int width = descriptor.getColumnWidth(c);
      column.setPreferredWidth(width);
      column.setWidth(width);
      column.setMinWidth(10);
    }
  }

  /**
   * Retorna o arquivo selecionado.
   *
   * @return o arquivo.
   */
  final public M getSelectedObject() {
    final int row = table.getSelectedRow();
    if (row < 0) {
      return null;
    }
    @SuppressWarnings("unchecked")
    M object = (M) table.getValueAt(row, 0);
    return object;
  }

  /**
   * Inicializao da tabela (relativo ao modelo).
   */
  private void initTableAttributes() {
    final RegistrationTableModel<M, I> model =
      new RegistrationTableModel<M, I>(descriptor);
    table.setModel(model);

    final RegistrationTableRenderer<M, I> renderer =
      new RegistrationTableRenderer<M, I>(descriptor);
    table.setDefaultRenderer(Object.class, renderer);
    table.sort(0, SortOrder.ASCENDING);
    table.setSelectionMode(ListSelectionModel.SINGLE_SELECTION);
    if (model.getRowCount() > 0) {
      table.setRowSelectionInterval(0, 0);
    }
    table.setAutoResizeMode(JTable.AUTO_RESIZE_ALL_COLUMNS);
  }

  /**
   * Inicializao da tabela (relativo ao modelo).
   */
  private void updateTableAttributes() {
    final RegistrationTableModel<M, I> model =
      new RegistrationTableModel<M, I>(descriptor);
    table.setModel(model);
    table.sort(0, SortOrder.ASCENDING);
    if (model.getRowCount() > 0) {
      table.setRowSelectionInterval(0, 0);
    }
  }

  /**
   * Montagem de painel de botes
   *
   * @param othersButton boto others
   *
   * @return painel.
   */
  private JToolBar buildButtonsPanel(JButton othersButton) {
    final JToolBar toolbar = new JToolBar(JToolBar.HORIZONTAL);
    toolbar.setFloatable(false);

    final List<RegistrationAction<M, I>> toolbarActions =
      descriptor.getToolbarActions(mainPanel);
    for (RegistrationAction<M, I> registrationAction : toolbarActions) {
      toolbar.add(registrationAction);
    }

    toolbar.add(new JPanel());
    toolbar.add(othersButton);

    return toolbar;
  }

  /**
   * Construtor
   *
   * @param mainPanel painel
   * @param descriptor descritor
   */
  public RegistrationTablePanel(final RegistrationMainPanel<M, I> mainPanel,
    final IRegistrationDescriptor<M, I> descriptor) {
    this.mainPanel = mainPanel;
    this.descriptor = descriptor;
    this.scrollPane = new JScrollPane(table);

    initOthersButton();

    setLayout(new BorderLayout());
    final JToolBar buttonsPanel = buildButtonsPanel(othersButton);
    add(BorderLayout.CENTER, scrollPane);
    add(BorderLayout.SOUTH, buttonsPanel);

    initTableAttributes();

    setAddModeListener(mainPanel);
    setEditModeListener(mainPanel);
    setViewModeListener(mainPanel);

    setSelectionListener(mainPanel);

    updateTableColumns();
    updateTableAttributes();
    final RegistrationTableMouseListener<M, I> lst =
      new RegistrationTableMouseListener<M, I>(this, descriptor);
    table.addMouseListener(lst);

    setMinimumSize(new Dimension(100, 100));
  }

  /**
   * Ajusta o listener de seleo.
   *
   * @param mainPanel painel
   */
  private void setSelectionListener(final RegistrationMainPanel<M, I> mainPanel) {
    table.getSelectionModel().addListSelectionListener(
      new ListSelectionListener() {
        @Override
        public void valueChanged(ListSelectionEvent e) {
          mainPanel.setMode(Mode.VIEW);
          final List<RegistrationAction<M, I>> actions =
            descriptor.getOthersMenuActions(mainPanel);
          othersButton.setEnabled(actions != null && !actions.isEmpty());
        }
      });
  }

  /**
   * Ajusta listener: {@link Mode#ADD}
   *
   * @param mainPanel painel
   */
  private void setAddModeListener(final RegistrationMainPanel<M, I> mainPanel) {
    final RegistrationTablePanel<M, I> self = RegistrationTablePanel.this;
    mainPanel.addModeListener(new RegistrationModeListener() {
      @Override
      public void modeChanged(Mode oldMode, Mode newMode) {
        if (newMode != Mode.ADD) {
          return;
        }
        self.setActive(false);
      }
    });
  }

  /**
   * Ajusta listener: {@link Mode#EDIT}
   *
   * @param mainPanel painel
   */
  private void setEditModeListener(final RegistrationMainPanel<M, I> mainPanel) {
    final RegistrationTablePanel<M, I> self = RegistrationTablePanel.this;
    mainPanel.addModeListener(new RegistrationModeListener() {
      @Override
      public void modeChanged(Mode oldMode, Mode newMode) {
        if (newMode != Mode.EDIT) {
          return;
        }
        self.setActive(false);
      }
    });
  }

  /**
   * Ajusta listener: {@link Mode#VIEW}
   *
   * @param mainPanel painel
   */
  private void setViewModeListener(final RegistrationMainPanel<M, I> mainPanel) {
    final RegistrationTablePanel<M, I> self = RegistrationTablePanel.this;
    mainPanel.addModeListener(new RegistrationModeListener() {
      @Override
      public void modeChanged(Mode oldMode, Mode newMode) {
        if (newMode != Mode.VIEW) {
          return;
        }
        self.setActive(true);
      }
    });
  }

  /**
   * Ajuste interno de inatividade.
   *
   * @param flag indicativo
   */
  private void setActive(final boolean flag) {
    table.setEnabled(flag);
  }

  /**
   * Inicializao do boto: others.
   */
  private void initOthersButton() {
    othersButton.setText(mainPanel.getString("button.others"));
    othersButton.setIcon(RegistrationImages.ICON_DOWN_4);
    othersButton.setHorizontalTextPosition(SwingConstants.RIGHT);
    othersButton.addActionListener(new ActionListener() {
      @Override
      public void actionPerformed(ActionEvent e) {
        final RegistrationMainPanel<M, I> mainPanel = getMainPanel();
        final List<RegistrationAction<M, I>> actions =
          descriptor.getOthersMenuActions(mainPanel);
        if (actions == null || actions.size() == 0) {
          return;
        }
        final JPopupMenu menu = new JPopupMenu();
        for (AbstractAction action : actions) {
          menu.add(action);
        }
        final int y = othersButton.getHeight();
        menu.show(othersButton, 0, y);
      }
    });
  }

  /**
   * Consulta modelo da tabela.
   *
   * @return modelo.
   */
  private AbstractTableModel getTableModel() {
    return (AbstractTableModel) table.getModel();
  }

  /**
   * Atualizao da tabela.
   */
  public void updateTable() {
    final AbstractTableModel model = getTableModel();
    final int selectedRow = table.getSelectedRow();
    model.fireTableDataChanged();
    if (selectedRow < 0) {
      return;
    }
    final int numRows = model.getRowCount();
    if (selectedRow < numRows) {
      table.setRowSelectionInterval(selectedRow, selectedRow);
    }
  }

  /**
   * Consulta painel principal.
   *
   * @return painel
   */
  RegistrationMainPanel<M, I> getMainPanel() {
    return mainPanel;
  }

  /**
   * Consulta o descritor.
   *
   * @return descritor
   */
  IRegistrationDescriptor<M, I> getDescriptor() {
    return descriptor;
  }

}
