/*
 * $Id$
 */
package tecgraf.javautils.gui.print;

import javax.swing.event.TableModelListener;
import javax.swing.table.TableModel;

/**
 * Implementao de PrintableTableModel, um modelo para a tabela que ser
 * impressa.
 * 
 * - Encapsula o modelo da tabela a ser impressa delegando chamadas ao mesmo. -
 * Implementa mtodos definidos em PrintableTableModel que acrescenta ao modelo
 * original o recurso de configurar e informar quando o modelo possui ou no
 * <b>nomes de linha</b> ou <b>corner</b>.
 * 
 */
public class DefaultPrintableTableModel implements PrintableTableModel {
  /** Modelo da tabela a ser impressa. */
  private TableModel model;

  /**
   * Construtor. Inicializa os campos necessrios. Caso <code>null</code> seja o
   * valor do modelo passado por parmetro ser lanada uma
   * IllegalArgumentException.
   * 
   * @param model modelo da tabela que ser impressa.
   * @throws IllegalArgumentException quando o modelo passado por parmetro 
   *         nulo.
   */
  public DefaultPrintableTableModel(TableModel model) {
    if (model == null) {
      throw new IllegalArgumentException("O modelo no pode ser null.");
    }
    this.model = model;
  }

  /**
   * {@inheritDoc} A implementao padro delega o mtodo para o modelo original
   * caso o mesmo seja uma implementao de PrintableTableModel; caso no seja,
   * retorna <code>false</code>, assumindo que no existem headers de linha.
   */
  @Override
  public boolean hasRowNames() {
    if (model instanceof PrintableTableModel) {
      return ((PrintableTableModel) model).hasRowNames();
    }
    return false;
  }

  /**
   * {@inheritDoc} A implementao padro delega o mtodo para o modelo original
   * caso o mesmo seja uma implementao de PrintableTableModel; caso no seja,
   * retorna <code>null</code>, assumindo que no h header para a linha passada
   * por parmetro.
   */
  @Override
  public String getRowName(int rowIndex) {
    if (model instanceof PrintableTableModel) {
      return ((PrintableTableModel) model).getRowName(rowIndex);
    }
    return null;
  }

  /**
   * {@inheritDoc} A implementao padro delega o mtodo para o modelo original
   * caso o mesmo seja uma implementao de PrintableTableModel; caso no seja,
   * retorna <code>null</code>, assumindo que no h corner para este modelo.
   */
  @Override
  public String getCornerName() {
    if (model instanceof PrintableTableModel) {
      return ((PrintableTableModel) model).getCornerName();
    }
    return null;
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public int getRowCount() {
    return this.model.getRowCount();
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public int getColumnCount() {
    return this.model.getColumnCount();
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public String getColumnName(int columnIndex) {
    return this.model.getColumnName(columnIndex);
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public Class<?> getColumnClass(int columnIndex) {
    return this.model.getColumnClass(columnIndex);
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public boolean isCellEditable(int rowIndex, int columnIndex) {
    return this.model.isCellEditable(rowIndex, columnIndex);
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public Object getValueAt(int rowIndex, int columnIndex) {
    return this.model.getValueAt(rowIndex, columnIndex);
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public void setValueAt(Object aValue, int rowIndex, int columnIndex) {
    this.model.setValueAt(aValue, rowIndex, columnIndex);
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public void addTableModelListener(TableModelListener l) {
    this.model.addTableModelListener(l);
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public void removeTableModelListener(TableModelListener l) {
    this.model.removeTableModelListener(l);
  }

  /**
   * Obtm o modelo da tabela que ser impressa.
   * 
   * @return o modelo da tabela que ser impressa.
   */
  public TableModel getModel() {
    return model;
  }
}
