/*
 * $Id: LargeColumnModel.java 138821 2013-03-11 17:32:49Z allan $
 */
package tecgraf.javautils.gui.print;

import java.util.BitSet;

import javax.swing.event.TableModelListener;
import javax.swing.table.TableModel;

import tecgraf.javautils.gui.table.FooterCell;

/**
 * Modelo da tabela a ser impressa com uma coluna muito grande. Esta coluna no
 * ser impressa como coluna, mas como uma nova linha abaixo da linha original.
 */
public class LargeColumnModel implements TableModel {
  /** Coluna a partir da qual ser iniciado o join */
  static final int START_JOIN_COLUMN = 0;

  /** Modelo original da tabela a ser impressa */
  private TableModel model;
  /** Coluna muito grande a ser impressa como uma linha */
  private int largeCol;
  /** Linhas inseridas no modelo */
  private BitSet insertedRows;

  /**
   * Construtor.
   * 
   * @param model modelo da tabela a ser impressa.
   * @param largeCol coluna muito grande a ser impressa como uma linha.
   */
  public LargeColumnModel(TableModel model, int largeCol) {
    this.model = model;
    this.largeCol = largeCol;
    insertedRows = new BitSet();
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public void addTableModelListener(TableModelListener l) {
    model.addTableModelListener(l);
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public Class<?> getColumnClass(int columnIndex) {
    return model.getColumnClass(getModelColumnIndex(columnIndex));
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public int getColumnCount() {
    return model.getColumnCount() - 1;
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public String getColumnName(int columnIndex) {
    return model.getColumnName(getModelColumnIndex(columnIndex));
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public int getRowCount() {
    int rowCount = model.getRowCount();
    insertedRows.clear();
    for (int row = 0; row < rowCount; row++) {
      Object value = model.getValueAt(row, largeCol);
      if (FooterCell.class.isInstance(value)) {
        value = FooterCell.class.cast(value).getValue();
      }
      if ((value != null) && !value.equals("")) {
        insertedRows.set(row + 1 + insertedRows.cardinality());
      }
    }
    return rowCount + insertedRows.cardinality();
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public Object getValueAt(int rowIndex, int columnIndex) {
    int col = getModelColumnIndex(rowIndex, columnIndex);
    if (col == -1) {
      return null;
    }
    int row = getModelRowIndex(rowIndex);
    Object value = model.getValueAt(row, col);
    if (FooterCell.class.isInstance(value)) {
      value = FooterCell.class.cast(value).getValue();
    }
    if (col == largeCol) {
      value = "   " + model.getColumnName(col) + ": " + value;
    }
    return value;
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public boolean isCellEditable(int rowIndex, int columnIndex) {
    int col = getModelColumnIndex(rowIndex, columnIndex);
    if (col == -1) {
      return false;
    }
    int row = getModelRowIndex(rowIndex);
    return model.isCellEditable(row, col);
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public void removeTableModelListener(TableModelListener l) {
    model.removeTableModelListener(l);
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public void setValueAt(Object aValue, int rowIndex, int columnIndex) {
    int col = getModelColumnIndex(rowIndex, columnIndex);
    if (col == -1) {
      return;
    }
    int row = getModelRowIndex(rowIndex);
    model.setValueAt(aValue, row, col);
  }

  /**
   * Obtm as linhas inseridas no modelo.
   * 
   * @return linhas inseridas no modelo.
   */
  public BitSet getInsertedRows() {
    return insertedRows;
  }

  /**
   * Obtm a coluna muito grande a ser impressa como uma linha.
   * 
   * @return coluna muito grande a ser impressa como uma linha.
   */
  public int getLargeCol() {
    return largeCol;
  }

  /**
   * Obtm a coluna do modelo original correspondente a <code>columnIndex</code>
   * .
   * 
   * @param columnIndex ndice da coluna a ter obtida a coluna do modelo.
   * 
   * @return coluna do modelo original.
   */
  private int getModelColumnIndex(int columnIndex) {
    return (columnIndex < largeCol) ? columnIndex : columnIndex + 1;
  }

  /**
   * Obtm a coluna da tabela correspondente a <code>columnIndex</code> do
   * modelo original. Retorna -1 se a coluna no existir na tabela.
   * 
   * @param columnIndex ndice da coluna a ter obtida a coluna da tabela.
   * 
   * @return coluna da tabela.
   */
  int getViewColumnIndex(int columnIndex) {
    if (columnIndex == largeCol) {
      return -1;
    }
    return (columnIndex < largeCol) ? columnIndex : columnIndex - 1;
  }

  /**
   * Obtm a coluna do modelo original correspondente a <code>columnIndex</code>
   * para a linha <code>rowIndex</code>. Retorna -1 se a coluna no for
   * apresentada na tabela.
   * 
   * @param columnIndex ndice da coluna a ter obtida a coluna do modelo.
   * @param rowIndex ndice da linha a ter obtida a coluna do modelo.
   * 
   * @return coluna do modelo original.
   */
  private int getModelColumnIndex(int rowIndex, int columnIndex) {
    int col = (columnIndex < largeCol) ? columnIndex : columnIndex + 1;
    /* Linha no inserida */
    if (!insertedRows.get(rowIndex)) {
      return col;
    }
    /* Linha inserida comea na coluna START_JOIN_COLUMN */
    if (columnIndex == START_JOIN_COLUMN) {
      return largeCol;
    }
    return -1;
  }

  /**
   * Obtm a linha do modelo original correspondente a <code>rowIndex</code>.
   * 
   * @param rowIndex ndice da linha a ter obtida a linha do modelo.
   * 
   * @return linha do modelo original.
   */
  private int getModelRowIndex(int rowIndex) {
    int addedRows = insertedRows.get(0, rowIndex + 1).cardinality();
    return rowIndex - addedRows;
  }

  /**
   * Obtm a linha da tabela correspondente a <code>rowIndex</code> do modelo.
   * 
   * @param rowIndex ndice da linha a ter obtida a linha da tabela.
   * 
   * @return linha da tabela.
   */
  int getViewRowIndex(int rowIndex) {
    int viewRowIndex = rowIndex;
    for (int row = insertedRows.nextSetBit(0); row >= 0; row =
      insertedRows.nextSetBit(row + 1)) {
      if ((row - 1) > viewRowIndex) {
        return viewRowIndex;
      }
      viewRowIndex++;
    }
    return viewRowIndex;
  }
}
