/*
 * $Id$
 */
package tecgraf.javautils.gui.print;

import java.awt.Color;
import java.awt.Font;
import java.text.DateFormat;
import java.text.NumberFormat;

/**
 * Tabela a ser impressa.
 * 
 * - title  o ttulo que aparece apenas uma vez, acima da tabela - header  um
 * comentrio que aparece colado na tabela, acima dela, e  repetido toda vez
 * que uma pgina  quebrada - posText  um comentrio que aparece colado na
 * tabela, ao seu final, i.e., no  repetido a cada vez que uma pgina 
 * quebrada
 * 
 * - o pageFormat pode mudar a cada chamada, a implementao deve guardar qual
 * foi a ltima 'coisa' (title, header, linha, etc.) desenhada
 * 
 * - desenha linhas horizontais e verticais - ocupa apenas o espao vertical
 * necessrio - se uma linha no couber na vertical, no a desenha, para que
 * seja desenhada na prxima pgina - nmeros alinhados  direita e todo o resto
 *  esquerda
 */
public interface PrintableTable extends PrintableReportItem {
  /** Alinhamento  esquerda (default) */
  public static final int LEFT = 0;
  /** Alinhamento  direita */
  public static final int RIGHT = 1;
  /** Alinhamento centralizado */
  public static final int CENTER = 2;

  /**
   * Configura a fonte padro. Todas os textos que no tiverem uma fonte
   * explicitamente definida sero impressos com esta fonte padro.
   * 
   * @param font fonte padro para impresso.
   */
  public void setDefaultFont(Font font);

  /**
   * Configura a fonte a ser usada em um conjunto de clulas. Todas as clulas
   * compreendidas entre as colunas e as linhas fornecidas sero impressas com a
   * fonte especificada. As coordenadas utilizadas so referentes ao modelo. A
   * configurao da fonte dos nomes de linha, coluna e corner so realizadas
   * por mtodos prprios.
   * 
   * @param fromRow primeira coluna a usar a fonte.
   * @param toRow ltima coluna a usar a fonte.
   * @param fromColumn primeira linha a usar a fonte.
   * @param toColumn ltima linha a usar a fonte.
   * @param font fonte da impresso.
   * @see #setRowNameFont(int, int, Font) configura fonte dos nomes de linha
   * @see #setColumnNameFont(int, int, Font) configura fonte dos nomes de coluna
   * @see #setCornerNameFont(Font) configura fonte do corner da tabela
   */
  public void setCellFont(int fromRow, int toRow, int fromColumn, int toColumn,
    Font font);

  /**
   * Configura a fonte a ser usada em um conjunto de clulas do nome de coluna.
   * Todas as clulas compreendidas entre as colunas fornecidas sero impressas
   * com a fonte especificada.
   * 
   * @param fromColumn primeira coluna a usar a fonte.
   * @param toColumn ltima coluna a usar a fonte.
   * @param font fonte da impresso.
   */
  public void setColumnNameFont(int fromColumn, int toColumn, Font font);

  /**
   * Configura a fonte a ser usada nos nomes de linha ou coluna caso o modelo
   * possua nomes de linha ou coluna.
   * 
   * @param font fonte utilizada na impresso.
   */
  public void setNamesDefaultFont(Font font);

  /**
   * Configura a fonte a ser usada em um conjunto de clulas do nome de linha.
   * Todas as clulas compreendidas entre as linhas fornecidas sero impressas
   * com a fonte especificada.
   * 
   * @param fromRow primeira clula a usar a fonte.
   * @param toRow ltima clula a usar a fonte.
   * @param font fonte da impresso.
   */
  public void setRowNameFont(int fromRow, int toRow, Font font);

  /**
   * Configura fonte a ser usada no corner do modelo. Caso no exista corner o
   * mtodo no surtir efeito algum.
   * 
   * @param font fonte da impresso do corner.
   */
  public void setCornerNameFont(Font font);

  /**
   * Configura a espessura das linhas que separam as clulas da tabela. Uma
   * espessura negativa faz com que as linhas no sejam desenhadas. Espessura
   * zero  a linha mais fina possvel.
   * 
   * @param lineWidth espessura das linhas da tabela.
   */
  public void setLineWidth(float lineWidth);

  /**
   * Configura o espao entre os textos e as linhas que separam as clulas da
   * tabela. O valor padro  zero tanto em X como em Y.
   * 
   * @param dx espaamento horizontal entre os textos e as linhas
   * @param dy espaamento vertical entre os textos e as linhas
   */
  public void setInsets(float dx, float dy);

  /**
   * Indica o comentrio que aparece colado na tabela, ao seu final, e no 
   * repetido a cada vez que uma pgina  quebrada.
   * 
   * @param text comentrio que aparece colado na tabela ao seu final.
   */
  public void setPosText(String text);

  /**
   * Indica o comentrio que aparece colado na tabela, ao seu final, e no 
   * repetido a cada vez que uma pgina  quebrada e a fonte da impresso do
   * comentrio.
   * 
   * @param text comentrio que aparece colado na tabela ao seu final.
   * @param font fonte da impresso do comentrio.
   */
  public void setPosText(String text, Font font);

  /**
   * Indica o comentrio que aparece colado na tabela, acima dela, e  repetido
   * toda vez que uma pgina  quebrada.
   * 
   * @param header comentrio que aparece colado na tabela acima dela.
   */
  public void setTableHeader(String header);

  /**
   * Indica o comentrio que aparece colado na tabela, acima dela, e  repetido
   * toda vez que uma pgina  quebrada e a fonte da impresso do comentrio.
   * 
   * @param header comentrio que aparece colado na tabela acima dela.
   * @param font fonte da impresso do comentrio.
   */
  public void setTableHeader(String header, Font font);

  /**
   * Indica o ttulo que aparece apenas uma vez, acima da tabela.
   * 
   * @param title ttulo que aparece apenas uma vez acima da tabela.
   */
  public void setTitle(String title);

  /**
   * Indica o ttulo que aparece apenas uma vez, acima da tabela e a fonte da
   * impresso do ttulo.
   * 
   * @param title ttulo que aparece apenas uma vez acima da tabela.
   * @param font fonte da impresso do ttulo.
   */
  public void setTitle(String title, Font font);

  /**
   * Indica o formato em que as datas devem ser impressas.
   * 
   * @param format formato em que as datas devem ser impressas.
   */
  public void setDateFormat(DateFormat format);

  /**
   * Indica o formato em que os nmeros decimais devem ser impressos.
   * 
   * @param format formato em que os nmeros decimais devem ser impressos.
   */
  public void setDecimalFormat(NumberFormat format);

  /**
   * Indica a largura proporcional de cada coluna da tabela. Uma vez indicados
   * estes valores, a tabela no mais se auto ajustar, mas seguir estas
   * indicaes. A tabela ocupar toda a largura da pgina.
   * 
   * @param columnPart largura proporcional de cada coluna.
   */
  public void setColumnPart(float[] columnPart);

  /**
   * Indica se deve ou no ajustar a tabela  largura da pgina. Se a largura
   * das colunas for definida proporcionalmente, a tabela se ajustar  pgina
   * independente do valor <code>adjustWidth</code>. O valor default 
   * verdadeiro.
   * 
   * @param adjustWidth verdadeiro se quiser ajustar a tabela.
   */
  public void setAdjustWidth(boolean adjustWidth);

  /**
   * Indica o alinhamento de uma ou mais colunas da tabela (<code>LEFT</code>,
   * <code>RIGHT</code> ou <code>CENTER</code>.
   * 
   * @param columnIndexes array de ndices das colunas cujos alinhamentos sero
   *        indicados (0-based).
   * @param alignment alinhamento para a coluna.
   */
  public void setColumnAlignment(int[] columnIndexes, int alignment);

  /**
   * Indica o alinhamento da tabela ( esquerda, direita ou centralizado).
   * 
   * @param alignment alinhamento da tabela.
   */
  public void setAlignment(int alignment);

  /**
   * Indica quando as bordas da linha devem se tornar visveis ou invisveis. A
   * borda acima da <code>startRow</code> e a borda abaixo da
   * <code>endRow</code> no so alteradas.
   * 
   * @param startRow linha a partir da qual a borda ser visvel ou invisvel.
   * @param endRow linha at a qual a borda ser visvel ou invisvel.
   * @param visible falso para tornar invisvel a borda.
   */
  public void setLineBorderVisible(int startRow, int endRow, boolean visible);

  /**
   * Indica quando as bordas da coluna devem se tornar visveis ou invisveis. A
   * borda  esquerda da <code>startCol</code> e a borda  direita da
   * <code>endCol</code> no so alteradas.
   * 
   * @param startCol coluna a partir da qual a borda ser visvel ou invisvel.
   * @param endCol coluna at a qual a borda ser visvel ou invisvel.
   * @param visible falso para tornar invisvel a borda.
   */
  public void setColumnBorderVisible(int startCol, int endCol, boolean visible);

  /**
   * Indica que determinadas colunas de uma linha da tabela devem ser unidas. O
   * contedo lido do modelo na coluna <code>startCol</code> ocupar as colunas
   * que foram unidas.
   * 
   * @param row linha em que as colunas sero unidas.
   * @param startCol coluna a partir da qual a unio acontecer.
   * @param endCol coluna at a qual a unio acontecer.
   */
  public void joinColumns(int row, int startCol, int endCol);

  /**
   * Atribui determinada cor de fundo para uma linha da tabela, com a
   * possibilidade de preencher toda ou somente parte das clulas da linha.
   * 
   * @param row ndice da linha.
   * @param startCol ndice da coluna inicial.
   * @param endCol ndice da coluna final.
   * @param color cor escolhida para o fundo da linha.
   */
  public void setRowColor(int row, int startCol, int endCol, Color color);

  /**
   * Atribui determinada cor de fundo para uma coluna da tabela, com a
   * possibilidade de preencher toda ou somente parte das clulas da coluna.
   * 
   * @param column ndice da coluna.
   * @param startRow ndice da linha inicial.
   * @param endRow ndice da linha final.
   * @param color cor escolhida para o fundo da coluna.
   */
  public void setColumnColor(int column, int startRow, int endRow, Color color);
}
