/*
 * Created on Apr 14, 2005
 */
package tecgraf.javautils.gui.table;

import java.util.ArrayList;
import java.util.List;

import javax.swing.event.EventListenerList;
import javax.swing.event.TableModelEvent;
import javax.swing.event.TableModelListener;
import javax.swing.table.TableModel;

/**
 * Modelo que filtra linhas de outro modelo de tabela.
 */
public class FilteredTableModel implements TableModel, TableModelListener {
  /** Lista de listeners deste modelo */
  private EventListenerList listenerList;
  /** Modelo a ser filtrado */
  private TableModel model;
  /** Lista dos ndices das linhas includas */
  private List<Integer> filteredRows;
  /** Filtro a ser usado */
  private Filter filter;

  /**
   * Construtor.
   * 
   * @param model modelo a ser filtrado.
   * @param filter filtro a ser usado.
   */
  public FilteredTableModel(TableModel model, Filter filter) {
    if (model == null) {
      throw new IllegalArgumentException("model == null");
    }
    this.model = model;
    this.filter = filter;
    this.filteredRows = new ArrayList<Integer>();
    this.listenerList = new EventListenerList();
    setFilteredRows();
    model.addTableModelListener(this);
  }

  /**
   * Indica as linhas que sero apresentadas.
   */
  private void setFilteredRows() {
    filteredRows.clear();
    for (int rowIndex = 0; rowIndex < model.getRowCount(); rowIndex++) {
      if ((filter == null) || (filter.includes(model, rowIndex))) {
        filteredRows.add(new Integer(rowIndex));
      }
    }
  }

  /**
   * Filtra a tabela de acordo com o filtro especificado.
   */
  public void filter() {
    setFilteredRows();
    fireTableFiltered();
    fireTableChanged(new TableModelEvent(this));
  }

  /**
   * Notifica os listeners de que este modelo foi filtrado.
   */
  private void fireTableFiltered() {
    Object[] listeners = listenerList.getListenerList();
    for (int i = listeners.length - 2; i >= 0; i -= 2) {
      if (listeners[i] == FilteredTableListener.class) {
        ((FilteredTableListener) listeners[i + 1]).tableFiltered(this);
      }
    }
  }

  /**
   * Notifica os listeners de que este modelo sofreu uma mudana em todas as
   * suas linhas.
   * 
   * @param e evento.
   */
  private void fireTableChanged(TableModelEvent e) {
    Object[] listeners = listenerList.getListenerList();
    for (int i = listeners.length - 2; i >= 0; i -= 2) {
      if (listeners[i] == TableModelListener.class) {
        ((TableModelListener) listeners[i + 1]).tableChanged(e);
      }
    }
  }

  /**
   * Obtm o modelo (no filtrado).
   * 
   * @return modelo (no filtrado).
   */
  public TableModel getModel() {
    return model;
  }

  /**
   * {@inheritDoc}
   */
  public int getRowCount() {
    return filteredRows.size();
  }

  /**
   * {@inheritDoc}
   */
  public int getColumnCount() {
    return model.getColumnCount();
  }

  /**
   * {@inheritDoc}
   */
  public Object getValueAt(int viewRowIndex, int columnIndex) {
    int modelIndex = (filteredRows.get(viewRowIndex)).intValue();
    return model.getValueAt(modelIndex, columnIndex);
  }

  /**
   * {@inheritDoc}
   */
  public void setValueAt(Object value, int viewRowIndex, int columnIndex) {
    int modelRowIndex = (filteredRows.get(viewRowIndex)).intValue();
    model.setValueAt(value, modelRowIndex, columnIndex);
  }

  /**
   * {@inheritDoc}
   */
  public String getColumnName(int columnIndex) {
    return model.getColumnName(columnIndex);
  }

  /**
   * {@inheritDoc}
   */
  public Class<?> getColumnClass(int columnIndex) {
    return model.getColumnClass(columnIndex);
  }

  /**
   * {@inheritDoc}
   */
  public boolean isCellEditable(int rowIndex, int columnIndex) {
    return model.isCellEditable(rowIndex, columnIndex);
  }

  /**
   * {@inheritDoc}
   */
  public void addTableModelListener(TableModelListener l) {
    listenerList.add(TableModelListener.class, l);
  }

  /**
   * {@inheritDoc}
   */
  public void removeTableModelListener(TableModelListener l) {
    listenerList.remove(TableModelListener.class, l);
  }

  /**
   * {@inheritDoc}<br>
   * Mtodo chamado sempre que ocorre uma alterao no modelo.
   */
  public void tableChanged(TableModelEvent e) {
    filter();
    fireTableChanged(new TableModelEvent(this));
  }

  /**
   * Adiciona o listener de filtro especificado  lista de observadores deste
   * modelo.
   * 
   * @param l listener de filtro.
   */
  public void addFilteredTableListener(FilteredTableListener l) {
    listenerList.add(FilteredTableListener.class, l);
  }

  /**
   * Remove o listener de filtro especificado da lista de observadores deste
   * modelo.
   * 
   * @param l listener de filtro.
   */
  public void removeFilteredTableListener(FilteredTableListener l) {
    listenerList.remove(FilteredTableListener.class, l);
  }

  /**
   * Obtm o ndice da linha na viso filtrada a partir do ndice da linha do
   * modelo original.
   * 
   * @param modelRowIndex ndice de linha no modelo original.
   * 
   * @return ndice da linha na viso filtrada. Retorna -1 se a linha
   *         especificada no for encontrada no modelo.
   */
  public int getFilteredRowIndex(int modelRowIndex) {
    if (modelRowIndex < 0 || modelRowIndex > model.getRowCount() - 1) {
      return -1;
    }
    int size = filteredRows.size();
    for (int i = 0; i < size; i++) {
      Integer modelIndex = filteredRows.get(i);
      if (modelIndex.equals(Integer.valueOf(modelRowIndex))) {
        return i;
      }
    }
    return -1;
  }

  /**
   * Obtm o ndice da linha no modelo a partir do ndice da linha da viso
   * filtrada.
   * 
   * @param filteredRowIndex ndice de linha na viso filtrada.
   * 
   * @return ndice da linha no modelo original. Retorna -1 se a linha
   *         especificada no for encontrada na viso filtrada.
   */
  public int getModelRowIndex(int filteredRowIndex) {
    if (filteredRowIndex < 0 || filteredRowIndex > getRowCount() - 1) {
      return -1;
    }
    return (filteredRows.get(filteredRowIndex)).intValue();
  }
}
