/*
 * $Id: GroupableTableHeader.java 151642 2014-04-16 19:36:48Z cviana $
 */
package tecgraf.javautils.gui.table;

import java.awt.Component;
import java.awt.Dimension;
import java.awt.Graphics;
import java.awt.Point;
import java.awt.Rectangle;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Collections;
import java.util.Enumeration;
import java.util.HashMap;
import java.util.HashSet;
import java.util.LinkedHashMap;
import java.util.List;
import java.util.Map;
import java.util.Map.Entry;

import javax.swing.JComponent;
import javax.swing.JLabel;
import javax.swing.JTable;
import javax.swing.SwingUtilities;
import javax.swing.UIManager;
import javax.swing.event.MouseInputListener;
import javax.swing.plaf.basic.BasicTableHeaderUI;
import javax.swing.table.DefaultTableCellRenderer;
import javax.swing.table.JTableHeader;
import javax.swing.table.TableCellRenderer;
import javax.swing.table.TableColumn;
import javax.swing.table.TableColumnModel;

/**
 * Header que permite agrupar colunas e dar um ttulo a este grupo.
 */
public class GroupableTableHeader extends JTableHeader {
  /** Os agrupamentos de coluna */
  private List<ColumnGroup> columnGroups;
  /** Indica as posies Y dos headers em determinada coluna */
  private Map<TableColumn, Map<Rectangle, Object>> columnGroupY;
  /**
   * Mapeia as colunas da tabela com os agrupamentos de colunas que as contm.
   * Este mapa foi criado com o intuito de otimizar a busca dos grupos de
   * colunas que contm uma coluna determinada.
   */
  private Map<TableColumn, List<ColumnGroup>> groupsByColumn;

  /**
   * Cria um header que permite agrupamento de colunas.
   * 
   * @param model modelo das colunas da tabela.
   */
  public GroupableTableHeader(TableColumnModel model) {
    super(model);
    this.columnGroupY = new HashMap<TableColumn, Map<Rectangle, Object>>();
    // Cria o renderizador padro para o agrupamento das colunas.
    setDefaultRenderer(createGroupRenderer());
    setUI(new GroupableTableHeaderUI());
    setReorderingAllowed(false);
  }

  /**
   * {@inheritDoc}
   * 
   * O mtodo <code>updateUI</code> est sendo sobrescrito para garantir que o
   * <code>GroupableTableHeaderUI</code> no seja substitudo pelo UI padro do
   * <code>JTableHeader</code>.
   */
  @Override
  public void updateUI() {
    setUI(ui);
    TableCellRenderer renderer = getDefaultRenderer();
    if (renderer instanceof Component) {
      SwingUtilities.updateComponentTreeUI((Component) renderer);
    }
  }

  /**
   * Define se as colunas podem ser arrastadas. Neste header no permitiremos
   * que sejam arrastadas.
   * 
   * @param b <code>true</code> ou <code>false</code>  indiferente.
   */
  @Override
  public void setReorderingAllowed(boolean b) {
    reorderingAllowed = false;
  }

  /**
   * Adiciona um agrupamento de colunas ao header.
   * 
   * @param group agrupamento de colunas.
   */
  public void addColumnGroup(ColumnGroup group) {
    if (columnGroups == null) {
      columnGroups = new ArrayList<ColumnGroup>();
    }
    columnGroups.add(group);
    groupsByColumn = null;
  }

  /**
   * Apaga os agrupamentos de colunas do header.
   */
  public void clearColumnGroups() {
    columnGroups = null;
    groupsByColumn = null;
  }

  /**
   * Informa os agrupamentos de colunas que contm a coluna determinada.
   * 
   * @param col coluna a partir da qual sero definidos os agrupamentos.
   * 
   * @return enumerao com os agrupamentos que contm a coluna col.
   */
  public List<ColumnGroup> getColumnGroups(TableColumn col) {
    if (groupsByColumn == null) {
      fillGroupsByColumnMap();
    }
    if (groupsByColumn == null) {
      return null;
    }
    return groupsByColumn.get(col);
  }

  /**
   * Informa o nome apresentado em determinado ponto.
   * 
   * @param point ponto a ser verificado.
   * 
   * @return nome apresentado em determinado ponto do header, ou
   *         <code>null</code> caso no haja.
   */
  public String getColumnNameAt(Point point) {
    Object value = getColumnValueAt(point);
    if (value == null) {
      return null;
    }
    return value.toString();
  }

  /**
   * Recupera o valor que representa a coluna ou grupamento de colunas num
   * ponto.
   * 
   * @param point o ponto.
   * @return o valor que representa a coluna ou grupamento de colunas existente
   *         no ponto, ou <code>null</code> caso no haja.
   */
  public Object getColumnValueAt(Point point) {
    int col = getColumnModel().getColumnIndexAtX(point.x);
    if (col < 0) {
      return null;
    }
    TableColumn aColumn = columnModel.getColumn(col);
    Map<Rectangle, Object> yNames = columnGroupY.get(aColumn);
    for (Entry<Rectangle, Object> entry : yNames.entrySet()) {
      Rectangle rect = entry.getKey();
      if (rect.y <= point.y && (rect.y + rect.height) >= point.y) {
        return entry.getValue();
      }
    }
    return null;
  }

  /**
   * Recupera o objeto que representa o valor de um agrupamento de colunas.
   * 
   * @param cGroup o agrupamento de colunas.
   * @return o objeto que representa o valor de <code>cGroup</code>.
   */
  protected Object getGroupValue(ColumnGroup cGroup) {
    return cGroup.getHeaderValue();
  }

  /**
   * Recupera o objeto que representa o valor de um agrupamento de colunas.
   * 
   * @param aColumn o ndice de uma coluna.
   * @return o objeto que representa o valor da coluna.
   */
  protected Object getColumnValue(TableColumn aColumn) {
    return aColumn.getHeaderValue();
  }

  /**
   * Inicializa o renderer responsvel por desenhar os agrupamentos.
   * 
   * @return o renderizador usado para desenhar os agrupamentos
   */
  protected TableCellRenderer createGroupRenderer() {
    return new DefaultTableCellRenderer() {
      @Override
      public Component getTableCellRendererComponent(JTable table,
        Object value, boolean isSelected, boolean hasFocus, int row, int column) {
        JTableHeader header = table.getTableHeader();
        if (header != null) {
          setForeground(header.getForeground());
          setBackground(header.getBackground());
          setFont(header.getFont());
        }
        setHorizontalAlignment(JLabel.CENTER);
        setText((value == null) ? "" : value.toString());
        setBorder(UIManager.getBorder("TableHeader.cellBorder"));
        return this;
      }
    };
  }

  /**
   * Mtodo que preenche o mapa que relaciona colunas com os agrupamentos que as
   * contm.
   */
  private void fillGroupsByColumnMap() {

    if (columnGroups == null) {
      return;
    }
    groupsByColumn = new HashMap<TableColumn, List<ColumnGroup>>();

    HashSet<TableColumn> colsSet =
      new HashSet<TableColumn>(Collections.list(this.getColumnModel()
        .getColumns()));
    HashSet<ColumnGroup> validGroups = new HashSet<ColumnGroup>();

    for (ColumnGroup cGroup : columnGroups) {
      fillFromGroup(cGroup, new ArrayList<ColumnGroup>(), colsSet, validGroups);
    }
    columnGroups.clear();
    columnGroups.addAll(validGroups);
  }

  /**
   * Mtodo recursivo que preenche, a partir do agrupamento recebido como
   * parmetro, o mapa que relaciona colunas com os agrupamentos que as contm. <br>
   * <br>
   * O agrupamento de colunas recebido como parmetro  percorrido at chegar ao
   * caso base, ou seja, at chegar a uma coluna que no seja uma instncia de
   * ColumnGroup. Neste momento,  inserido no mapa a coluna corrente e a lista
   * de agrupamentos de colunas percorridos at chegar  coluna atual. <br>
   * <br>
   * O resultado final das chamadas recursivas deste mtodo  a insero no mapa
   * das colunas folhas descendentes do agrupamento recebido como parmetro que
   * estiverem sendo usadas pelo modelo de colunas, referenciando a lista de
   * todos os grupamentos que contm estas colunas folhas. As colunas no usadas
   * pelo modelo de colunas no sero inseridas no mapa. <br>
   * <br>
   * Adicionalmente, este mtodo lista todos os grupos que so vlidos, ou seja,
   * que comtm ao menos uma coluna folha que seja usada pelo modelo.
   * 
   * @param cGroup Agrupamento de colunas a ser percorrido
   * @param allGroups Agrupamentos de colunas percorridos at chegar ao
   *        agrupamento atual.
   * @param modelColumns Colunas usadas pelo modelo de colunas.
   * @param validGroups Lista de grupos de colunas vlidos a ser preenchido.
   */
  private void fillFromGroup(ColumnGroup cGroup, List<ColumnGroup> allGroups,
    HashSet<TableColumn> modelColumns, Collection<ColumnGroup> validGroups) {
    List<Object> columns = cGroup.getAllColumns();
    allGroups.add(cGroup);
    for (Object col : columns) {
      ArrayList<ColumnGroup> antecessorsList = new ArrayList<ColumnGroup>();
      antecessorsList.addAll(allGroups);

      if (ColumnGroup.class.isInstance(col)) {
        fillFromGroup(ColumnGroup.class.cast(col), antecessorsList,
          modelColumns, validGroups);
      }
      else {
        if (modelColumns.contains(col)) {
          groupsByColumn.put(TableColumn.class.cast(col), antecessorsList);
          validGroups.add(antecessorsList.get(0));
        }
      }
    }
  }

  /**
   * Look and Feel de um header com agrupamentos de colunas.
   */
  protected class GroupableTableHeaderUI extends BasicTableHeaderUI {

    /**
     * Mtodo extendido apenas para permitir que uma subclasse tambm extenda
     * esse mtodo {@inheritDoc}
     */
    @Override
    protected MouseInputListener createMouseInputListener() {
      return super.createMouseInputListener();
    }

    /**
     * Recupera o componente usado para desenhar o header.
     * 
     * @return o componente usado para desenhar o header.
     */
    public JTableHeader getHeader() {
      return header;
    }

    /**
     * Informa a altura do header do agrupamento.
     * 
     * @param c componente a ser definido (ignorado).
     * 
     * @return dimenso do header do agrupamento.
     */
    @Override
    public Dimension getPreferredSize(JComponent c) {
      long width = 0;
      Enumeration<TableColumn> enumeration =
        header.getColumnModel().getColumns();
      while (enumeration.hasMoreElements()) {
        TableColumn aColumn = enumeration.nextElement();
        width = width + aColumn.getPreferredWidth();
      }
      return createHeaderSize(width);
    }

    /**
     * Informa a altura do header do agrupamento.
     * 
     * @param width largura do header.
     * 
     * @return dimenso do header.
     */
    private Dimension createHeaderSize(long width) {
      TableColumnModel columnModel = header.getColumnModel();
      width += (columnModel.getColumnMargin() * columnModel.getColumnCount());
      if (width > Integer.MAX_VALUE) {
        width = Integer.MAX_VALUE;
      }
      return new Dimension((int) width, getHeaderHeight());
    }

    /**
     * Informa a altura do header do agrupamento.
     * 
     * @return int altura do header.
     */
    private int getHeaderHeight() {
      int height = 0;
      TableColumnModel columnModel = header.getColumnModel();
      HashMap<ColumnGroup, Integer> cGroupsHeight =
        new HashMap<ColumnGroup, Integer>();
      for (int column = 0; column < columnModel.getColumnCount(); column++) {
        TableColumn aColumn = columnModel.getColumn(column);
        Object value = getColumnValue(aColumn);
        int cHeight = 0;
        if (value != null) {
          Component comp = getHeaderComponent(aColumn);
          cHeight = comp.getPreferredSize().height;
        }
        List<ColumnGroup> cGroups =
          ((GroupableTableHeader) header).getColumnGroups(aColumn);
        if (cGroups != null) {
          for (ColumnGroup cGroup : cGroups) {
            Integer groupHeight = cGroupsHeight.get(cGroup);
            if (groupHeight == null) {
              groupHeight = getHeight(cGroup);
              cGroupsHeight.put(cGroup, groupHeight);
            }
            cHeight += groupHeight;
          }
        }
        height = Math.max(height, cHeight);
      }
      return height;
    }

    /**
     * Informa a altura do agrupamento.
     * 
     * @param group o agrupamento.
     * @return altura do agrupamento.
     */
    public int getHeight(ColumnGroup group) {
      Component comp = getHeaderComponent(group);
      int height = comp.getPreferredSize().height;
      return height;
    }

    /**
     * Desenha o header com agrupamentos de colunas.
     * 
     * @param g grfico em que o header vai ser desenhado.
     * @param c componente a ser desenhado (ignorado).
     */
    @Override
    public void paint(Graphics g, JComponent c) {
      TableColumnModel columnModel = header.getColumnModel();
      if (columnModel == null || columnModel.getColumnCount() == 0) {
        return;
      }

      // Altura de cada nvel
      List<Integer> heights = getPreferredHeights();
      // y onde cada nvel comea
      List<Integer> ys = new ArrayList<Integer>();
      // Altura total
      int preferredHeight = 0;
      for (int height : heights) {
        ys.add(preferredHeight);
        preferredHeight += height;
      }

      columnGroupY.clear();
      HashMap<ColumnGroup, Rectangle> groupRectangles =
        new HashMap<ColumnGroup, Rectangle>();
      Dimension size = header.getSize();

      int x = 0;
      Enumeration<TableColumn> columns = header.getColumnModel().getColumns();
      while (columns.hasMoreElements()) {
        TableColumn aColumn = columns.nextElement();
        List<ColumnGroup> cGroups =
          ((GroupableTableHeader) header).getColumnGroups(aColumn);

        // Desenha o cabealho da coluna.
        int y = ys.get(cGroups == null ? 0 : cGroups.size());
        int height = size.height - y;
        int width = aColumn.getWidth();
        Rectangle cellRect = new Rectangle(x, y, width, height);

        setCellPosition(aColumn, cellRect, getColumnValue(aColumn));
        paintCell(g, cellRect, getHeaderComponent(aColumn));

        // Desenha os cabealhos de grupo ancestrais da coluna.
        if (cGroups != null) {
          for (int inx = 0; inx < cGroups.size(); inx++) {
            ColumnGroup cGroup = cGroups.get(inx);
            Rectangle groupRect = groupRectangles.get(cGroup);
            if (groupRect == null) {
              int groupY = ys.get(inx);
              int groupHeight = heights.get(inx);
              int groupWidth = cGroup.getWidth();

              groupRect = new Rectangle(x, groupY, groupWidth, groupHeight);
              groupRectangles.put(cGroup, groupRect);
              paintCell(g, groupRect, getHeaderComponent(cGroup));
            }
            setCellPosition(aColumn, groupRect, getGroupValue(cGroup));
          }
        }

        x += cellRect.width;
      }
    }

    /**
     * Guarda o nome apresentado em determinada posio do header.
     * 
     * @param column coluna referente ao nome.
     * @param cellRect retngulo que define a posio do nome.
     * @param cellValue valor apresentado.
     */
    private void setCellPosition(TableColumn column, Rectangle cellRect,
      Object cellValue) {

      Map<Rectangle, Object> yNames = columnGroupY.get(column);
      if (yNames == null) {
        yNames = new LinkedHashMap<Rectangle, Object>();
        columnGroupY.put(column, yNames);
      }
      yNames.put(cellRect, cellValue);
    }

    /**
     * Desenha o header de uma determinada coluna.
     * 
     * @param g grfico em que o header vai ser desenhado.
     * @param cellRect retngulo que define a dimenso do header.
     * @param component representao grfica de parte de um cabealho.
     */
    private void paintCell(Graphics g, Rectangle cellRect, Component component) {
      rendererPane.add(component);
      rendererPane.paintComponent(g, component, header, cellRect.x, cellRect.y,
        cellRect.width, cellRect.height, true);
    }

    /**
     * Obtm o componente grfico que ir representar o cabealho do grupo.
     * 
     * @param group o grupo a ser representado.
     * 
     * @return o componente grfico que ir representar o cabealho do grupo
     */
    private Component getHeaderComponent(ColumnGroup group) {
      TableCellRenderer renderer = group.getHeaderRenderer();
      if (renderer == null) {
        renderer = getDefaultRenderer();
      }

      return renderer.getTableCellRendererComponent(table,
        getGroupValue(group), false, false, -1, -1);
    }

    /**
     * Obtm o componente grfico que ir representar o cabealho da coluna.
     * 
     * @param column a coluna a ser representada.
     * 
     * @return o componente grfico que ir representar o cabealho da coluna.
     */
    private Component getHeaderComponent(TableColumn column) {
      TableCellRenderer renderer = column.getHeaderRenderer();
      if (renderer == null) {
        renderer = getDefaultRenderer();
      }
      return renderer.getTableCellRendererComponent(table,
        getColumnValue(column), false, false, -1, table
          .convertColumnIndexToView(column.getModelIndex()));
    }

    /**
     * Obtm uma lista contendo a altura preferida de cada nvel do cabealho.<br>
     * A altura do ndice 0 representa a altura do nvel mais alto do cabealho
     * e da por diante.
     * 
     * @return lista contendo a altura preferida de cada nvel do cabealho.
     */
    private List<Integer> getPreferredHeights() {
      List<Integer> heights = new ArrayList<Integer>();
      if (columnGroups != null && !columnGroups.isEmpty()) {
        for (ColumnGroup group : columnGroups) {
          fillPreferredHeights(group, 0, heights);
        }
      }
      else {
        int height = 0;
        Enumeration<TableColumn> columns = getColumnModel().getColumns();
        while (columns.hasMoreElements()) {
          int columnHeight =
            getHeaderComponent(columns.nextElement()).getPreferredSize().height;
          height = Math.max(height, columnHeight);
        }
        heights.add(height);
      }
      return heights;
    }

    /**
     * Preenche uma lista contendo a altura preferida de cada nvel do
     * cabealho.<br>
     * A altura do ndice 0 representa a altura do nvel mais alto do cabealho
     * e da por diante.
     * 
     * @param group grupo de colunas em um mesmo nvel.
     * @param level nvel do grupo decolunas.
     * @param heights lista de alturas a ser preenchida.
     */
    private void fillPreferredHeights(ColumnGroup group, int level,
      List<Integer> heights) {
      if (heights.size() < level) {
        throw new IllegalArgumentException("heights.size() < level");
      }
      if (table.getColumnCount() == 0) {
        return;
      }
      /*
       * Calcula a nova altura do nvel level como sendo a maior altura entre a
       * altura j calculada para aquele nvel e a altura necessria para
       * desenhar o cabealho de group.
       */
      if (heights.size() == level) {
        heights.add(0);
      }
      int currentHeight = heights.get(level);
      int groupHeight = getHeaderComponent(group).getPreferredSize().height;
      heights.set(level, Math.max(currentHeight, groupHeight));

      int childLevel = level + 1;
      for (Object child : group.getAllColumns()) {
        if (child instanceof TableColumn) {
          /*
           * Calcula a nova altura do nvel childLevel como sendo a maior altura
           * entre a altura j calculada para aquele nvel e a altura necessria
           * para desenhar o cabealho de column.
           */
          TableColumn column = (TableColumn) child;
          if (heights.size() < childLevel) {
            throw new IllegalArgumentException("heights.size() < level");
          }
          if (heights.size() == childLevel) {
            heights.add(0);
          }
          int childCurrentHeight = heights.get(childLevel);
          if (column.getHeaderValue() != null) {
            int childHeight =
              getHeaderComponent(column).getPreferredSize().height;
            heights.set(childLevel, Math.min(childCurrentHeight, childHeight));
          }
        }
        else {
          fillPreferredHeights((ColumnGroup) child, childLevel, heights);
        }
      }
    }
  }
}
