/* 
 * $Author$
 * $Date$
 * $Revision: 150399 $
 */
package tecgraf.javautils.gui.tree;

import java.awt.BorderLayout;
import java.awt.Color;
import java.awt.GridBagConstraints;
import java.awt.GridBagLayout;
import java.awt.Insets;
import java.awt.event.ActionEvent;
import java.awt.event.ActionListener;
import java.awt.event.FocusEvent;
import java.awt.event.FocusListener;
import java.util.regex.Matcher;
import java.util.regex.Pattern;
import java.util.regex.PatternSyntaxException;

import javax.swing.BorderFactory;
import javax.swing.JCheckBox;
import javax.swing.JPanel;
import javax.swing.JScrollPane;
import javax.swing.JTextField;
import javax.swing.JTree;
import javax.swing.ScrollPaneConstants;
import javax.swing.border.Border;
import javax.swing.event.DocumentEvent;
import javax.swing.event.DocumentListener;

import tecgraf.javautils.core.lng.LNG;

/**
 * Componente rvore filtrvel pelo rtulo do n: este componente possui um 
 * campo de texto onde o usurio digita o prefixo do rtulo que ele deseja e 
 * uma rvore que exibe apenas os nomes que possuem aquele prefixo.
 * 
 * @see FilterableTreePanelSample
 * 
 * @author lmoreira
 */
public final class FilterableTreePanel extends JPanel {

  /**
   * A borda de erro.
   */
  private static final Border ERROR_BORDER = 
    BorderFactory.createLineBorder(Color.RED, 2);
  
  /**
   * A borda padro do campo de texto.
   */
  private Border defaultTextFieldBorder;

  /**
   * Checkbox para indicar se deve ou no ignorar maisculas/minsculas.
   */
  private JCheckBox ignoreCaseCheckBox;
  
  /**
   * A rvore.
   */
  private Tree tree;
  
  /**
   * O campo de texto.
   */
  private JTextField textField;
  
  /**
   * Checkbox para indicar uso ou no de expresso regular.
   */
  private JCheckBox regexCheckBox;
  
  /**
   * Cria o painel.
   * 
   * @param root A raz da rvore (No aceita {@code null}).
   */
  public FilterableTreePanel(Node root) {
    tree = new Tree(root, new Filter() {
    
      public boolean isAccepted(Node node) {
        String label = node.getLabel();
        
        if (label == null) {
          label = "";
        }
        
        String text = textField.getText();
        if (regexCheckBox.isSelected()) {
          Pattern pattern = getRegexPattern();
          if (pattern == null) {
            return false;
          }
          Matcher matcher = pattern.matcher(label);
          return matcher.find();
        }
        
        if (ignoreCaseCheckBox.isSelected()) {
          label = label.toUpperCase();
          text = text.toUpperCase();
        }
        
        return label.contains(text);
      }
    
      public boolean isEnabled() {
        if (textField == null) {
          return false;
        }

        String text = textField.getText();
        
        if (text.length() == 0) {
          return false;
        }
        return true;
      }
      
    });
    textField = new JTextField();
    textField.getDocument().addDocumentListener(new DocumentListener() {
    
      public void removeUpdate(DocumentEvent e) {
        update();
      }
    
      public void insertUpdate(DocumentEvent e) {
        update();
      }
    
      public void changedUpdate(DocumentEvent e) {
        update();
      }
    });
    textField.addFocusListener(new FocusListener() {
    
      public void focusLost(FocusEvent e) {
        
      }
    
      public void focusGained(FocusEvent e) {
        String text = textField.getText();
        if (text.length() != 0) {
          textField.select(0, text.length());
        }
      }
    
    });
    defaultTextFieldBorder = textField.getBorder();
    ignoreCaseCheckBox = new JCheckBox(getMessage("ignoreCaseCheckBox.label"));
    ignoreCaseCheckBox.setSelected(true);
    ignoreCaseCheckBox.addActionListener(new ActionListener() {
      public void actionPerformed(ActionEvent e) {
        update();
      }
    });
    regexCheckBox = new JCheckBox(getMessage("regexCheckBox.label"));
    regexCheckBox.addActionListener(new ActionListener(){
      public void actionPerformed(ActionEvent e) {
        update();
        if (regexCheckBox.isSelected()) {
          ignoreCaseCheckBox.setEnabled(false);
        }
        else {
          ignoreCaseCheckBox.setEnabled(true);
        }
      }
    });
    JPanel searchPanel = new JPanel(new GridBagLayout());
    GridBagConstraints constraints = new GridBagConstraints();
    constraints.anchor = GridBagConstraints.NORTHWEST;
    constraints.fill = GridBagConstraints.NONE;
    constraints.gridx = 1;
    constraints.gridy = 1;
    constraints.gridwidth = 1;
    constraints.gridheight = 1;
    constraints.insets = new Insets(2, 2, 2, 2);
    constraints.weightx = 0.0;
    constraints.weighty = 0.0;
    searchPanel.add(ignoreCaseCheckBox, constraints);
    constraints.gridy++;
    searchPanel.add(regexCheckBox, constraints);
    constraints.gridy++;
    constraints.fill = GridBagConstraints.HORIZONTAL;
    constraints.weightx = 1.0;
    constraints.weighty = 0.0;
    searchPanel.add(textField, constraints);
    searchPanel.setBorder(BorderFactory.createEtchedBorder());
    setLayout(new BorderLayout());
    add(searchPanel, BorderLayout.NORTH);
    add(new JScrollPane(tree, ScrollPaneConstants.VERTICAL_SCROLLBAR_ALWAYS,
      ScrollPaneConstants.HORIZONTAL_SCROLLBAR_ALWAYS), BorderLayout.CENTER);
  }

  /**
   * Obtm o campo de texto.
   * 
   * @return O campo de texto.
   */
  public JTextField getTextField() {
    return textField;
  }

  /**
   * Obtm a rvore.
   * 
   * @return A rvore.
   */
  public JTree getTree() {
    return tree;
  }
  
  /**
   * Obtm uma mensagem no mecanismo de internacionalizao.
   * 
   * @param suffixKey O sufixo da chave. A chave ser o nome desta classe 
   * acrescido de '.' e o sufixo. (No aceita {@code null}).
   * 
   * @return A mensagem.
   */
  private String getMessage(String suffixKey) {
    return LNG.get(getClass().getName()+"."+suffixKey);
  }

  /**
   * Obtm o padro de expresso regular a ser utilizado na busca.
   * 
   * @return .
   */
  private Pattern getRegexPattern() {
    String text = textField.getText();
    try {
     return Pattern.compile(text); 
    }
    catch (PatternSyntaxException e) {
      return null;
    }
  }

  /**
   * Atualiza a rvore e o campo de busca.
   */
  private void update() {
    tree.filter();
    if (regexCheckBox.isSelected()) {
      if (getRegexPattern() == null) {
        textField.setBorder(ERROR_BORDER);
        return;
      }
    }
    textField.setBorder(defaultTextFieldBorder);
  }
}
