/*
 * $Id: Tree.java 65761 2007-07-25 18:16:50Z clinio $
 */
package tecgraf.javautils.gui.tree;

import java.awt.Component;
import java.util.ArrayList;
import java.util.Collections;
import java.util.HashMap;
import java.util.LinkedList;
import java.util.List;
import java.util.Map;

import javax.swing.Icon;
import javax.swing.JTree;
import javax.swing.event.TreeModelEvent;
import javax.swing.event.TreeModelListener;
import javax.swing.tree.DefaultTreeCellRenderer;
import javax.swing.tree.TreeModel;
import javax.swing.tree.TreePath;

/**
 * rvore Filtrvel: uma rvore que exibe ns que atendam a um determinado
 * critrio implementado por um {@link Filter filtro}.
 * 
 * @author Tecgraf/PUC-Rio
 */
public final class Tree extends JTree {
  /**
   * O filtro.
   */
  private Filter filter;

  /**
   * A lista de filhos visiveis indexada pelo pai.
   */
  private Map<Node, List<Node>> visibleChildrenByParentNode;

  /**
   * A raz da rvore.
   */
  private Node root;

  /**
   * Cria a rvore.
   * 
   * @param root A raz da rvore.
   * @param filter O filtro.
   */
  public Tree(Node root, Filter filter) {
    if (root == null) {
      throw new IllegalArgumentException("O parmetro root est nulo.");
    }
    this.root = root;
    if (filter == null) {
      throw new IllegalArgumentException("O parmetro filter est nulo.");
    }
    this.filter = filter;
    visibleChildrenByParentNode = new HashMap<Node, List<Node>>();
    setShowsRootHandles(true);
    setModel(createTreeModel());
    setCellRenderer(createCellRenderer());
    filter();
  }

  /**
   * Filtra a rvore: exibe apenas o n que atentam ao critrio do filtro, ou
   * seja, {@code Filter.isAccepted(Node) == true}.
   */
  public void filter() {
    visibleChildrenByParentNode.clear();
    populateVisibleChildrenByParentNode(root);
    final MyTreeModel tModel = (MyTreeModel) getModel();
    tModel.fireStructuredChanged();
    if (filter.isEnabled()) {
      collapsePath(getTreePath(root));
      updateTreeExpandableStatus(root);
    }
  }

  /**
   * Verifica se um n ou um desse seus ascendentes  aceito pelo filtro.
   * 
   * @param node O n (No aceita {@code null}).
   * 
   * @return .
   */
  private boolean accept(Node node) {
    if (!filter.isEnabled()) {
      return true;
    }
    Node currentNode = node;
    while (currentNode != null) {
      switch (currentNode.getFiltrageMode()) {
        case EVALUATE:
          if (filter.isAccepted(currentNode)) {
            return true;
          }
          break;
        case IGNORE:
          break;
        default:
          throw new IllegalStateException("Modo de filtragem inesperado: "
            + currentNode.getFiltrageMode());
      }
      currentNode = currentNode.getParent();
    }
    return false;
  }

  /**
   * Cria o {@link javax.swing.tree.TreeCellRenderer} desta rvore.
   * 
   * @return .
   */
  private DefaultTreeCellRenderer createCellRenderer() {
    return new DefaultTreeCellRenderer() {

      @Override
      public Component getTreeCellRendererComponent(JTree tree, Object value,
        boolean selectedItems, boolean expanded, boolean leaf, int row,
        boolean focusFlag) {
        super.getTreeCellRendererComponent(tree, value, selectedItems, expanded,
          leaf, row, focusFlag);
        Node node = (Node) value;
        String label = node.getLabel();
        if (label == null) {
          label = "";
        }
        setText(label);
        Icon icon = node.getIcon();
        if (icon != null) {
          setIcon(icon);
        }
        return this;
      }

    };
  }

  /**
   * Cria o {@link TreeModel} desta rvore.
   * 
   * @return .
   */
  private TreeModel createTreeModel() {
    return new MyTreeModel();
  }

  /**
   * Obtm um {@link TreePath} para um n da rvore.
   * 
   * @param node O n (No aceita {@code null}).
   * 
   * @return .
   */
  private TreePath getTreePath(Node node) {
    List<Node> nodes = new ArrayList<Node>();
    Node currentNode = node;
    while (currentNode != null) {
      nodes.add(currentNode);
      currentNode = currentNode.getParent();
    }
    Collections.reverse(nodes);
    TreePath path = new TreePath(nodes.toArray());
    return path;
  }

  /**
   * Preenche o mapa de filhos vsiveis.
   * 
   * @param node O n (No aceita {@code null}).
   * 
   * @return {@code true} se houve mudana no mapa.
   */
  private boolean populateVisibleChildrenByParentNode(Node node) {
    List<Node> visibleChildren = new LinkedList<Node>();
    for (Node child : node.getChildren()) {
      if (populateVisibleChildrenByParentNode(child)) {
        visibleChildren.add(child);
      }
    }
    if (!accept(node) && visibleChildren.isEmpty()) {
      return false;
    }
    this.visibleChildrenByParentNode.put(node, visibleChildren);
    return true;
  }

  /**
   * Atualiza o estado expandido/colapsado da rvore.
   * 
   * @param node O n (No aceita {@code null}).
   */
  private void updateTreeExpandableStatus(Node node) {
    TreePath treePath = getTreePath(node);
    for (Node child : node.getChildren()) {
      if (filter.isAccepted(child)) {
        expandPath(treePath);
        break;
      }
    }
    for (Node child : node.getChildren()) {
      updateTreeExpandableStatus(child);
    }
  }

  /**
   * A implementao de {@link TreeModel} desta rvore.
   */
  private final class MyTreeModel implements TreeModel {
    /**
     * Os observadores.
     */
    private List<TreeModelListener> listeners = new LinkedList<TreeModelListener>();

    /**
     * {@inheritDoc}
     */
    public void addTreeModelListener(TreeModelListener l) {
      listeners.add(l);
    }

    /**
     * {@inheritDoc}
     */
    public Object getChild(Object parent, int index) {
      List<Node> children = visibleChildrenByParentNode.get(parent);
      if (children == null) {
        return null;
      }
      return children.get(index);
    }

    /**
     * {@inheritDoc}
     */
    public int getChildCount(Object parent) {
      List<Node> children = visibleChildrenByParentNode.get(parent);
      if (children == null) {
        return 0;
      }
      return children.size();
    }

    /**
     * {@inheritDoc}
     */
    public int getIndexOfChild(Object parent, Object child) {
      List<Node> children = visibleChildrenByParentNode.get(parent);
      if (children == null) {
        return -1;
      }
      return children.indexOf(child);
    }

    /**
     * {@inheritDoc}
     */
    public Node getRoot() {
      return root;
    }

    /**
     * {@inheritDoc}
     */
    public boolean isLeaf(Object current) {
      Node currentNode = (Node) current;
      return currentNode.getChildren().isEmpty();
    }

    /**
     * {@inheritDoc}
     */
    public void removeTreeModelListener(TreeModelListener l) {
      listeners.remove(l);

    }

    /**
     * {@inheritDoc}
     */
    public void valueForPathChanged(TreePath path, Object newValue) {
    }

    /**
     * Dispara o evento
     * {@link TreeModelListener#treeStructureChanged(TreeModelEvent)}.
     */
    void fireStructuredChanged() {
      for (TreeModelListener listener : listeners) {
        listener.treeStructureChanged(new TreeModelEvent(Tree.this,
          new TreePath(root)));
      }
    }
  }
}
