package tecgraf.javautils.jexpression;

import java.util.ArrayList;
import java.util.List;

import tecgraf.javautils.jexpression.exception.JExpressionException;
import tecgraf.javautils.jexpression.exception.JExpressionSyntaxErrorException;
import tecgraf.javautils.jexpression.parser.JParser;
import tecgraf.javautils.jexpression.parser.model.BinaryOp;
import tecgraf.javautils.jexpression.parser.model.DoubleValue;
import tecgraf.javautils.jexpression.parser.model.Exp;
import tecgraf.javautils.jexpression.parser.model.Field;
import tecgraf.javautils.jexpression.parser.model.FunctionCall;
import tecgraf.javautils.jexpression.parser.model.Group;
import tecgraf.javautils.jexpression.parser.model.Index;
import tecgraf.javautils.jexpression.parser.model.Question;
import tecgraf.javautils.jexpression.parser.model.UnaryOp;
import tecgraf.javautils.jexpression.parser.model.Var;

/**
 * Objeto que encapsula uma expresso.
 * 
 * @author Tecgraf
 */
public class JExpression {

  /** rvore sinttica da expresso. */
  private Exp exp;

  /**
   * Construtor.
   * 
   * @param exp rvore sinttica da expresso.
   */
  private JExpression(Exp exp) {
    if (exp == null) {
      throw new IllegalArgumentException("exp no pode ser nulo.");
    }
    this.exp = exp;
  }

  /**
   * Compila a expresso dada e verifica se h erros de sintaxe.
   * 
   * @param input entrada.
   * @return objeto que representa a expresso.
   * @throws JExpressionSyntaxErrorException em caso de erro de sintaxe.
   */
  public static JExpression compile(String input)
    throws JExpressionSyntaxErrorException {
    if (input == null) {
      throw new IllegalArgumentException("input no pode ser nulo.");
    }
    Exp expression = new JParser().parse(input);
    return new JExpression(expression);
  }

  /**
   * Interpreta a expresso e executa as dadas callbacks de acordo com a
   * expresso.
   * 
   * @param handler objeto que encapsula as callbacks da expresso.
   * @return resultado da expresso.
   * 
   * @throws Exception em caso de erro durante a avaliao.
   */
  public Object eval(JExpressionHandler handler) throws Exception {
    return eval(handler, Object.class);
  }

  /**
   * Interpreta a expresso e executa as dadas callbacks de acordo com a
   * expresso.
   * 
   * @param handler objeto que encapsula as callbacks da expresso.
   * @param c classe do resultado da expresso.
   * @return resultado da expresso.
   * 
   * @throws Exception em caso de erro durante a avaliao.
   */
  @SuppressWarnings("unchecked")
  public <T> T eval(JExpressionHandler handler, Class<T> c) throws Exception {
    return (T) evaluate(exp, handler);
  }

  /**
   * Retorna a expresso compilada.
   * 
   * @return expresso compilada.
   */
  public String getCompiledExpression() {
    return exp.toString();
  }

  /**
   * Interpreta expresso.
   * 
   * @param expression expresso.
   * @param handler objeto que encapsula as callbacks da expresso.
   * @return resultado.
   * 
   * @throws Exception em caso de erro durante a avaliao.
   */
  private Object evaluate(Exp expression, JExpressionHandler handler)
    throws Exception {
    if (expression instanceof DoubleValue) {
      return evalueate((DoubleValue) expression, handler);
    }
    if (expression instanceof BinaryOp) {
      return evaluate((BinaryOp) expression, handler);
    }
    if (expression instanceof UnaryOp) {
      return evaluate((UnaryOp) expression, handler);
    }
    if (expression instanceof Field) {
      return evaluate((Field) expression, handler);
    }
    if (expression instanceof Index) {
      return evaluate((Index) expression, handler);
    }
    if (expression instanceof FunctionCall) {
      return evaluate((FunctionCall) expression, handler);
    }
    if (expression instanceof Group) {
      return evaluate((Group) expression, handler);
    }
    if (expression instanceof Question) {
      return evaluate((Question) expression, handler);
    }
    if (expression instanceof Var) {
      return evaluate((Var) expression, handler);
    }

    String f = "Erro durante a avaliao da expresso '%s'";
    throw new JExpressionException(String.format(f, expression));
  }

  /**
   * Interpreta {@link Double}.
   * 
   * @param valor valor em {@link Double}.
   * @param handler objeto que encapsula as callbacks da expresso.
   * @return resultado.
   * 
   * @throws Exception em caso de erro durante a avaliao.
   */
  private Object evalueate(DoubleValue valor, JExpressionHandler handler)
    throws Exception {
    return handler.handleDouble(valor.getValue());
  }

  /**
   * Interpreta operador binrio.
   * 
   * @param binaryOp operador binrio.
   * @param handler objeto que encapsula as callbacks da expresso.
   * @return resultado.
   * 
   * @throws Exception em caso de erro durante a avaliao.
   */
  private Object evaluate(BinaryOp binaryOp, JExpressionHandler handler)
    throws Exception {
    Object first = evaluate(binaryOp.getFirst(), handler);
    Object second = evaluate(binaryOp.getSecond(), handler);

    switch (binaryOp.getOp()) {
      case PLUS:
        return handler.handlePlus(first, second);
      case MINUS:
        return handler.handleMinus(first, second);
      case TIMES:
        return handler.handleTimes(first, second);
      case DIVIDE:
        return handler.handleDivision(first, second);
      case POW:
        return handler.handlePow(first, second);
      case GREATER:
        return handler.handleGreater(first, second);
      case GREATER_EQ:
        return handler.handleGreaterEqual(first, second);
      case LOWER:
        return handler.handleLower(first, second);
      case LOWER_EQ:
        return handler.handleLowerEqual(first, second);
      case EQUAL:
        return handler.handleEqual(first, second);
      case NOT_EQUAL:
        return handler.handleNotEqual(first, second);
      case AND:
        return handler.handleAnd(first, second);
      case OR:
        return handler.handleOr(first, second);
      default:
        String f = "Erro durante a avaliao da expresso '%s'";
        throw new JExpressionException(String.format(f, binaryOp));
    }
  }

  /**
   * Interpreta operador unrio.
   * 
   * @param unaryOp operador unrio.
   * @param handler objeto que encapsula as callbacks da expresso.
   * @return resultado.
   * 
   * @throws Exception em caso de erro durante a avaliao.
   */
  private Object evaluate(UnaryOp unaryOp, JExpressionHandler handler)
    throws Exception {
    Object result = evaluate(unaryOp.getExpression(), handler);

    switch (unaryOp.getOp()) {
      case NOT:
        return handler.handleNot(result);
      case MINUS:
        return handler.handleUnaryMinus(result);
      default:
        String f = "Erro durante a avaliao da expresso '%s'";
        throw new JExpressionException(String.format(f, unaryOp));
    }
  }

  /**
   * Interpreta chamada de funo.
   * 
   * @param functioncall chamada de funo.
   * @param handler objeto que encapsula as callbacks da expresso.
   * @return resultado.
   * 
   * @throws Exception em caso de erro durante a avaliao.
   */
  private Object evaluate(FunctionCall functioncall, JExpressionHandler handler)
    throws Exception {
    List<Object> params = new ArrayList<Object>();
    for (Exp param : functioncall.getParams()) {
      params.add(evaluate(param, handler));
    }
    return handler.handleFunctionCall(functioncall.getName(), params);
  }

  /**
   * Interpreta if-then-else.
   * 
   * @param question if-then-else.
   * @param handler objeto que encapsula as callbacks da expresso.
   * @return resultado.
   * 
   * @throws Exception em caso de erro durante a avaliao.
   */
  private Object evaluate(Question question, JExpressionHandler handler)
    throws Exception {
    Object condition = evaluate(question.getCondition(), handler);
    Object then = evaluate(question.getThen(), handler);
    Object otherwise = evaluate(question.getOtherwise(), handler);

    return handler.handleQuestion(condition, then, otherwise);
  }

  /**
   * Interpreta varivel.
   * 
   * @param var varivel.
   * @param handler objeto que encapsula as callbacks da expresso.
   * @return resultado.
   * 
   * @throws Exception em caso de erro durante a avaliao.
   */
  private Object evaluate(Var var, JExpressionHandler handler) throws Exception {
    return handler.handleVar(var.getName());
  }

  /**
   * Interpreta campo.
   * 
   * @param field campo.
   * @param handler objeto que encapsula as callbacks da expresso.
   * @return resultado.
   * 
   * @throws Exception em caso de erro durante a avaliao.
   */
  private Object evaluate(Field field, JExpressionHandler handler)
    throws Exception {
    Object result = evaluate(field.getExpression(), handler);
    return handler.handleField(result, field.getName());
  }

  /**
   * Interpreta indexao.
   * 
   * @param index indexao.
   * @param handler objeto que encapsula as callbacks da expresso.
   * @return resultado.
   * 
   * @throws Exception em caso de erro durante a avaliao.
   */
  private Object evaluate(Index index, JExpressionHandler handler)
    throws Exception {
    Object result = evaluate(index.getExpression(), handler);
    Object i = evaluate(index.getIndex(), handler);

    return handler.handleIndex(result, i);
  }

  /**
   * Interpreta agrupamento.
   * 
   * @param group agrupamento.
   * @param handler objeto que encapsula as callbacks da expresso.
   * @return resultado.
   * 
   * @throws Exception em caso de erro durante a avaliao.
   */
  private Object evaluate(Group group, JExpressionHandler handler)
    throws Exception {
    return evaluate(group.getExpression(), handler);
  }
}
