package tecgraf.javautils.jexpression.util;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

import tecgraf.javautils.jexpression.JExpressionHandler;
import tecgraf.javautils.jexpression.exception.JExpressionException;
import tecgraf.javautils.jexpression.util.function.BasicFunctions;
import tecgraf.javautils.jexpression.util.function.JExpressionFunction;

/**
 * Implementao de um tratador de expresses que adota as seguintes convenes:
 * 
 * 1 - Operadores aritmticos operam sobre {@link Double}.<br/>
 * 2 - Operadores relacionais operam sobre {@link Boolean}.<br/>
 * 3 - Operaes aritmticas operam usado uma tolerncia que pode ser definida
 * no construtor. <br/>
 * 4 - Variveis podem ser definidas a partir do mtodo
 * {@link CalculatorHandler#addVariable(String, Object)}. <br/>
 * 5 - Funes podem ser definidas a partir do mtodo
 * {@link CalculatorHandler#addFunction(JExpressionFunction)}.
 * 
 * @author Tecgraf
 */
public class CalculatorHandler implements JExpressionHandler {

  /** Tolerncia para doubles. */
  private double tolerance;

  /** Mapa do nome das variveis a seus valores. */
  private Map<String, Object> vars;

  /** Mapa que associa o nome das funes a suas respectivas implementaes. */
  private Map<String, JExpressionFunction> functions;

  /** Construtor. */
  public CalculatorHandler() {
    this(1e-6);
  }

  /**
   * Construtor.
   * 
   * @param tolerance tolerncia nas operaes de double.
   */
  public CalculatorHandler(double tolerance) {
    this.tolerance = tolerance;
    this.vars = new HashMap<String, Object>();
    this.functions = new HashMap<String, JExpressionFunction>();
  }

  /** Registra funes bsicas. */
  public void registerBasicFunctions() {
    for (BasicFunctions value : BasicFunctions.values()) {
      addFunction(value.function);
    }
  }

  /**
   * Adiciona uma varivel e seu respectivo valor.
   * 
   * @param name nome da varivel.
   * @param value valor.
   */
  public void addVariable(String name, Object value) {
    if (name == null) {
      throw new IllegalArgumentException("name no pode ser nulo.");
    }
    vars.put(name, value);
  }

  /**
   * Retorna true se existir uma varivel com o dado nome, false caso contrrio.
   * 
   * @param name nome da varivel.
   * @return flag que define se existe uma varivel com o dado nome.
   */
  public boolean hasVariable(String name) {
    if (name == null) {
      throw new IllegalArgumentException("name no pode ser nulo.");
    }
    return vars.containsKey(name);
  }

  /**
   * Remove varivel.
   * 
   * @param name nome da varivel.
   */
  public void removeVariable(String name) {
    if (!hasVariable(name)) {
      throw new IllegalArgumentException("no existe varivel com o dado nome.");
    }
    vars.remove(name);
  }

  /** Remove todas as variveis. */
  public void removeAllVariables() {
    vars.clear();
  }

  /**
   * Adiciona uma funo.
   * 
   * @param function funo.
   */
  public void addFunction(JExpressionFunction function) {
    if (function == null) {
      throw new IllegalArgumentException("function no pode ser nulo.");
    }
    functions.put(function.getName(), function);
  }

  /**
   * Retorna true se existir uma funo com o dado nome, false caso contrrio.
   * 
   * @param name nome da funo.
   * @return flag que define se existe uma varivel com o dado nome.
   */
  public boolean hasFunction(String name) {
    if (name == null) {
      throw new IllegalArgumentException("name no pode ser nulo.");
    }
    return functions.containsKey(name);
  }

  /**
   * Remove funo.
   * 
   * @param name nome da funo.
   */
  public void removeFunction(String name) {
    if (!hasFunction(name)) {
      throw new IllegalArgumentException("no existe funo com o dado nome.");
    }
    functions.remove(name);
  }

  /** Remove todas as funes. */
  public void removeAllFunctions() {
    functions.clear();
  }

  /**
   * Retorna a lista com o nome das variveis definidas.
   * 
   * @return lista com as variveis.
   */
  public List<String> getVarNames() {
    return new ArrayList<String>(vars.keySet());
  }

  /**
   * Retorna a lista com o nome das funes definidas.
   * 
   * @return lista com as funes.
   */
  public List<String> getFunctionNames() {
    return new ArrayList<String>(functions.keySet());
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public Object handleDouble(Double value) throws Exception {
    return value;
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public Object handlePlus(Object first, Object second) throws Exception {
    if (first instanceof Double && second instanceof Double) {
      return ((Double) first) + (Double) second;
    }

    String f = "('+') Erro na converso dos valores '%s' e '%s' para double.";
    throw new JExpressionException(String.format(f, first, second));
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public Object handleMinus(Object first, Object second) throws Exception {
    if (first instanceof Double && second instanceof Double) {
      return ((Double) first) - (Double) second;
    }
    String f = "('-') Erro na converso dos valores '%s' e '%s' para double.";
    throw new JExpressionException(String.format(f, first, second));
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public Object handleTimes(Object first, Object second) throws Exception {
    if (first instanceof Double && second instanceof Double) {
      return ((Double) first) * (Double) second;
    }

    String f = "('*') Erro na converso dos valores '%s' e '%s' para double.";
    throw new JExpressionException(String.format(f, first, second));
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public Object handleDivision(Object first, Object second) throws Exception {
    if (first instanceof Double && second instanceof Double) {
      return ((Double) first) / (Double) second;
    }
    String f = "('/') Erro na converso dos valores '%s' e '%s' para double.";
    throw new JExpressionException(String.format(f, first, second));
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public Object handlePow(Object first, Object second) throws Exception {
    if (first instanceof Double && second instanceof Double) {
      return Math.pow((Double) first, (Double) second);
    }
    String f = "('^') Erro na converso dos valores '%s' e '%s' para double.";
    throw new JExpressionException(String.format(f, first, second));
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public Object handleOr(Object first, Object second) throws Exception {
    if (first instanceof Boolean && second instanceof Boolean) {
      return ((Boolean) first) || (Boolean) second;
    }
    String f = "('||') Erro na converso dos valores '%s' e '%s' para boleano.";
    throw new JExpressionException(String.format(f, first, second));
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public Object handleAnd(Object first, Object second) throws Exception {
    if (first instanceof Boolean && second instanceof Boolean) {
      return ((Boolean) first) && (Boolean) second;
    }
    String f = "('&&') Erro na converso dos valores '%s' e '%s' para boleano.";
    throw new JExpressionException(String.format(f, first, second));
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public Object handleEqual(Object first, Object second) throws Exception {
    if (first instanceof Boolean && second instanceof Boolean) {
      return ((Boolean) first).equals(second);
    }
    if (first instanceof Double && second instanceof Double) {
      return eq((Double) first, (Double) second);
    }

    String f = "Erro na verificao de igualdade dos valores '%s' e '%s'.";
    throw new JExpressionException(String.format(f, first, second));
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public Object handleNotEqual(Object first, Object second) throws Exception {
    Boolean result = (Boolean) handleEqual(first, second);
    return !result;
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public Object handleGreater(Object first, Object second) throws Exception {
    if (first instanceof Double && second instanceof Double) {
      return ((Double) first) > (Double) second;
    }
    String f = "('>') Erro na converso dos valores '%s' e '%s' para double.";
    throw new JExpressionException(String.format(f, first, second));
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public Object handleGreaterEqual(Object first, Object second)
    throws Exception {
    if (first instanceof Double && second instanceof Double) {
      double a = (Double) first;
      double b = (Double) second;
      return a > b || eq(a, b);
    }
    String f = "('>=') Erro na converso dos valores '%s' e '%s' para double.";
    throw new JExpressionException(String.format(f, first, second));
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public Object handleLower(Object first, Object second) throws Exception {
    if (first instanceof Double && second instanceof Double) {
      return ((Double) first) < (Double) second;
    }
    String f = "('<') Erro na converso dos valores '%s' e '%s' para double.";
    throw new JExpressionException(String.format(f, first, second));
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public Object handleLowerEqual(Object first, Object second) throws Exception {
    if (first instanceof Double && second instanceof Double) {
      double a = (Double) first;
      double b = (Double) second;
      return a < b || eq(a, b);
    }
    String f = "('<=') Erro na converso dos valores '%s' e '%s' para double.";
    throw new JExpressionException(String.format(f, first, second));
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public Object handleQuestion(Object condition, Object then, Object otherwise)
    throws Exception {
    if (condition instanceof Boolean) {
      return (Boolean) condition ? then : otherwise;
    }
    String f = "('?') Erro na converso da condio '%s' para boleano.";
    throw new JExpressionException(String.format(f, condition));
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public Object handleUnaryMinus(Object object) throws Exception {
    if (object instanceof Double) {
      return -((Double) object);
    }
    String f = "('-' unrio) Erro na converso do valor '%s' para double.";
    throw new JExpressionException(String.format(f, object));
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public Object handleNot(Object object) throws Exception {
    if (object instanceof Boolean) {
      return !((Boolean) object);
    }

    String f = "('!') Erro na converso o valor '%s' para boleano.";
    throw new JExpressionException(String.format(f, object));
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public Object handleVar(String name) throws Exception {
    if (!hasVariable(name)) {
      String f = "Varivel '%s' no foi definida.";
      throw new JExpressionException(String.format(f, name));
    }
    return vars.get(name);
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public Object handleFunctionCall(String functionName, List<Object> params)
    throws Exception {
    if (!hasFunction(functionName)) {
      String f = "Funo '%s' no foi definida.";
      throw new JExpressionException(String.format(f, functionName));
    }
    JExpressionFunction function = functions.get(functionName);
    return function.call(params.toArray());
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public Object handleIndex(Object object, Object index) throws Exception {
    return null;
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public Object handleField(Object object, String field) throws Exception {
    return null;
  }

  /**
   * Verifica se dado dois valores double so iguais considerando uma tolerncia
   * entre seus valores.
   * 
   * @param a primeiro valor.
   * @param b segundo valor.
   * @return true se a distncia entre os dois valores for abaixo da tolerncia,
   *         false caso contrrio.
   */
  private boolean eq(double a, double b) {
    return a == b || Math.abs(a - b) <= tolerance;
  }
}
