package tecgraf.javautils.jexpression.parser;

import java.util.ArrayList;
import java.util.List;

import tecgraf.javautils.jexpression.exception.JExpressionSyntaxErrorException;
import tecgraf.javautils.jexpression.scanner.JScanner;
import tecgraf.javautils.jexpression.scanner.Symbol;
import tecgraf.javautils.jexpression.scanner.Token;

/**
 * Classe auxiliar que encapsula os tokens durante a anlise sinttica e
 * facilita verificarmos o token atual, anterior e o prximo.
 *
 * @author Tecgraf
 */
class TokenMemoization {

  /** Analisador lxico. */
  private JScanner scanner;

  /** Lista com os tokens. */
  private List<Token> tokens;

  /** ndice do token atual. */
  private int index;

  /**
   * Construtor.
   *
   * @param input entrada.
   */
  TokenMemoization(String input) {
    if (input == null) {
      throw new IllegalArgumentException("input no pode ser nulo.");
    }
    scanner = new JScanner(input);
    tokens = new ArrayList<>();
    index = 0;
  }

  /**
   * Retorna o token atual.
   *
   * @return token atual.
   *
   * @throws JExpressionSyntaxErrorException em caso de token desconhecido.
   */
  Token token() throws JExpressionSyntaxErrorException {
    if (index >= tokens.size()) {
      for (int i = 0; i <= index - tokens.size(); i++) {
        Token token = scanner.nextToken();
        if (token == null) {
          return null;
        }
        if (token.isUnknown()) {
          String f = "Caracter desconhecido '%s'";
          String msg = String.format(f, token.getValue());
          throw new JExpressionSyntaxErrorException(msg, token.getLineNumber());
        }
        tokens.add(token);
      }
    }

    return tokens.get(index);
  }

  /**
   * Verifica se o token atual  de um dado smbolo.
   *
   * @param symbol smbolo.
   * @return true se o token atual for de um dado smbolo, false caso contrrio.
   * @throws JExpressionSyntaxErrorException em caso de erro de sintaxe.
   */
  boolean isTerminal(Symbol... symbol) throws JExpressionSyntaxErrorException {
    Token token = token();
    if (token == null) {
      return false;
    }
    for (Symbol s : symbol) {
      if (s == token.getType()) {
        return true;
      }
    }
    return false;
  }

  /**
   * Verifica se ainda h tokens a serem consumidos.
   *
   * @return true se ainda existem tokens no-consumidos, false caso contrrio.
   *
   * @throws JExpressionSyntaxErrorException em caso de erro de sintaxe.
   */
  boolean hasMoreTokens() throws JExpressionSyntaxErrorException {
    return token() != null;
  }

  /** Indica que o token atual foi consumido e aponta para o prximo token. */
  void commit() {
    index++;
  }

  /** Indica que o token anterior ser o atual. */
  void rollback() {
    if (index != 0) {
      index--;
    }
  }
}
