package tecgraf.javautils.parsers;

import java.util.List;

import tecgraf.javautils.parsers.exception.AutomatonException;
import tecgraf.javautils.parsers.exception.InvalidStateException;
import tecgraf.javautils.parsers.iterators.SymbolIterator;

/**
 * Representa um analisador lxico (Parser) que atua sobre iteradores de
 * smbolos ({@link tecgraf.javautils.parsers.iterators.SymbolIterator}).
 */
public abstract class Parser {
  /**
   * O autmato finito que ser utilizado para fazer efetivamente o parser.
   */
  private FiniteAutomaton automaton;

  /**
   * Cria um parser que utilizar o autmato finito recebido.
   * 
   * @param automaton o autmato finito.
   */
  protected Parser(FiniteAutomaton automaton) {
    if (automaton == null) {
      final String err = "O autmato finito no pode ser nulo (null).";
      throw new IllegalArgumentException(err);
    }
    this.automaton = automaton;
  }

  /**
   * Gera os tokens a partir dos smbolos recebidos.
   * 
   * @param symbolIterator Um iterador para os smbolos.
   * 
   * @return Uma {@link List} com os tokens da linha recebida. Cada elemento da
   *         lista  do tipo {@link Token}.
   * 
   * @throws InvalidStateException Caso acabem os smbolos do iterador e o
   *         autmato esteja num estado no-final.
   * @throws AutomatonException Caso ocorra algum erro durante as iteraes.
   */
  protected final List<Token> parse(SymbolIterator symbolIterator)
    throws InvalidStateException, AutomatonException {
    return this.automaton.start(symbolIterator);
  }
}
