package busexplorer;

import javax.swing.SwingWorker;
import java.util.logging.Logger;

import org.omg.CORBA.COMM_FAILURE;
import org.omg.CORBA.NO_PERMISSION;
import org.omg.CORBA.ORB;
import org.omg.CORBA.ORBPackage.InvalidName;
import org.omg.CORBA.Object;
import org.omg.CORBA.TRANSIENT;
import org.omg.PortableServer.POAPackage.ServantNotActive;
import org.omg.PortableServer.POAPackage.WrongPolicy;
import com.google.common.collect.ArrayListMultimap;

import tecgraf.openbus.Connection;
import tecgraf.openbus.OfferObserver;
import tecgraf.openbus.OnReloginCallback;
import tecgraf.openbus.OpenBusContext;
import tecgraf.openbus.RemoteOffer;
import tecgraf.openbus.admin.BusAdminFacade;
import tecgraf.openbus.admin.BusAdminImpl;
import tecgraf.openbus.admin.BusAuditFacade;
import tecgraf.openbus.admin.BusAuditImpl;
import tecgraf.openbus.core.ORBInitializer;
import tecgraf.openbus.core.v2_1.services.ServiceFailure;
import tecgraf.openbus.core.v2_1.services.UnauthorizedOperation;
import tecgraf.openbus.core.v2_1.services.access_control.AccessDenied;
import tecgraf.openbus.core.v2_1.services.access_control.LoginInfo;
import tecgraf.openbus.core.v2_1.services.access_control.NoLoginCode;
import tecgraf.openbus.core.v2_1.services.access_control.TooManyAttempts;
import tecgraf.openbus.core.v2_1.services.access_control.UnknownDomain;
import tecgraf.openbus.core.v2_1.services.access_control.WrongEncoding;
import tecgraf.openbus.exception.AlreadyLoggedIn;
import tecgraf.openbus.extension.BusExtensionFacade;
import tecgraf.openbus.extension.BusExtensionImpl;

import busexplorer.utils.BusAddress;

/**
 * Trata, analisa e armazena dados de login no barramento.
 *
 * @author Tecgraf
 */
public class BusExplorerLogin {
  /** Entidade. */
  public final LoginInfo info;
  /** Endereo. */
  public final BusAddress address;
  /** Domnio de autenticao */
  public final String domain;
  /** Instncia de administrao do baramento. */
  public BusAdminFacade admin;
  /** Instncia de configurao da auditoria. */
  public BusAuditFacade audit;
  /** Instncia da fachada de extenso a governana. */
  public BusExtensionFacade extension;
  /** Tentativas de login */
  private static final short MAX_RETRIES = 3;
  /** Contexto de conexes OpenBus */
  private final OpenBusContext context;
  /** Indica se o login possui permisses administrativas. */
  private boolean adminRights = false;
  /** Conexo com o barramento */
  private Connection conn;
  /** Callback de Relogin do OpenBus SDK Java que pode ser customizada pela aplicao */
  private OnReloginCallback onReloginCallback;
  private final SwingWorker<Void, Void> worker;

  /**
   * Construtor para o objeto que representa o login no barramento.
   *  realizada a validao do endereo usando os mtodos fornecidos no {@link BusAddress}.
   *
   * @param address endereo do barramento
   * @param entity entidade a ser autenticada no barramento
   * @param domain domnio de autenticao para autenticar a entidade no barramento
   *
   * @see BusAddress#checkBusReference()
   * @see BusAddress#checkBusVersion()
   *
   * @throws InvalidName caso o ORB do OpenBus SDK Java no possua referncia para {@link OpenBusContext}
   */
  public BusExplorerLogin(BusAddress address, String entity, String domain) throws InvalidName {
    address.checkBusVersion();
    address.checkBusReference();

    this.info = new LoginInfo();
    this.info.entity = entity;
    this.address = address;
    this.domain = domain;
    this.context = (OpenBusContext) ORBInitializer.initORB()
      .resolve_initial_references("OpenBusContext");
    this.worker = new SwingWorker<Void, Void>() {
      @Override
      protected Void doInBackground() {
        context.ORB().run();
        return null;
      }

      @Override
      protected void done() {
        Logger.getLogger("tecgraf.openbus").info(String
          .format("ORB finalizou com sucesso para login do usurio %s no barramento %s", entity,
            address.toString()));
      }
    };
    this.worker.execute();
  }

  /** Permite a personalizao da {@link OnReloginCallback} presente no OpenBus SDK Java
   *  e garante o comportamento necessrio para atualizar as informaes armazenadas no {@link BusExplorerLogin}.
   *
   *  @param callback implementao personalizada do {@link OnReloginCallback}
   */
  public void onRelogin(OnReloginCallback callback) {
    this.onReloginCallback = callback;
  }

  /**
   * Realiza o login no barramento usando a senha fornecida nessa chamada.
   *
   * @param password Senha.
   *
   * @throws AccessDenied caso as credenciais (entidade ou senha) no sejam vlidas
   * @throws AlreadyLoggedIn caso a conexo com o barramento j esteja autenticada
   * @throws IllegalArgumentException caso no seja possvel determinar o {@code corbaloc} do endereo do barramento
   * @throws ServantNotActive caso no seja possvel instanciar o observador de ofertas do BusExtension
   * @throws ServiceFailure caso no seja possvel acessar o servio remoto
   * @throws TooManyAttempts caso tenham havido muitas tentativas repetidas de autenticao por senha no barramento com a mesma entidade
   * @throws UnknownDomain caso o domnio de autenticao no seja conhecido pelo barramento
   * @throws WrongEncoding caso tenha havido um erro na codificao do handshake do protocolo OpenBus (pode ser um erro interno na biblioteca do OpenBus SDK Java)
   * @throws WrongPolicy caso tenha havido um erro no adaptador de objetos que impediu a instanciao do observador de ofertas do BusExtension
   */
  public void doLogin(String password)
  throws WrongEncoding, AlreadyLoggedIn, ServiceFailure, UnknownDomain, TooManyAttempts, AccessDenied, ServantNotActive,
    WrongPolicy {
    Object reference = getORB().string_to_object(address.toIOR());
    conn = context.connectByReference(reference);
    context.defaultConnection(conn);

    boolean done = false;
    Exception lastFailure = null;
    for (short i = 0; i < MAX_RETRIES; i++) {
      try {
        conn.loginByPassword(info.entity, password.getBytes(), domain);
        conn.onReloginCallback((connection, oldLogin) -> {
          BusExplorerLogin.this.info.id = connection.login().id;
          try {
            BusExplorerLogin.this.checkAdminRights();
          } catch (ServiceFailure ex) {
            //TODO: a OnReloginCallback no aceita excees, exibir outro dilogo?
            ex.printStackTrace();
          }
          if (onReloginCallback != null) {
            onReloginCallback.onRelogin(connection, oldLogin);
          }
        });
        info.id = conn.login().id;
        admin = new BusAdminImpl(reference);
        audit = new BusAuditImpl(reference, conn);
        extension = new BusExtensionImpl(conn.offerRegistry());
        // observador do registro de ofertas do busextension
        ArrayListMultimap<String, String> busExtensionServiceProps = ArrayListMultimap.create();
        busExtensionServiceProps.put(BusExtensionImpl.SEARCH_CRITERIA_KEY, BusExtensionImpl.SEARCH_CRITERIA_VALUE);
        conn.offerRegistry().subscribeObserver(offer -> {
          try {
            // observador da remoo da nova oferta registrada
            offer.subscribeObserver(new OfferObserver() {
              @Override
              public void propertiesChanged(RemoteOffer offer) {
              }

              @Override
              public void removed(RemoteOffer offer) {
                // disparo da callback tambm quando a oferta for novamente removida
                onReloginCallback.onRelogin(context.defaultConnection(), conn.login());
              }
            });
          }
          catch (ServantNotActive | WrongPolicy ex) {
            // TODO: subscrio no aceita lanar exceo, exibir outro dilogo?
            ex.printStackTrace();
          }
          // callback que dispara o update nos paineis
          onReloginCallback.onRelogin(context.defaultConnection(), conn.login());
        }, busExtensionServiceProps);
        checkAdminRights();
        done = true;
        break;
      }
      catch (TRANSIENT | COMM_FAILURE e) {
        // retentar
        lastFailure = e;
      }
      catch (NO_PERMISSION e) {
        if (e.minor != NoLoginCode.value) {
          throw e;
        }
        // retentar
        lastFailure = e;
      }
      catch (Exception e) {
        logout();
        throw e;
      }

      try {
        Thread.sleep(250);
      }
      catch (InterruptedException ignored) {}
    }

    if (!done) {
      throw new IllegalArgumentException(address.toIOR(), lastFailure);
    }
  }

  /**
   * Indica se o usurio autenticado no momento possui permisso de
   * administrao.
   *
   * @return <code>true</code> se possui permisso de administrao e
   *         <code>false</code> caso contrrio. Se o login no foi realizado, o
   *         retorno ser <code>false</code>.
   */
  public boolean hasAdminRights() {
    return adminRights;
  }

  /**
   * Realiza o logout na conexo e o shutdown no ORB, interrompendo o loop de eventos CORBA.
   */
  public void logout() {
    try {
      if (conn != null) {
        org.omg.CORBA.ORB orb = conn.ORB();
        conn.logout();
        orb.shutdown(true);
      }
    } catch (Exception ignored) {}
  }

  /**
   * Realiza uma verificao sobre a permisso de administrao deste login e
   * armazena o resultado em uma membro auxiliar.
   *
   * @throws ServiceFailure caso acontea um erro imprevisto no servio remoto
   */
  private void checkAdminRights() throws ServiceFailure {
    try {
      admin.getLogins();
      adminRights = true;
    }
    catch (UnauthorizedOperation e) {
      adminRights = false;
    }
  }

  /**
   * Obtm o contexto da comunicao com o barramento.
   * @return o contexto da biblioteca do OpenBus.
   */
  public OpenBusContext getOpenBusContext() {
    return context;
  }

  /**
   * Obtm o ORB associado ao contexto da comunicao com o barramento.
   * @return a instncia de {@link ORB} obtida atravs do {@link OpenBusContext#ORB()}
   */
  public ORB getORB() {
    return getOpenBusContext().ORB();
  }
}
