/**
 * $Id: EnumerationListParameterImpl.java 132580 2012-08-22 21:41:28Z fpina $
 */
package tecgraf.openbus.algorithmservice.v1_1.parameters;

import java.security.InvalidParameterException;
import java.util.Vector;

/**
 * A classe <code>FormulaParameterImpl</code> implementa um valutype
 * <code>FormulaParameter</code>
 * 
 * @author Tecgraf
 */
public class EnumerationListParameterImpl extends EnumerationListParameter {
  /**
   * Vetor para armazenar os itens
   */
  private Vector<EnumerationItem> vectorItems = new Vector<EnumerationItem>();

  /**
   * Caractere que define o incio de uma lista.
   */
  private static final char START_LINE_CHAR = '{';

  /**
   * Caractere que define o fim de uma lista.
   */
  private static final char END_LINE_CHAR = '}';

  /**
   * Caractere que separa os elementos da lista.
   */
  private static final char ELEMENT_SEPARATOR_CHAR = ',';

  /**
   * Caractere que separa as listas.
   */
  private static final char LIST_SEPARATOR_CHAR = ':';

  /**
   * Formato de um item
   */
  private static final String ITEM_FORMAT =
    "{id=%s/label=%s/value=%s/description=%s/visible=%s}";

  /**
   * Construtor padro.
   */
  public EnumerationListParameterImpl() {
    this.selectedIds = new String[0];
  }

  /**
   * Construtor que recebe o valor do parmetro.
   * 
   * @param selectedIds rtulos do itens selecionados
   */
  public EnumerationListParameterImpl(String[] selectedIds) {
    this.selectedIds = selectedIds;
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public String convertToText() {
    String allItemsList = START_LINE_CHAR + getItems() + END_LINE_CHAR;
    String selectedItemsList =
      START_LINE_CHAR + getSelectedItems() + END_LINE_CHAR;

    return allItemsList + LIST_SEPARATOR_CHAR + selectedItemsList;
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public String getIDLType() {
    return EnumerationListParameterHelper.id();
  }

  /**
   * Obtm os itens da enumerao no formato definido em {@code ITEM_FORMAT}
   * 
   * @return todos os itens
   */
  private String getItems() {
    StringBuffer itemsAsString = new StringBuffer();

    for (int i = 0; i < this.items.length; i++) {
      itemsAsString.append(String.format(ITEM_FORMAT, this.items[i].id,
        this.items[i].label, this.items[i].value, this.items[i].description,
        this.items[i].visible));
      if (i < this.items.length - 1) {
        itemsAsString.append(ELEMENT_SEPARATOR_CHAR);
      }
    }

    return itemsAsString.toString();
  }

  /**
   * Obtm o label dos itens selecionados
   * 
   * @return retorna o label dos itens selecionados
   */
  private String getSelectedItems() {
    StringBuffer itemsAsString = new StringBuffer();

    for (int i = 0; i < this.selectedIds.length; i++) {
      itemsAsString.append(getLabel(this.selectedIds[i]));
      if (i < this.selectedIds.length - 1) {
        itemsAsString.append(ELEMENT_SEPARATOR_CHAR);
      }
    }

    return itemsAsString.toString();
  }

  /**
   * Obtm o label de um item
   * 
   * @param itemId o identificador do item
   * 
   * @return o label do item
   */
  private String getLabel(String itemId) {
    for (EnumerationItem item : this.items) {
      if (item.id.equals(itemId)) {
        return item.label;
      }
    }

    return null;
  }

  /**
   * Adiciona um item aos valores disponveis (usando na construo do
   * parmetro).
   * 
   * @param item item a ser adicionado
   */
  public void addItem(EnumerationItem item) {
    vectorItems.add(item);
    this.items = vectorItems.toArray(new EnumerationItem[vectorItems.size()]);
  }

  /**
   * Coloca os valores informados como os valores selecionados para o parmetro.
   * 
   * @param selectedIds identificadores dos valores selecionados
   * 
   * @throws InvalidParameterException caso algum valor seja invlido
   */
  public void setSelectedItems(String[] selectedIds)
    throws InvalidParameterException {
    if (validateItems(selectedIds)) {
      this.selectedIds = selectedIds;
    }
    else {
      throw new InvalidParameterException();
    }
  }

  /**
   * Verifica se os valores informados so validos (so uma das opes
   * disponveis).
   * 
   * @param itemsIds identificadores dos itens
   * 
   * @return true se todos os valores forem vlidos e false caso um valor no
   *         seja vlido
   */
  private boolean validateItems(String[] itemsIds) {
    if (this.items == null) {
      return false;
    }
    if (this.items.length == 0) {
      return false;
    }

    for (String itemId : itemsIds) {
      boolean isValid = false;
      for (EnumerationItem item : this.items) {
        if (item.id.equals(itemId)) {
          isValid = true;
          break;
        }
      }
      if (!isValid) {
        return false;
      }
    }
    return true;
  }
}