package br.pucrio.tecgraf.soma.job.application.appservice;

import br.pucrio.tecgraf.soma.job.api.model.JobResponse;
import br.pucrio.tecgraf.soma.job.application.service.JobService;
import br.pucrio.tecgraf.soma.job.application.service.ProjectService;
import br.pucrio.tecgraf.soma.job.domain.model.Job;
import br.pucrio.tecgraf.soma.job.domain.model.JobAlgorithm;
import org.jboss.logging.Logger;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;

import javax.ws.rs.ForbiddenException;
import javax.ws.rs.NotFoundException;
import java.util.List;

@Service
public class JobAppService {
  private static final Logger LOG = Logger.getLogger(JobAppService.class);

  @Value("${job.history.project.permission.check.enable}")
  public boolean checkProjectPermissionEnabled;

  @Autowired private ProjectService projectService;
  @Autowired private JobService jobService;

  @Transactional
  public List<JobAlgorithm> findDistinctAlgorithms(String rsqlQuery) {
    rsqlQuery = this.filterUserProjects(rsqlQuery);
    return jobService.findDistinctAlgorithms(rsqlQuery);
  }

  @Transactional
  public void editJobComment(String jobId, String newComment) throws ForbiddenException {
    Job job = null;
    try {
      job = jobService.findJobById(jobId);
    } catch (Exception e) {
      LOG.error(String.format("Error retrieving job with id %s", jobId), e);
    }
    if (job == null) {
      throw new NotFoundException(String.format("Job not found: %s", jobId));
    }

    if (!projectService.hasPermission(job.getProjectId())) {
      throw new ForbiddenException("User has no permission to edit this job");
    }

    try {
      jobService.editJobComment(jobId, newComment);
    } catch (Exception e) {
      LOG.error(String.format("Error updating job with id %s", jobId), e);
    }
  }

  @Transactional
  public void markJobAsDeleted(String jobId) {
    jobService.markJobAsDeleted(jobId);
  }

  @Transactional
  public void markJobsAsDeleted(List<String> jobIds) {
    jobService.markJobsAsDeleted(jobIds);
  }

  @Transactional
  public void findJobs(
      String rsqlQuery,
      int limit,
      int offset,
      boolean ascending,
      String sortAttribute,
      JobResponse response) {
    rsqlQuery = this.filterUserProjectsAndHandleDeleted(rsqlQuery);
    jobService.findJobs(rsqlQuery, limit, offset, ascending, sortAttribute, response);
  }

  @Transactional
  public void findGroupedJobs(
      String rsqlQuery,
      int limit,
      int offset,
      boolean ascending,
      String sortAttribute,
      JobResponse response) {
    rsqlQuery = this.filterUserProjectsAndHandleDeleted(rsqlQuery);
    jobService.findGroupedJobs(rsqlQuery, limit, offset, ascending, sortAttribute, response);
  }

  public String filterUserProjects(String q) {
    String projectsQuery = "";
    if (checkProjectPermissionEnabled) {
      LOG.info("Checking user project permission");
      projectsQuery = this.getUserProjectsPredicate("job.projectId");
    } else {
      LOG.info("Skipping user project permission check");
    }

    if (q == null || q.length() == 0) { // caso de query vazia
      q = projectsQuery;
    } else if (projectsQuery.length() > 0) { // predicado de projeto seja vazio
      q = String.format("(%s);(%s)", q, projectsQuery);
    }
    return q;
  }

  public String filterUserProjectsAndHandleDeleted(String q) {
    // Predicado de jobs não deletados
    String notDeletedQuery = "isDeleted==false";

    if (q == null || q.length() == 0) {
      // Query vazia vira query de jobs não deletados a query
      q = notDeletedQuery;
    } else if (!q.contains("isDeleted")) {
      // Se query nãp tem predicado para estado de deleção
      // adiciona predicado de jobs não deletados a query
      q = String.format("(%s);(%s)", q, notDeletedQuery);
    }

    // Se verificação de permissão de projetos está ativa, tenta adicionar predicado a query
    if (checkProjectPermissionEnabled) {
      LOG.info("Checking user project permission");
      String projectsQuery = this.getUserProjectsPredicate("projectId");

      // Adiciona predicado de projetos que o usuário tem acesso caso tenha
      if (projectsQuery.length() > 0) {
        q = String.format("(%s);(%s)", q, projectsQuery);
      }
    } else {
      LOG.info("Skipping user project permission check");
    }

    return q;
  }

  /**
   * Cria a predicado RSQL usando nome da coluna e lista de projetos que o usuário tem acesso. Por
   * exemplo: prj_col=in=(prj_1, prj_2).
   *
   * @param projectColumn nome da coluna com o identificador de projetos
   * @return predicado RSQL dos jobs do projeto
   * @throws NotFoundException Caso o usuário não tenha acesso a nenhum projeto
   */
  public String getUserProjectsPredicate(String projectColumn) throws NotFoundException {

    List<String> userProjects = projectService.getUserProjects();

    if (userProjects.isEmpty()) {
      String msg = "User has access to no projects";
      LOG.info(msg);
      throw new NotFoundException(msg);
    }

    StringBuilder projectsQuery = new StringBuilder(projectColumn);
    projectsQuery.append("=in=(");
    for (String project : userProjects) {
      projectsQuery.append(project);
      projectsQuery.append(",");
    }
    projectsQuery.deleteCharAt(projectsQuery.lastIndexOf(","));
    projectsQuery.append(")");

    return projectsQuery.toString();
  }
}
