/*
 * Copyright 2014 Goldman Sachs.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package com.gs.collections.impl.bag.mutable.primitive;

import java.util.Collection;
import java.util.Collections;

import com.gs.collections.api.CharIterable;
import com.gs.collections.api.LazyCharIterable;
import com.gs.collections.api.bag.MutableBag;
import com.gs.collections.api.bag.primitive.ImmutableCharBag;
import com.gs.collections.api.bag.primitive.MutableCharBag;
import com.gs.collections.api.block.function.primitive.CharToObjectFunction;
import com.gs.collections.api.block.predicate.primitive.CharPredicate;
import com.gs.collections.api.block.procedure.primitive.CharIntProcedure;
import com.gs.collections.api.iterator.CharIterator;
import com.gs.collections.impl.collection.mutable.primitive.AbstractSynchronizedCharCollection;
import com.gs.collections.impl.factory.primitive.CharBags;
import com.gs.collections.impl.lazy.primitive.LazyCharIterableAdapter;
import net.jcip.annotations.GuardedBy;
import net.jcip.annotations.ThreadSafe;

/**
 * A synchronized view of a {@link MutableCharBag}. It is imperative that the user manually synchronize on the collection when iterating over it using the
 * {@link CharIterator}, as per {@link Collections#synchronizedCollection(Collection)}.
 * <p>
 * This file was automatically generated from template file synchronizedPrimitiveBag.stg.
 *
 * @see MutableCharBag#asSynchronized()
 * @see MutableBag#asSynchronized()
 * @since 3.1.
 */
@ThreadSafe
public final class SynchronizedCharBag
        extends AbstractSynchronizedCharCollection
        implements MutableCharBag
{
    private static final long serialVersionUID = 1L;

    SynchronizedCharBag(MutableCharBag bag)
    {
        super(bag);
    }

    SynchronizedCharBag(MutableCharBag bag, Object newLock)
    {
        super(bag, newLock);
    }

    @GuardedBy("getLock()")
    private MutableCharBag getMutableCharBag()
    {
        return (MutableCharBag) this.getCharCollection();
    }

    @Override
    public SynchronizedCharBag with(char element)
    {
        synchronized (this.getLock())
        {
            this.getMutableCharBag().add(element);
        }
        return this;
    }

    @Override
    public SynchronizedCharBag without(char element)
    {
        synchronized (this.getLock())
        {
            this.getMutableCharBag().remove(element);
        }
        return this;
    }

    @Override
    public SynchronizedCharBag withAll(CharIterable elements)
    {
        synchronized (this.getLock())
        {
            this.getMutableCharBag().addAll(elements.toArray());
        }
        return this;
    }

    @Override
    public SynchronizedCharBag withoutAll(CharIterable elements)
    {
        synchronized (this.getLock())
        {
            this.getMutableCharBag().removeAll(elements);
        }
        return this;
    }

    public void addOccurrences(char item, int occurrences)
    {
        synchronized (this.getLock())
        {
            this.getMutableCharBag().addOccurrences(item, occurrences);
        }
    }

    public boolean removeOccurrences(char item, int occurrences)
    {
        synchronized (this.getLock())
        {
            return this.getMutableCharBag().removeOccurrences(item, occurrences);
        }
    }

    public int sizeDistinct()
    {
        synchronized (this.getLock())
        {
            return this.getMutableCharBag().sizeDistinct();
        }
    }

    public int occurrencesOf(char item)
    {
        synchronized (this.getLock())
        {
            return this.getMutableCharBag().occurrencesOf(item);
        }
    }

    public void forEachWithOccurrences(CharIntProcedure procedure)
    {
        synchronized (this.getLock())
        {
            this.getMutableCharBag().forEachWithOccurrences(procedure);
        }
    }

    @Override
    public MutableCharBag select(CharPredicate predicate)
    {
        synchronized (this.getLock())
        {
            return this.getMutableCharBag().select(predicate);
        }
    }

    @Override
    public MutableCharBag reject(CharPredicate predicate)
    {
        synchronized (this.getLock())
        {
            return this.getMutableCharBag().reject(predicate);
        }
    }

    @Override
    public <V> MutableBag<V> collect(CharToObjectFunction<? extends V> function)
    {
        synchronized (this.getLock())
        {
            return this.getMutableCharBag().collect(function);
        }
    }

    @Override
    public boolean equals(Object otherBag)
    {
        synchronized (this.getLock())
        {
            return this.getMutableCharBag().equals(otherBag);
        }
    }

    @Override
    public int hashCode()
    {
        synchronized (this.getLock())
        {
            return this.getMutableCharBag().hashCode();
        }
    }

    @Override
    public LazyCharIterable asLazy()
    {
        synchronized (this.getLock())
        {
            return new LazyCharIterableAdapter(this);
        }
    }

    @Override
    public MutableCharBag asUnmodifiable()
    {
        return new UnmodifiableCharBag(this);
    }

    @Override
    public MutableCharBag asSynchronized()
    {
        return this;
    }

    @Override
    public ImmutableCharBag toImmutable()
    {
        return CharBags.immutable.withAll(this);
    }
}
