/*
 * Copyright 2014 Goldman Sachs.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package com.gs.collections.impl.collection.mutable.primitive;

import java.io.Serializable;

import com.gs.collections.api.IntIterable;
import com.gs.collections.api.LazyIntIterable;
import com.gs.collections.api.bag.primitive.MutableIntBag;
import com.gs.collections.api.block.function.primitive.IntToObjectFunction;
import com.gs.collections.api.block.function.primitive.ObjectIntToObjectFunction;
import com.gs.collections.api.block.predicate.primitive.IntPredicate;
import com.gs.collections.api.block.procedure.primitive.IntProcedure;
import com.gs.collections.api.collection.MutableCollection;
import com.gs.collections.api.collection.primitive.ImmutableIntCollection;
import com.gs.collections.api.collection.primitive.MutableIntCollection;
import com.gs.collections.api.iterator.MutableIntIterator;
import com.gs.collections.api.list.primitive.MutableIntList;
import com.gs.collections.api.set.primitive.MutableIntSet;
import com.gs.collections.impl.lazy.primitive.LazyIntIterableAdapter;
import net.jcip.annotations.GuardedBy;
import net.jcip.annotations.ThreadSafe;

/**
 * This file was automatically generated from template file abstractSynchronizedPrimitiveCollection.stg.
 *
 * @since 3.1.
 */
@ThreadSafe
public abstract class AbstractSynchronizedIntCollection
        implements MutableIntCollection, Serializable
{
    private static final long serialVersionUID = 1L;

    private final Object lock;
    @GuardedBy("this.lock")
    private final MutableIntCollection collection;

    protected AbstractSynchronizedIntCollection(MutableIntCollection collection)
    {
        this(collection, null);
    }

    protected AbstractSynchronizedIntCollection(MutableIntCollection collection, Object newLock)
    {
        this.collection = collection;
        this.lock = newLock == null ? this : newLock;
    }

    protected Object getLock()
    {
        return this.lock;
    }

    protected MutableIntCollection getIntCollection()
    {
        return this.collection;
    }

    public int size()
    {
        synchronized (this.lock)
        {
            return this.collection.size();
        }
    }

    public boolean isEmpty()
    {
        synchronized (this.lock)
        {
            return this.collection.isEmpty();
        }
    }

    public boolean notEmpty()
    {
        synchronized (this.lock)
        {
            return this.collection.notEmpty();
        }
    }

    public void clear()
    {
        synchronized (this.lock)
        {
            this.collection.clear();
        }
    }

    public MutableIntCollection select(IntPredicate predicate)
    {
        synchronized (this.lock)
        {
            return this.collection.select(predicate);
        }
    }

    public MutableIntCollection reject(IntPredicate predicate)
    {
        synchronized (this.lock)
        {
            return this.collection.reject(predicate);
        }
    }

    public <V> MutableCollection<V> collect(IntToObjectFunction<? extends V> function)
    {
        synchronized (this.lock)
        {
            return this.collection.collect(function);
        }
    }

    public MutableIntCollection with(int element)
    {
        synchronized (this.lock)
        {
            this.add(element);
        }
        return this;
    }

    public MutableIntCollection without(int element)
    {
        synchronized (this.lock)
        {
            this.remove(element);
        }
        return this;
    }

    public MutableIntCollection withAll(IntIterable elements)
    {
        synchronized (this.lock)
        {
            this.addAll(elements);
        }
        return this;
    }

    public MutableIntCollection withoutAll(IntIterable elements)
    {
        synchronized (this.lock)
        {
            this.removeAll(elements);
        }
        return this;
    }

    public MutableIntCollection asUnmodifiable()
    {
        return new UnmodifiableIntCollection(this);
    }

    public MutableIntCollection asSynchronized()
    {
        return this;
    }

    public ImmutableIntCollection toImmutable()
    {
        synchronized (this.lock)
        {
            return this.collection.toImmutable();
        }
    }

    public LazyIntIterable asLazy()
    {
        return new LazyIntIterableAdapter(this);
    }

    public boolean contains(int value)
    {
        synchronized (this.lock)
        {
            return this.collection.contains(value);
        }
    }

    public boolean containsAll(int... source)
    {
        synchronized (this.lock)
        {
            return this.collection.containsAll(source);
        }
    }

    public boolean containsAll(IntIterable source)
    {
        synchronized (this.lock)
        {
            return this.collection.containsAll(source);
        }
    }

    public boolean add(int newItem)
    {
        synchronized (this.lock)
        {
            return this.collection.add(newItem);
        }
    }

    public boolean addAll(int... source)
    {
        synchronized (this.lock)
        {
            return this.collection.addAll(source);
        }
    }

    public boolean addAll(IntIterable source)
    {
        synchronized (this.lock)
        {
            return this.collection.addAll(source);
        }
    }

    public boolean remove(int value)
    {
        synchronized (this.lock)
        {
            return this.collection.remove(value);
        }
    }

    public boolean removeAll(IntIterable source)
    {
        synchronized (this.lock)
        {
            return this.collection.removeAll(source);
        }
    }

    public boolean removeAll(int... source)
    {
        synchronized (this.lock)
        {
            return this.collection.removeAll(source);
        }
    }

    public boolean retainAll(IntIterable source)
    {
        synchronized (this.lock)
        {
            return this.collection.retainAll(source);
        }
    }

    public boolean retainAll(int... source)
    {
        synchronized (this.lock)
        {
            return this.collection.retainAll(source);
        }
    }

    /**
     * Must be called in a synchronized block.
     */
    public MutableIntIterator intIterator()
    {
        return this.collection.intIterator();
    }

    public void forEach(IntProcedure procedure)
    {
        synchronized (this.lock)
        {
            this.collection.forEach(procedure);
        }
    }

    public int count(IntPredicate predicate)
    {
        synchronized (this.lock)
        {
            return this.collection.count(predicate);
        }
    }

    public boolean anySatisfy(IntPredicate predicate)
    {
        synchronized (this.lock)
        {
            return this.collection.anySatisfy(predicate);
        }
    }

    public boolean allSatisfy(IntPredicate predicate)
    {
        synchronized (this.lock)
        {
            return this.collection.allSatisfy(predicate);
        }
    }

    public boolean noneSatisfy(IntPredicate predicate)
    {
        synchronized (this.lock)
        {
            return this.collection.noneSatisfy(predicate);
        }
    }

    public int detectIfNone(IntPredicate predicate, int ifNone)
    {
        synchronized (this.lock)
        {
            return this.collection.detectIfNone(predicate, ifNone);
        }
    }

    public long sum()
    {
        synchronized (this.lock)
        {
            return this.collection.sum();
        }
    }

    public int max()
    {
        synchronized (this.lock)
        {
            return this.collection.max();
        }
    }

    public int min()
    {
        synchronized (this.lock)
        {
            return this.collection.min();
        }
    }

    public int minIfEmpty(int defaultValue)
    {
        synchronized (this.lock)
        {
            return this.collection.minIfEmpty(defaultValue);
        }
    }

    public int maxIfEmpty(int defaultValue)
    {
        synchronized (this.lock)
        {
            return this.collection.maxIfEmpty(defaultValue);
        }
    }

    public double average()
    {
        synchronized (this.lock)
        {
            return this.collection.average();
        }
    }

    public double median()
    {
        synchronized (this.lock)
        {
            return this.collection.median();
        }
    }

    public MutableIntList toSortedList()
    {
        synchronized (this.lock)
        {
            return this.collection.toSortedList();
        }
    }

    public int[] toSortedArray()
    {
        synchronized (this.lock)
        {
            return this.collection.toSortedArray();
        }
    }

    public int[] toArray()
    {
        synchronized (this.lock)
        {
            return this.collection.toArray();
        }
    }

    @Override
    public String toString()
    {
        synchronized (this.lock)
        {
            return this.collection.toString();
        }
    }

    public String makeString()
    {
        synchronized (this.lock)
        {
            return this.collection.makeString();
        }
    }

    public String makeString(String separator)
    {
        synchronized (this.lock)
        {
            return this.collection.makeString(separator);
        }
    }

    public String makeString(String start, String separator, String end)
    {
        synchronized (this.lock)
        {
            return this.collection.makeString(start, separator, end);
        }
    }

    public void appendString(Appendable appendable)
    {
        synchronized (this.lock)
        {
            this.collection.appendString(appendable);
        }
    }

    public void appendString(Appendable appendable, String separator)
    {
        synchronized (this.lock)
        {
            this.collection.appendString(appendable, separator);
        }
    }

    public void appendString(
            Appendable appendable,
            String start,
            String separator,
            String end)
    {
        synchronized (this.lock)
        {
            this.collection.appendString(appendable, start, separator, end);
        }
    }

    public MutableIntList toList()
    {
        synchronized (this.lock)
        {
            return this.collection.toList();
        }
    }

    public MutableIntSet toSet()
    {
        synchronized (this.lock)
        {
            return this.collection.toSet();
        }
    }

    public MutableIntBag toBag()
    {
        synchronized (this.lock)
        {
            return this.collection.toBag();
        }
    }

    public <T> T injectInto(T injectedValue, ObjectIntToObjectFunction<? super T, ? extends T> function)
    {
        synchronized (this.lock)
        {
            return this.collection.injectInto(injectedValue, function);
        }
    }
}
