/*
 * Copyright 2014 Goldman Sachs.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package com.gs.collections.impl.stack.mutable.primitive;

import java.io.Externalizable;
import java.io.IOException;
import java.io.ObjectInput;
import java.io.ObjectOutput;
import java.util.EmptyStackException;

import com.gs.collections.api.DoubleIterable;
import com.gs.collections.api.LazyDoubleIterable;
import com.gs.collections.api.bag.primitive.MutableDoubleBag;
import com.gs.collections.api.block.function.primitive.DoubleToObjectFunction;
import com.gs.collections.api.block.function.primitive.ObjectDoubleToObjectFunction;
import com.gs.collections.api.block.predicate.primitive.DoublePredicate;
import com.gs.collections.api.block.procedure.primitive.DoubleProcedure;
import com.gs.collections.api.iterator.DoubleIterator;
import com.gs.collections.api.list.primitive.DoubleList;
import com.gs.collections.api.list.primitive.MutableDoubleList;
import com.gs.collections.api.set.primitive.MutableDoubleSet;
import com.gs.collections.api.stack.MutableStack;
import com.gs.collections.api.stack.primitive.DoubleStack;
import com.gs.collections.api.stack.primitive.ImmutableDoubleStack;
import com.gs.collections.api.stack.primitive.MutableDoubleStack;
import com.gs.collections.impl.bag.mutable.primitive.DoubleHashBag;
import com.gs.collections.impl.factory.primitive.DoubleStacks;
import com.gs.collections.impl.iterator.UnmodifiableDoubleIterator;
import com.gs.collections.impl.lazy.primitive.LazyDoubleIterableAdapter;
import com.gs.collections.impl.list.mutable.primitive.DoubleArrayList;
import com.gs.collections.impl.set.mutable.primitive.DoubleHashSet;
import com.gs.collections.impl.stack.mutable.ArrayStack;
import net.jcip.annotations.NotThreadSafe;

/**
 * DoubleArrayStack is similar to {@link ArrayStack}, and is memory-optimized for double primitives.
 * This file was automatically generated from template file primitiveArrayStack.stg.
 */
@NotThreadSafe
public final class DoubleArrayStack
        implements MutableDoubleStack, Externalizable
{
    private static final long serialVersionUID = 1L;

    private transient DoubleArrayList delegate;

    public DoubleArrayStack()
    {
        this.delegate = new DoubleArrayList();
    }

    private DoubleArrayStack(int size)
    {
        this.delegate = new DoubleArrayList(size);
    }

    private DoubleArrayStack(double... items)
    {
        this.delegate = new DoubleArrayList(items);
    }

    public static DoubleArrayStack newStackFromTopToBottom(double... items)
    {
        DoubleArrayStack stack = new DoubleArrayStack(items.length);
        for (int i = items.length - 1; i >= 0; i--)
        {
            stack.push(items[i]);
        }
        return stack;
    }

    public static DoubleArrayStack newStackWith(double... items)
    {
        return new DoubleArrayStack(items);
    }

    public static DoubleArrayStack newStack(DoubleIterable items)
    {
        DoubleArrayStack stack = new DoubleArrayStack(items.size());
        stack.delegate = DoubleArrayList.newList(items);
        return stack;
    }

    public static DoubleArrayStack newStackFromTopToBottom(DoubleIterable items)
    {
        DoubleArrayStack stack = new DoubleArrayStack(items.size());
        stack.delegate = DoubleArrayList.newList(items).reverseThis();
        return stack;
    }

    public void push(double item)
    {
        this.delegate.add(item);
    }

    public double pop()
    {
        this.checkEmptyStack();
        return this.delegate.removeAtIndex(this.delegate.size() - 1);
    }

    private void checkEmptyStack()
    {
        if (this.delegate.isEmpty())
        {
            throw new EmptyStackException();
        }
    }

    public DoubleList pop(int count)
    {
        this.checkPositiveValueForCount(count);
        this.checkSizeLessThanCount(count);
        if (count == 0)
        {
            return new DoubleArrayList(0);
        }
        MutableDoubleList subList = new DoubleArrayList(count);
        while (count > 0)
        {
            subList.add(this.pop());
            count--;
        }
        return subList;
    }

    private void checkPositiveValueForCount(int count)
    {
        if (count < 0)
        {
            throw new IllegalArgumentException("Count must be positive but was " + count);
        }
    }

    private void checkSizeLessThanCount(int count)
    {
        if (this.delegate.size() < count)
        {
            throw new IllegalArgumentException("Count must be less than size: Count = " + count + " Size = " + this.delegate.size());
        }
    }

    public MutableDoubleStack select(DoublePredicate predicate)
    {
        return newStackFromTopToBottom(this.delegate.asReversed().select(predicate));
    }

    public MutableDoubleStack reject(DoublePredicate predicate)
    {
        return newStackFromTopToBottom(this.delegate.asReversed().reject(predicate));
    }

    public double peek()
    {
        this.checkEmptyStack();
        return this.delegate.getLast();
    }

    public DoubleList peek(int count)
    {
        this.checkPositiveValueForCount(count);
        this.checkSizeLessThanCount(count);
        if (count == 0)
        {
            return new DoubleArrayList(0);
        }
        MutableDoubleList subList = new DoubleArrayList(count);
        int index = this.delegate.size() - 1;
        for (int i = 0; i < count; i++)
        {
            subList.add(this.delegate.get(index - i));
        }
        return subList;
    }

    public double peekAt(int index)
    {
        this.rangeCheck(index);
        return this.delegate.get(this.delegate.size() - 1 - index);
    }

    private void rangeCheck(int index)
    {
        if (index < 0 || index > this.delegate.size() - 1)
        {
            throw new IllegalArgumentException("Index " + index + " out of range.Should be between 0 and " + (this.delegate.size() - 1));
        }
    }

    public DoubleIterator doubleIterator()
    {
        return new UnmodifiableDoubleIterator(this.delegate.asReversed().doubleIterator());
    }

    public void forEach(DoubleProcedure procedure)
    {
        this.delegate.asReversed().forEach(procedure);
    }

    public int size()
    {
        return this.delegate.size();
    }

    public boolean isEmpty()
    {
        return this.delegate.isEmpty();
    }

    public boolean notEmpty()
    {
        return this.delegate.notEmpty();
    }

    public int count(DoublePredicate predicate)
    {
        return this.delegate.asReversed().count(predicate);
    }

    public boolean anySatisfy(DoublePredicate predicate)
    {
        return this.delegate.asReversed().anySatisfy(predicate);
    }

    public boolean allSatisfy(DoublePredicate predicate)
    {
        return this.delegate.asReversed().allSatisfy(predicate);
    }

    public boolean noneSatisfy(DoublePredicate predicate)
    {
        return this.delegate.asReversed().noneSatisfy(predicate);
    }

    public double detectIfNone(DoublePredicate predicate, double ifNone)
    {
        return this.delegate.asReversed().detectIfNone(predicate, ifNone);
    }

    public <V> MutableStack<V> collect(DoubleToObjectFunction<? extends V> function)
    {
        return ArrayStack.newStackFromTopToBottom(this.delegate.asReversed().collect(function));
    }

    public <V> V injectInto(V injectedValue, ObjectDoubleToObjectFunction<? super V, ? extends V> function)
    {
        return this.delegate.asReversed().injectInto(injectedValue, function);
    }

    public double sum()
    {
        return this.delegate.sum();
    }

    public double max()
    {
        return this.delegate.max();
    }

    public double min()
    {
        return this.delegate.min();
    }

    public double minIfEmpty(double defaultValue)
    {
        if (this.isEmpty())
        {
            return defaultValue;
        }
        return this.min();
    }

    public double maxIfEmpty(double defaultValue)
    {
        if (this.isEmpty())
        {
            return defaultValue;
        }
        return this.max();
    }

    public double average()
    {
        return this.delegate.average();
    }

    public double median()
    {
        return this.delegate.median();
    }

    public double[] toSortedArray()
    {
        return this.delegate.toSortedArray();
    }

    public double[] toArray()
    {
        return this.delegate.asReversed().toArray();
    }

    public boolean contains(double value)
    {
        return this.delegate.asReversed().contains(value);
    }

    public boolean containsAll(double... source)
    {
        return this.delegate.asReversed().containsAll(source);
    }

    public boolean containsAll(DoubleIterable source)
    {
        return this.delegate.asReversed().containsAll(source);
    }

    public void clear()
    {
        this.delegate.clear();
    }

    @Override
    public boolean equals(Object otherStack)
    {
        if (otherStack == this)
        {
            return true;
        }
        if (!(otherStack instanceof DoubleStack))
        {
            return false;
        }
        DoubleStack stack = (DoubleStack) otherStack;
        if (this.size() != stack.size())
        {
            return false;
        }
        for (int i = 0; i < this.size(); i++)
        {
            if (Double.compare(this.peekAt(i), stack.peekAt(i)) != 0)
            {
                return false;
            }
        }
        return true;
    }

    @Override
    public int hashCode()
    {
        int hashCode = 1;
        DoubleIterable iterable = this.delegate.asReversed();
        DoubleIterator iterator = iterable.doubleIterator();
        while (iterator.hasNext())
        {
            double item = iterator.next();
            hashCode = 31 * hashCode + (int) (Double.doubleToLongBits(item) ^ Double.doubleToLongBits(item) >>> 32);
        }
        return hashCode;
    }

    @Override
    public String toString()
    {
        return this.delegate.asReversed().toString();
    }

    public String makeString()
    {
        return this.delegate.asReversed().makeString();
    }

    public String makeString(String separator)
    {
        return this.delegate.asReversed().makeString(separator);
    }

    public String makeString(String start, String separator, String end)
    {
        return this.delegate.asReversed().makeString(start, separator, end);
    }

    public void appendString(Appendable appendable)
    {
        this.delegate.asReversed().appendString(appendable);
    }

    public void appendString(Appendable appendable, String separator)
    {
        this.delegate.asReversed().appendString(appendable, separator);
    }

    public void appendString(
            Appendable appendable,
            String start,
            String separator,
            String end)
    {
        this.delegate.asReversed().appendString(appendable, start, separator, end);
    }

    public MutableDoubleList toList()
    {
        return DoubleArrayList.newList(this);
    }

    public MutableDoubleList toSortedList()
    {
        return DoubleArrayList.newList(this).sortThis();
    }

    public MutableDoubleSet toSet()
    {
        return DoubleHashSet.newSet(this);
    }

    public MutableDoubleBag toBag()
    {
        return DoubleHashBag.newBag(this);
    }

    public LazyDoubleIterable asLazy()
    {
        return new LazyDoubleIterableAdapter(this);
    }

    public MutableDoubleStack asUnmodifiable()
    {
        return new UnmodifiableDoubleStack(this);
    }

    public MutableDoubleStack asSynchronized()
    {
        return new SynchronizedDoubleStack(this);
    }

    public ImmutableDoubleStack toImmutable()
    {
        return DoubleStacks.immutable.withAll(this.delegate);
    }

    public void writeExternal(ObjectOutput out) throws IOException
    {
        out.writeInt(this.size());
        DoubleIterator iterator = this.delegate.asReversed().doubleIterator();
        while (iterator.hasNext())
        {
            double each = iterator.next();
            out.writeDouble(each);
        }
    }

    public void readExternal(ObjectInput in) throws IOException
    {
        int size = in.readInt();
        double[] array = new double[size];
        for (int i = size - 1; i >= 0; i--)
        {
            array[i] = in.readDouble();
        }
        this.delegate = DoubleArrayList.newListWith(array);
    }
}
