/**
 * Fixed mask formatter. Uses strings like "##-####-####" to specify the
 * desired format, which must be enforced by the formatter/validators.
 *
 *  Leonardo Quijano (based on MaskEdit component in Tapestry).
 *  Mar 2006
 *
 * The "format" flag specifies the mask to be used:
 *
 * <th>Mask character</th>
 * <th>Meaning in mask</th>
 * </tr>
 * <tr>
 * <td>&nbsp;l</td>
 * <td>&nbsp;Mixed case letter character [a..z, A..Z]</td>
 * </tr>
 * <tr>
 * <td>&nbsp;L</td>
 * <td>&nbsp;Upper case letter character [A..Z]</td>
 * </tr>
 * <tr>
 * <td>&nbsp;a</td>
 * <td>&nbsp;Mixed case alpha numeric character [a..z, A..Z, 0..1]</td>
 * </tr>
 * <tr>
 * <td>&nbsp;A</td>
 * <td>&nbsp;Upper case alpha numeric character [A..Z, 0..9]</td>
 * </tr>
 * <tr>
 * <td>&nbsp;#</td>
 * <td>&nbsp;Numeric character [0..9]</td>
 * </tr>
 * <tr>
 * <td>&nbsp;_</td>
 * <td>&nbsp;Reserved character for display, do not use.</td>
 * </tr>
 * <tr>
 * <td>&nbsp;others</td>
 * <td>&nbsp;Non editable character for display.</td>
 * </tr>
 * </table>
 */
dojo.require("dojo.validate");
dojo.require("dojo.string");

function MaskFormatter() {
    /* Create a flags object. */
    this.flags = new Object();
    this.flags.mask = "";               // This must be specified.

    /* Masks strings. */
    this.format = function(str) {
        var maskLen = this.flags.mask.length;
        if(maskLen === 0) {
            return "";
        }

        if(str === null) {
            str = "";
        }

        var i = 0;  // Mask index.
        var k = 0;  // Str index.
        var result = "";
        while(i < maskLen) {
            var mch = this.flags.mask.substr(i, 1);
            var ch;

            if(k < str.length) {
                if(isEditMaskChar(mch)) {
                    ch = str.substr(k, 1);
                    if(!isValidMaskChar(ch, mch)) {
                        ch = maskChar(mch);
                    }
                }
                else {
                    ch = maskChar(mch);
                }
                k++;
            }
            else {
                ch = maskChar(mch);
            }

            result = result + ch;
            i++;
        }
        return result;
    };

    /* Validates masks using the specified flags. */
    this.validate = function(str) {
        var maskLen = this.flags.mask.length;
        if(maskLen === 0) {
            return true;
        }

        if(str.length != this.flags.mask.length) {
            return false;
        }

        var result = true;
        for(var i = 0; i < str.length; i++) {
            var ch = str.substr(i, 1);
            var mch = this.flags.mask.substr(i, 1);
            var editChar = isEditMaskChar(mch);
            var validChar = isValidMaskChar(ch, mch);

            if((!editChar && ch != mch) ||
               (editChar && !validChar)) {
                result = false;
                break;
            }
        }
        return result;
    };

    /** Validates whether the code is allowed between the two strings. */
    this.isAllowed = function(code, previousStr, nextStr) {
        var maskLen = this.flags.mask.length;
        if(maskLen === 0) {
            return true;
        }

        if(isSpecialKey(code)) {
            return true;
        }

        // Assume the other chars are valid.
        if(previousStr.length >= this.flags.mask.length) {
            return false;
        }

        var mch = this.flags.mask.substr(previousStr.length, 1);
        var strKey = String.fromCharCode(code);

        return (isEditMaskChar(mch) && isValidMaskChar(strKey, mch)) ||
               (!isEditMaskChar(mch) && strKey == mch);
    };

    /**
     * Before entering a char, test if there are fixed chars that
     * could be automatically entered.
     */
    this.nextFixedChars = function(code, previousStr, nextStr) {
        var maskLen = this.flags.mask.length;
        if(maskLen === 0) {
            return "";
        }

        // Assume the other chars are valid.
        if(previousStr.length >= this.flags.mask.length) {
            return "";
        }

        var result = "";
        var isMaskChar;
        var mch;
        var i = previousStr.length;
        do {
            mch = this.flags.mask.substr(i, 1);
            isMaskChar = !isEditMaskChar(mch);
            if(isMaskChar) {
                result = result + mch;
            }
            i++;
        } while(i < this.flags.mask.length && isMaskChar);

        return result;
    }

    /*
     * Enable overwrite mode.
     */
    this.assembleValue = function(previousStr, ch, nextStr) {
        var str = previousStr + ch;
        var strLen = str.length;
        var nextLen = nextStr.length;
        var maskLen = this.flags.mask.length;

        var maskLen = this.flags.mask.length;
        if(maskLen === 0) {
            return str + nextStr;
        }

        // Delete chars from nextStr, from the beginning.
        if(strLen + nextLen > maskLen) {
            var dif = strLen + nextLen - maskLen;

            if(dif > nextLen) {
                nextStr = "";
            }
            else {
                nextStr = nextStr.substr(dif, nextLen - dif);
            }
        }

        return str + nextStr;
    }
}

/**
 * Returns true if the char is a meaningful edition char.
 * @param c the char.
 * @return True if c is a meaningful edit char.
 */
function isEditMaskChar(c) {
    switch (c) {
        case "_":
        case "#":
        case "a":
        case "A":
        case "l":
        case "L":
            return true;
        default:
            return false;
    }
    return false;
}

/**
 * Returns true if the char is valid for the specified mask char.
 * @param c the char.
 * @param mch the mask char.
 * @return True if c is a valid edit char.
 */
function isValidMaskChar(c, mch) {
    var result;
    switch(mch) {
        case "_":
            result = true;
            break;
        case "#":
            result = isDigit(c);
            break;
        case "a":
            result = isLowerAlphaNumeric(c);
            break;
        case "A":
            result = isUpperAlphaNumeric(c);
            break;
        case "l":
            result = isLowerAlpha(c);
            break;
        case "L":
            result = isUpperAlpha(c);
            break;
        default:
            result = false;
            break;
    }
    return result;
}

/**
 * Mask a specified char.
 */
function maskChar(c) {
    if (isEditMaskChar(c)) {
        return "_";
    }
    else {
        return c;
    }
}
