/**
 * Numbers formatter. You can specify custom decimal and thousands
 * separator, and negative number formatting. Defaults to American formatting.
 */
dojo.require("dojo.validate");
dojo.require("dojo.math");
dojo.require("dojo.string");
dojo.require("dojo.event.browser");

function NumberFormatter() {
    /* We use the dojo toolkit flags to specify formatting rules. */
    this.flags = new Object();
    this.flags.signed    = [true, false]; // Allow optional leading plus-or-minus sign.
    this.flags.separator = ",";           // Thousands separator.
    this.flags.places    = Infinity;      // Number of allowed decimal places.
    this.flags.decimal   = ".";           // Decimal separator.

    // Extend Dojo's validation to allow for optional thousands separator and rounding.
    this.flags.forceSeparator = false; // Don't enforce the thousands separator.
    this.flags.roundedPlaces  = 2;     // Number of decimal places after rounding.

    /* Formats numbers. */
    this.format = function(str) {
        str = stripLeadingZeros(str, this.flags.decimal);

        if(!this.validate(str)) {
            return str;
        }

        // Parse, rounding if necessary, and then format.
        var n = parseLocalizedFloat(str, this.flags);
        var result = formatLocalizedFloat(n, this.flags);

        return result;
    };

    /* Validates numbers using the specified flags. */
    this.validate = function(str) {
        var valid = dojo.validate.isRealNumber(str, this.flags);

        if(!valid && !this.flags.forceSeparator) {
            var oldSeparator = this.flags.separator;
            str.replace(new RegExp(dojo.string.escape("regexp", oldSeparator), "g"), "");

            this.flags.separator = "";
            try {
                valid = dojo.validate.isRealNumber(str, this.flags);
            }
            finally {
                this.flags.separator = oldSeparator;
            }
        }

        return valid;
    };

    this.isAllowed = function(code, previousStr, nextStr) {
        var result;
        var strKey = String.fromCharCode(code);

        if(isDigit(strKey)) {
            // If it's a digit, just prevent inclusion before a
            // sign, like "3-4345".
            result = (nextStr.search(/\+|\-/) == -1);
        }
        else if(strKey == "+" || strKey == "-") {
            // Signs can only be entered on the first char.
            result = (previousStr === "");
        }
        else if(strKey == this.flags.separator) {
            // Thousands separators can't go after the decimal point.
            // Also, they must be entered in intervals of 3.
            result =
              (previousStr.indexOf(this.flags.decimal) == -1);
        }
        else if(strKey == this.flags.decimal) {
            // Prevent double entry of decimals and other anomalies.
            result =
              (previousStr.indexOf(this.flags.decimal) == -1) &&
              (nextStr.indexOf(this.flags.decimal) == -1) &&
              (previousStr.search(/[0-9]/) != -1);
        }
        else if(isSpecialKey(code)) {
            // Special keys are allowed.
            result = true;
        }
        else {
            result = false;
        }
        return result;
    };

}
