/*
 * Decompiled with CFR 0.152.
 */
package org.apache.cassandra.db.lifecycle;

import java.io.File;
import java.io.FilenameFilter;
import java.nio.file.Path;
import java.nio.file.Paths;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.Set;
import java.util.regex.Matcher;
import java.util.regex.Pattern;
import java.util.stream.Collectors;
import java.util.zip.CRC32;
import org.apache.cassandra.io.sstable.Descriptor;
import org.apache.cassandra.io.sstable.SSTable;
import org.apache.cassandra.io.sstable.format.SSTableReader;
import org.apache.cassandra.io.util.FileUtils;
import org.apache.cassandra.utils.FBUtilities;

final class LogRecord {
    public final Type type;
    public final Optional<String> absolutePath;
    public final long updateTime;
    public final int numFiles;
    public final String raw;
    public final long checksum;
    public final Status status;
    static Pattern REGEX = Pattern.compile("^(add|remove|commit|abort):\\[([^,]*),?([^,]*),?([^,]*)\\]\\[(\\d*)\\]$", 2);

    public static LogRecord make(String line) {
        try {
            Matcher matcher = REGEX.matcher(line);
            if (!matcher.matches()) {
                return new LogRecord(Type.UNKNOWN, null, 0L, 0, 0L, line).setError(String.format("Failed to parse [%s]", line));
            }
            Type type = Type.fromPrefix(matcher.group(1));
            return new LogRecord(type, matcher.group(2) + '-', Long.parseLong(matcher.group(3)), Integer.parseInt(matcher.group(4)), Long.parseLong(matcher.group(5)), line);
        }
        catch (IllegalArgumentException e) {
            return new LogRecord(Type.UNKNOWN, null, 0L, 0, 0L, line).setError(String.format("Failed to parse line: %s", e.getMessage()));
        }
    }

    public static LogRecord makeCommit(long updateTime) {
        return new LogRecord(Type.COMMIT, updateTime);
    }

    public static LogRecord makeAbort(long updateTime) {
        return new LogRecord(Type.ABORT, updateTime);
    }

    public static LogRecord make(Type type, SSTable table) {
        String absoluteTablePath = LogRecord.absolutePath(table.descriptor.baseFilename());
        return LogRecord.make(type, LogRecord.getExistingFiles(absoluteTablePath), table.getAllFilePaths().size(), absoluteTablePath);
    }

    public static Collection<LogRecord> make(Type type, Iterable<SSTableReader> tables) {
        HashMap<String, SSTableReader> absolutePaths = new HashMap<String, SSTableReader>();
        for (SSTableReader table : tables) {
            absolutePaths.put(LogRecord.absolutePath(table.descriptor.baseFilename()), table);
        }
        Map<String, List<File>> existingFiles = LogRecord.getExistingFiles(absolutePaths.keySet());
        ArrayList<LogRecord> records = new ArrayList<LogRecord>(existingFiles.size());
        for (Map.Entry<String, List<File>> entry : existingFiles.entrySet()) {
            List<File> filesOnDisk = entry.getValue();
            String baseFileName = entry.getKey();
            SSTable sstable = (SSTable)absolutePaths.get(baseFileName);
            records.add(LogRecord.make(type, filesOnDisk, sstable.getAllFilePaths().size(), baseFileName));
        }
        return records;
    }

    private static String absolutePath(String baseFilename) {
        return FileUtils.getCanonicalPath(baseFilename + '-');
    }

    public LogRecord withExistingFiles() {
        return LogRecord.make(this.type, this.getExistingFiles(), 0, this.absolutePath.get());
    }

    public static LogRecord make(Type type, List<File> files, int minFiles, String absolutePath) {
        List positiveModifiedTimes = files.stream().map(File::lastModified).filter(lm -> lm > 0L).collect(Collectors.toList());
        long lastModified = positiveModifiedTimes.stream().reduce(0L, Long::max);
        return new LogRecord(type, absolutePath, lastModified, Math.max(minFiles, positiveModifiedTimes.size()));
    }

    private LogRecord(Type type, long updateTime) {
        this(type, null, updateTime, 0, 0L, null);
    }

    private LogRecord(Type type, String absolutePath, long updateTime, int numFiles) {
        this(type, absolutePath, updateTime, numFiles, 0L, null);
    }

    private LogRecord(Type type, String absolutePath, long updateTime, int numFiles, long checksum, String raw) {
        assert (!type.hasFile() || absolutePath != null) : "Expected file path for file records";
        this.type = type;
        this.absolutePath = type.hasFile() ? Optional.of(absolutePath) : Optional.empty();
        this.updateTime = type == Type.REMOVE ? updateTime : 0L;
        this.numFiles = type.hasFile() ? numFiles : 0;
        this.status = new Status();
        if (raw == null) {
            assert (checksum == 0L);
            this.checksum = this.computeChecksum();
            this.raw = this.format();
        } else {
            this.checksum = checksum;
            this.raw = raw;
        }
    }

    LogRecord setError(String error) {
        this.status.setError(error);
        return this;
    }

    String error() {
        return this.status.error.orElse("");
    }

    void setPartial() {
        this.status.partial = true;
    }

    boolean partial() {
        return this.status.partial;
    }

    boolean isValid() {
        return !this.status.hasError() && this.type != Type.UNKNOWN;
    }

    boolean isInvalid() {
        return !this.isValid();
    }

    boolean isInvalidOrPartial() {
        return this.isInvalid() || this.partial();
    }

    private String format() {
        return String.format("%s:[%s,%d,%d][%d]", this.type.toString(), this.absolutePath(), this.updateTime, this.numFiles, this.checksum);
    }

    public List<File> getExistingFiles() {
        assert (this.absolutePath.isPresent()) : "Expected a path in order to get existing files";
        return LogRecord.getExistingFiles(this.absolutePath.get());
    }

    public static List<File> getExistingFiles(String absoluteFilePath) {
        Path path = Paths.get(absoluteFilePath, new String[0]);
        File[] files = path.getParent().toFile().listFiles((dir, name) -> name.startsWith(path.getFileName().toString()));
        return files == null ? Collections.emptyList() : Arrays.asList(files);
    }

    public static Map<String, List<File>> getExistingFiles(Set<String> absoluteFilePaths) {
        Set uniqueDirectories = absoluteFilePaths.stream().map(path -> Paths.get(path, new String[0]).getParent().toFile()).collect(Collectors.toSet());
        HashMap<String, List<File>> fileMap = new HashMap<String, List<File>>();
        FilenameFilter ff = (dir, name) -> {
            String absolutePath;
            Descriptor descriptor = null;
            try {
                descriptor = (Descriptor)Descriptor.fromFilename((File)dir, (String)name).left;
            }
            catch (Throwable throwable) {
                // empty catch block
            }
            String string = absolutePath = descriptor != null ? LogRecord.absolutePath(descriptor.baseFilename()) : null;
            if (absolutePath != null && absoluteFilePaths.contains(absolutePath)) {
                fileMap.computeIfAbsent(absolutePath, k -> new ArrayList()).add(new File(dir, name));
            }
            return false;
        };
        for (File f : uniqueDirectories) {
            f.listFiles(ff);
        }
        return fileMap;
    }

    public boolean isFinal() {
        return this.type.isFinal();
    }

    String fileName() {
        return this.absolutePath.isPresent() ? Paths.get(this.absolutePath.get(), new String[0]).getFileName().toString() : "";
    }

    boolean isInFolder(Path folder) {
        return this.absolutePath.isPresent() ? FileUtils.isContained(folder.toFile(), Paths.get(this.absolutePath.get(), new String[0]).toFile()) : false;
    }

    private String absolutePath() {
        if (!this.absolutePath.isPresent()) {
            return "";
        }
        String ret = this.absolutePath.get();
        assert (ret.charAt(ret.length() - 1) == '-') : "Invalid absolute path, should end with '-'";
        return ret.substring(0, ret.length() - 1);
    }

    public int hashCode() {
        return Objects.hash(new Object[]{this.type, this.absolutePath, this.numFiles, this.updateTime});
    }

    public boolean equals(Object obj) {
        if (!(obj instanceof LogRecord)) {
            return false;
        }
        LogRecord other = (LogRecord)obj;
        return this.type == other.type && this.absolutePath.equals(other.absolutePath) && this.numFiles == other.numFiles && this.updateTime == other.updateTime;
    }

    public String toString() {
        return this.raw;
    }

    long computeChecksum() {
        CRC32 crc32 = new CRC32();
        crc32.update(this.absolutePath().getBytes(FileUtils.CHARSET));
        crc32.update(this.type.toString().getBytes(FileUtils.CHARSET));
        FBUtilities.updateChecksumInt(crc32, (int)this.updateTime);
        FBUtilities.updateChecksumInt(crc32, (int)(this.updateTime >>> 32));
        FBUtilities.updateChecksumInt(crc32, this.numFiles);
        return crc32.getValue() & Long.MAX_VALUE;
    }

    public static final class Status {
        Optional<String> error = Optional.empty();
        boolean partial = false;
        LogRecord onDiskRecord;

        void setError(String error) {
            if (!this.error.isPresent()) {
                this.error = Optional.of(error);
            }
        }

        boolean hasError() {
            return this.error.isPresent();
        }
    }

    public static enum Type {
        UNKNOWN,
        ADD,
        REMOVE,
        COMMIT,
        ABORT;


        public static Type fromPrefix(String prefix) {
            return Type.valueOf(prefix.toUpperCase());
        }

        public boolean hasFile() {
            return this == ADD || this == REMOVE;
        }

        public boolean matches(LogRecord record) {
            return this == record.type;
        }

        public boolean isFinal() {
            return this == COMMIT || this == ABORT;
        }
    }
}

