package org.geotools.renderer.style.customshape;

import java.awt.Shape;
import java.awt.geom.Rectangle2D;

import org.apache.log4j.Logger;

/**
 * ShapeCreator for building rectangular shapes. Its shape specification string
 * follows the form 'rectangle?aspect=2.0', where 'aspect' stands for the
 * relation width/height. As such, a shape of size 10 and aspect 2 will have
 * height=10 and width=20.
 * 
 * @author milton
 */
public class RectangleShapeCreator extends AbstractShapeCreator
{
    private static final String SHAPE_NAME = "rectangle";
    private static final String ASPECT_PARAMETER_NAME = "aspect";
    private static Logger _logger = Logger.getLogger(RectangleShapeCreator.class);

    @Override
    protected Shape createShape(Parameter<String>... parameters)
    {
        String aspectStr = null;

        for (Parameter<String> parameter : parameters)
        {
            if (parameter.getName().equalsIgnoreCase(ASPECT_PARAMETER_NAME))
                aspectStr = parameter.getValue();
        }

        if (aspectStr == null)
        {
            _logger.warn("Missing required param 'aspect'");
            return null;
        }

        double aspect = 0.0;
        try
        {
            aspect = Double.parseDouble(aspectStr);
        }
        catch (NumberFormatException e)
        {
            _logger.warn("Parameter 'aspect' in invalid format", e);
            return null;
        }

        return createShape(aspect);

    }

    /**
     * Creates a rectangle shape based on the given aspect.
     * 
     * @param aspect The aspect ratio between the width and height of the
     *            rectangle (width/height).
     * @return The basic {@link Shape} for drawing a rectangle with the given
     *         aspect ratio.
     */
    public Shape createShape(double aspect)
    {
        aspect = Math.abs(aspect);
        return new Rectangle2D.Double(-aspect/2, -0.5, aspect, 1);
    }

    @Override
    public String getShapeName()
    {
        return SHAPE_NAME;
    }

    /**
     * This method is a utility to help creating a shape specification string
     * for style description purposes.
     * 
     * @param aspect The aspect ratio between the height and width of the
     *            rectangle.
     * @return The String that represents this shape specification.
     */
    @SuppressWarnings("unchecked")
    public String buildShapeSpecification(double aspect)
    {
        Parameter<String> aspectParameter = new Parameter<String>(ASPECT_PARAMETER_NAME, String.valueOf(aspect));
        return buildShapeSpecification(aspectParameter);
    }

}
