/**
 * Tecgraf - GIS development team
 * 
 * Tdk Framework
 * Copyright TecGraf 2009(c).
 * 
 * file: TypeAttributeTranslator.java
 * created: 14/01/2009
 */
package org.geotools.data.terralib.util;

import java.util.HashSet;
import java.util.Set;

import org.apache.log4j.Logger;
import org.geotools.data.terralib.TerralibService;
import org.geotools.data.terralib.exception.NullArgumentException;
import org.geotools.data.terralib.geometry.MultiTextGeometry;
import org.geotools.data.terralib.persistence.exception.InvalidAttributeException;
import org.geotools.data.terralib.swig.TeGeomRep;
import org.geotools.data.terralib.swig.TerralibAttributeDescriptor;
import org.geotools.data.terralib.swig.TerralibServiceNative;
import org.geotools.feature.AttributeTypeBuilder;
import org.geotools.feature.FeatureTypes;
import org.opengis.feature.type.AttributeDescriptor;
import org.opengis.feature.type.AttributeType;
import org.opengis.feature.type.GeometryDescriptor;
import org.opengis.feature.type.GeometryType;
import org.opengis.referencing.crs.CoordinateReferenceSystem;

import com.vividsolutions.jts.geom.MultiLineString;
import com.vividsolutions.jts.geom.MultiPoint;
import com.vividsolutions.jts.geom.MultiPolygon;

/**
 * This class provides static methods to translate TerralibAttributeDescriptor
 * to Geootool's AttributeDescriptor, and vice-versa.
 * @author fmoura
 * @since TDK3.0.0
 */
public class TerralibAttributeDescriptorTranslator
{
    private static final String DEFAULT_GEOMETRY_POLYGON_NAME = "default_geometry_polygon";
    private static final String DEFAULT_GEOMETRY_LINE_NAME = "default_geometry_line";
    private static final String DEFAULT_GEOMETRY_POINT_NAME = "default_geometry_point";
    private static final String DEFAULT_GEOMETRY_TEXT_NAME = "default_geometry_text";
    private static final String POLYGON_TYPE_NAME = "polygon";
    private static final String LINE_TYPE_NAME = "line";
    private static final String POINT_TYPE_NAME = "point";
    private static final String TEXT_TYPE_NAME = "text";
    
    private static final Logger _logger = Logger.getLogger(TerralibAttributeDescriptorTranslator.class);
    
    /**
     * Translate a TerralibAttributeDescriptor to an Geotools AttributeDescriptor.
     * @param attributeTypeBuilder
     * @param attribute
     * @return The TerralibAttributeDescriptor created.
     * @since TDK3.0.0
     */
    public static AttributeDescriptor terralibDescriptorToGeotoolsDescriptor (
            AttributeTypeBuilder attributeTypeBuilder, TerralibAttributeDescriptor attribute)
    {
    	if (attributeTypeBuilder == null)
    		throw new NullArgumentException("attributeTypeBuilder");
    	if (attribute == null)
    		throw new NullArgumentException("attribute");
    	
        String attributeName = attribute.getName();
        
        TypeAttributeMap typeMap = TypeAttributeMap.fromTeAttrDataType(attribute.getDataType());
        if (typeMap == null)
        	throw new InvalidAttributeException("The attribute binding class for \"" + attribute.getName() + "\" is not implemented at the Terralib Provider");

        int length;
        
        if (attribute.getLength() == FeatureTypes.ANY_LENGTH)
        	length = TerralibServiceNative.getANY_LENGHT();
        else
        	length = attribute.getLength();
        
        AttributeType attributeType = 
        	terralibToGeotoolsAttributeType(attributeTypeBuilder,typeMap.getBindingClass()
        			, attribute.isPrimaryKey(),attribute.isNullable(),attributeName,length);
        
        int pointIndex = attributeName.indexOf('.'); 
        if(pointIndex != -1)
        {
            attributeName = attributeName.substring(pointIndex+1);
        }
        
        AttributeDescriptor attributeDescriptor = attributeTypeBuilder.buildDescriptor(attributeName, attributeType);
        return attributeDescriptor;
    }
    
    /**
     * Translate an Geotools AttributeDescriptor to a TerralibAttributeDescriptor.
     * @param service Service connection to Terralib Provider
     * @param gtDescriptor Geotools AttributeDescriptor to translate
     * @return The TerralibAttributeDescriptor created, similar to the AttributeDescriptor
     * @since TDK3.0.0
     */
    public static TerralibAttributeDescriptor geotoolsDescriptorToTerralibDescriptor(TerralibService service, AttributeDescriptor gtDescriptor)
    {
    	if (service == null)
    		throw new NullArgumentException("service");
    	if (gtDescriptor == null)
    		throw new NullArgumentException("gtDescriptor");
    	
    	TypeAttributeMap typeMap = TypeAttributeMap.fromAttributeType(gtDescriptor.getType());

    	if (typeMap == null)
			throw new InvalidAttributeException("The attribute binding class for \"" + gtDescriptor.getLocalName() + "\" is not implemented at the Terralib Provider");				
    	
    	boolean primaryKey = gtDescriptor.getType().isIdentified();
    	
    	if (gtDescriptor.isNillable() && primaryKey)
    		throw new InvalidAttributeException("The attribute \"" + gtDescriptor.getLocalName() + "\" is a primary key, it can't accept null");
    	
    	int length;
    	if (typeMap.isGeometry())
    		length = TerralibServiceNative.getANY_LENGHT();
    	else if (FeatureTypes.getFieldLength(gtDescriptor) == FeatureTypes.ANY_LENGTH)
    		length = TerralibServiceNative.getANY_LENGHT();
    	else
    		length = FeatureTypes.getFieldLength(gtDescriptor);
    	
    	return
    		service.buildTerralibAttributeDescriptor(gtDescriptor.getName().getLocalPart(), gtDescriptor.isNillable(), primaryKey, length, typeMap.getTeAttributeType());
    	
    }
    
//    private static AttributeType translateToAttributeType(AttributeTypeBuilder attributeTypeBuilder, TeAttrDataType teAttributeType)
//    {
//    
//        Class<?> bindingClass;
//        switch (teAttributeType)
//        {
//        case TeREAL:
//            bindingClass = Double.class;
//            break;
//            
//        case TeINT:
//            bindingClass = Integer.class;
//            break;
//            
//        case TeSTRING:
//            bindingClass = String.class;
//            break;
//            
//        case TeDATETIME:
//            bindingClass = Date.class;
//            break;
//
//        default:
//            bindingClass = Object.class;
//            _logger.warn("Unsupported data type ("+teAttributeType.name()+"). Using Object class for binding.");
//            break;
//        }
//        attributeTypeBuilder.setBinding(bindingClass);
//        AttributeType attributeType = attributeTypeBuilder.buildType();
//        return attributeType;
//    
//    }
    
    public static Set<GeometryDescriptor> buildGeometryDescriptorSet( Set<TeGeomRep> geomTypeSet, CoordinateReferenceSystem crs)
    {
        Set<GeometryDescriptor> geometryDescriptorSet = new HashSet<GeometryDescriptor>();
        AttributeTypeBuilder attributeTypeBuilder = new AttributeTypeBuilder();
        
        GeometryType geometryType;
        GeometryDescriptor geometryDescriptor;
        
        for(TeGeomRep geomRep : geomTypeSet)
        {
            switch(geomRep)
            {
                case TePOINTS:
                    
                    attributeTypeBuilder.setBinding(MultiPoint.class);
                    attributeTypeBuilder.setName(POINT_TYPE_NAME);
                    attributeTypeBuilder.setCRS(crs);
                    
                    geometryType = attributeTypeBuilder.buildGeometryType();
                    geometryDescriptor = attributeTypeBuilder.buildDescriptor(DEFAULT_GEOMETRY_POINT_NAME, geometryType);
                    
                    geometryDescriptorSet.add(geometryDescriptor);
                    break;
                case TeLINES:
                    attributeTypeBuilder.setBinding(MultiLineString.class);
                    attributeTypeBuilder.setName(LINE_TYPE_NAME);
                    attributeTypeBuilder.setCRS(crs);
                    
                    geometryType = attributeTypeBuilder.buildGeometryType();
                    geometryDescriptor = attributeTypeBuilder.buildDescriptor(DEFAULT_GEOMETRY_LINE_NAME, geometryType);
                    
                    geometryDescriptorSet.add(geometryDescriptor);
                    break;
                    
                case TePOLYGONS:
                    attributeTypeBuilder.setBinding(MultiPolygon.class);
                    attributeTypeBuilder.setName(POLYGON_TYPE_NAME);
                    attributeTypeBuilder.setCRS(crs);
                    
                    geometryType = attributeTypeBuilder.buildGeometryType();
                    geometryDescriptor = attributeTypeBuilder.buildDescriptor(DEFAULT_GEOMETRY_POLYGON_NAME, geometryType);
                    
                    geometryDescriptorSet.add(geometryDescriptor);
                    break;
                    
                case TeTEXT:
                	attributeTypeBuilder.setBinding(MultiTextGeometry.class);
                	attributeTypeBuilder.setName(TEXT_TYPE_NAME);
                	attributeTypeBuilder.setCRS(crs);
                	
                	geometryType = attributeTypeBuilder.buildGeometryType();
                	geometryDescriptor = attributeTypeBuilder.buildDescriptor(DEFAULT_GEOMETRY_TEXT_NAME, geometryType);
                    
                    geometryDescriptorSet.add(geometryDescriptor);
                    break;
                    
                default:
                    _logger.warn(geomRep.name()+" geometry representation is not supported.");
                    break;
            }
        }
        
        return geometryDescriptorSet;
    }
    
	protected static AttributeType terralibToGeotoolsAttributeType(AttributeTypeBuilder attributeTypeBuilder
			, Class<?> bindingClass, boolean isIdentifiable, boolean isNillable
			, String name, int length)
	{
		if (attributeTypeBuilder == null)
			throw new NullArgumentException("attributeTypeBuilder");
		
        attributeTypeBuilder.setBinding(bindingClass);
        attributeTypeBuilder.setIdentifiable(isIdentifiable);
        attributeTypeBuilder.setNillable(isNillable);
        attributeTypeBuilder.setName(name);
        attributeTypeBuilder.setLength(length);
        
        return attributeTypeBuilder.buildType();
	}		    
}
