/**
 * Tecgraf - GIS development team
 * 
 * Tdk Framework
 * Copyright TecGraf 2008(c).
 * 
 * file: TerralibDataStore.java
 * created: 17/12/2008
 */
package org.geotools.data.terralib;

import static org.geotools.data.terralib.connection.DBConnectionParamUtils.DBDRIVER;
import static org.geotools.data.terralib.connection.DBConnectionParamUtils.DBNAME;
import static org.geotools.data.terralib.connection.DBConnectionParamUtils.DBPATH;
import static org.geotools.data.terralib.connection.DBConnectionParamUtils.HOST;
import static org.geotools.data.terralib.connection.DBConnectionParamUtils.PASSWD;
import static org.geotools.data.terralib.connection.DBConnectionParamUtils.PORT;
import static org.geotools.data.terralib.connection.DBConnectionParamUtils.USER;

import java.io.IOException;
import java.io.Serializable;
import java.net.URL;
import java.sql.Connection;
import java.sql.SQLException;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.HashSet;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.Map.Entry;

import org.apache.log4j.Logger;
import org.geotools.data.AbstractDataStore;
import org.geotools.data.DataStore;
import org.geotools.data.DefaultQuery;
import org.geotools.data.FeatureReader;
import org.geotools.data.FeatureSource;
import org.geotools.data.FeatureWriter;
import org.geotools.data.Query;
import org.geotools.data.Transaction;
import org.geotools.data.terralib.connection.DBConnectionFactory;
import org.geotools.data.terralib.connection.DBConnectionParamUtils;
import org.geotools.data.terralib.exception.AttributeNameColisionException;
import org.geotools.data.terralib.exception.IllegalStateException;
import org.geotools.data.terralib.exception.InvalidCrsWktException;
import org.geotools.data.terralib.exception.NullArgumentException;
import org.geotools.data.terralib.exception.TypeNotFoundException;
import org.geotools.data.terralib.feature.TerralibFeatureReader;
import org.geotools.data.terralib.feature.TerralibFeatureReaderWriter;
import org.geotools.data.terralib.feature.TerralibFeatureStore;
import org.geotools.data.terralib.feature.attribute.DefaultTerralibAttributePersistenceHandler;
import org.geotools.data.terralib.feature.attribute.TerralibAttributePersistenceHandler;
import org.geotools.data.terralib.feature.attribute.TerralibAttributePersistenceHandlerTypeOperation;
import org.geotools.data.terralib.geometry.MultiTextGeometry;
import org.geotools.data.terralib.query.JavaQuerier;
import org.geotools.data.terralib.query.QuerierParams;
import org.geotools.data.terralib.query.QueryData;
import org.geotools.data.terralib.query.check.AccessQueryChecker;
import org.geotools.data.terralib.query.check.QueryChecker;
import org.geotools.data.terralib.query.check.SqlServerQueryChecker;
import org.geotools.data.terralib.swig.DBConnection;
import org.geotools.data.terralib.swig.ViewElement;
import org.geotools.data.terralib.util.TerralibAttributeHelper;
import org.geotools.data.terralib.util.TypeAttributeMap;
import org.geotools.factory.Hints;
import org.geotools.factory.Hints.Key;
import org.geotools.feature.FeatureCollection;
import org.geotools.feature.NameImpl;
import org.geotools.feature.SchemaException;
import org.geotools.feature.type.BasicFeatureTypes;
import org.geotools.geometry.jts.ReferencedEnvelope;
import org.opengis.feature.simple.SimpleFeature;
import org.opengis.feature.simple.SimpleFeatureType;
import org.opengis.feature.type.AttributeDescriptor;
import org.opengis.feature.type.FeatureType;
import org.opengis.feature.type.GeometryDescriptor;
import org.opengis.feature.type.Name;
import org.opengis.filter.Filter;
import org.opengis.geometry.BoundingBox;
import org.opengis.geometry.MismatchedDimensionException;
import org.opengis.referencing.crs.CoordinateReferenceSystem;

import com.vividsolutions.jts.geom.Envelope;
import com.vividsolutions.jts.geom.MultiLineString;
import com.vividsolutions.jts.geom.MultiPoint;
import com.vividsolutions.jts.geom.MultiPolygon;

/**
 * DataStore implementation for the TerraLib (the Brazilian space agency's geospatial library). 
 * The primary goal of this DataStore is to provide means to access <b>vector</b> data stored in 
 * TerraLib databases.
 * @author alexmc, fmoura, fabiomano, milton
 * @since TDK 3.0.0
 */
public class TerralibDataStore extends AbstractDataStore implements DataStore
{
	//list of terralib reserved names (can't be used for attribute names)
	private static final String[] RESERVED_NAMES = {"object_id"};
	
	//list of terralib illegal words (can't be part of attribute names) 
	private static final String[] ILLEGAL_WORDS = {"DROP TABLE", "CREATE TABLE", "DELETE FROM", "INSERT INTO","UPDATE ","!",".","[","]","`"};
	
    //private Map<String, ? extends Serializable> _connectionParams;
    private DefaultTerralibService _terralibService;
//    private Querier _querier;
    private HashSet<Key> _supportedHints = new HashSet<Key>();
    private TerralibAttributePersistenceHandler _persistenceHandler = new DefaultTerralibAttributePersistenceHandler();

    private TerralibMetadata _metadata;
    
    private Connection _con = null;
    private JavaQuerier _jQuerier;  //java querier

    private static Logger _logger = Logger.getLogger(TerralibDataStore.class);
    
    /**
     * Creates a new TerralibDataStore.
     * @param params Map of TerraLib connection parameters. 
     */
    public TerralibDataStore(DBConnectionFactory connectionFactory, Map<String, ? extends Serializable> connectionParams )
    {
        if (connectionFactory == null)
        {
            throw new IllegalArgumentException("The connection factory reference can't be null.");
        }
        
        if (connectionParams == null)
        {
            throw new IllegalArgumentException("The TerraLib connection parameters map can't be null.");
        }

        // throws if the parameter map doesn't fulfill the requirements.
        DBConnectionParamUtils.checkParams(connectionParams);
        
        String dataStoreType = (String) connectionParams.get(DBDRIVER.key);
        
        DBConnection connection = null;
        boolean enableLocking = false;

        QueryChecker checker;
        
        if (DBConnectionParamUtils.DATABASE_DRIVER_ACCESS.equalsIgnoreCase(dataStoreType))
        {
            String path = (String) connectionParams.get(DBPATH.key);
            connection = connectionFactory.createAccessConnection(path);
            _con = connectionFactory.createJDBCAccessConnection(path);
            checker = new AccessQueryChecker();
            enableLocking = true;
        }
        else if (DBConnectionParamUtils.DATABASE_DRIVER_SQL_SERVER.equalsIgnoreCase(dataStoreType))
        {
            String host = (String) connectionParams.get(HOST.key);
            String userName = (String) connectionParams.get(USER.key);
            String databaseName = (String) connectionParams.get(DBNAME.key);
            String password = (String) connectionParams.get(PASSWD.key);
            // DBConnectionParamUtils.checkParams() already guarantees that parameters are not null
            int portNumber = Integer.parseInt(connectionParams.get(PORT.key).toString());
            
            _logger.debug("Connecting to the Sql-Server database with host=" + host + ", databaseName=" + databaseName);
            connection = connectionFactory.createSQLServerConnection(host, userName, password, databaseName, portNumber);
            _logger.debug("Sql-Server database connection estabilished.");
            
            enableLocking = true;
            _con = connectionFactory.createJDBCSqlServerConnection(host, userName, password, databaseName, portNumber);
            checker = new SqlServerQueryChecker();
        }
        else
        {
            throw new UnsupportedOperationException("Sorry, but this database type isn't supported yet.");
        }
        
        _terralibService = new DefaultTerralibService(connection,enableLocking);
//        _querier = new Querier(_terralibService);
        
        _metadata = new TerralibMetadata(this,_con);
        _jQuerier = new JavaQuerier(_terralibService,_con,_metadata,checker);
        
        _supportedHints.add(Hints.FEATURE_DETACHED);
    }
    
    protected  FeatureReader<SimpleFeatureType, SimpleFeature> getFeatureReader(String typeName, Query query)
    throws IOException
    {
        TerralibFeatureType schema = (TerralibFeatureType) getSchema(typeName);
        FeatureTypeInfo featureTypeInfo = new FeatureTypeInfo(schema);
        QuerierParams params = new QuerierParams(featureTypeInfo,query);
        QueryData queryData;
        try
        {
//            queryData = _querier.execute(params);
            queryData = _jQuerier.execute(params);
        }
        catch (TypeNotFoundException e)
        {
            throw new IOException("Error getting attribute reader for type '"+typeName+"'",e);
        }
        try
        {
            return new TerralibFeatureReader(queryData, schema);
        }
        catch (SchemaException e)
        {
            throw new IOException("Error creating feature reader for type '"+typeName+"'",e);
        }        
    }    
    
    /* (non-Javadoc)
     * @see org.geotools.data.AbstractDataStore#getFeatureReader(java.lang.String)
     */
    @Override
    protected FeatureReader<SimpleFeatureType, SimpleFeature> getFeatureReader(String typeName) throws IOException
    {
        if (typeName == null)
            throw new NullArgumentException(typeName);
        
        //this should never be called, we are overriding getFeatureReader from AbstractDataStore to use the filter.
        return getFeatureReader(new DefaultQuery(typeName,Filter.INCLUDE), Transaction.AUTO_COMMIT);
//        throw new UnsupportedOperationException();
    }
    
    /* (non-Javadoc)
     * @see org.geotools.data.AbstractDataStore#createFeatureWriter(java.lang.String, org.geotools.data.Transaction)
     */
    @Override
    protected FeatureWriter<SimpleFeatureType, SimpleFeature> createFeatureWriter(String typeName,
            Transaction transaction) throws IOException
    {
        TerralibFeatureType schema = (TerralibFeatureType) getSchema(typeName);
        FeatureTypeInfo featureTypeInfo = new FeatureTypeInfo(schema);
        QuerierParams params = new QuerierParams(featureTypeInfo);   //we are not supporting filters at the writer now, only at the reader
        QueryData queryData;
        try
        {
//            queryData = _querier.execute(params);
            queryData = _jQuerier.execute(params);
        }
        catch (TypeNotFoundException e)
        {
            throw new IOException("Error getting attribute reader for type '"+typeName+"'",e);
        }
        try
        {
            return new TerralibFeatureReaderWriter(listenerManager, _terralibService, queryData,schema,Transaction.AUTO_COMMIT,_persistenceHandler);
        }
        catch (SchemaException e)
        {
            throw new IOException("Error creating feature reader for type '"+typeName+"'",e);
            
        }        
    }
   

    /**
     * Retrieves the {@link SimpleFeatureType} for this DataStore. When mapping
     * the <a href="http://www.terralib.org/">TerraLib</a> layers' and themes' 
     * attributes and geometries, we map all the layer's or theme's attributes 
     * from all it's attribute tables, but <b>ignore the geometry attributes</b> 
     * (note that they're not the same as the layer or theme geometries). Since 
     * TerraLib layers and themes can contain than one geometry type, we use the 
     * following order of available geometries (uses the first type found from 
     * this list) to resolve the default geometry:
     * <ol>
     *    <li>Polygons</li>
     *    <li>Lines</li>
     *    <li>Points</li>
     * </ol>
     * @see org.geotools.data.AbstractDataStore#getSchema(java.lang.String)
     * @return The {@link SimpleFeatureType} (schema) for the given type name. 
     */
    @Override
    public SimpleFeatureType getSchema(String typeName) throws IOException
    {
        
        Map<String, List<AttributeDescriptor>> schema = null;
        Set<GeometryDescriptor> geometryDescriptorSet = null;
           
        try
        {
            schema = _terralibService.getTypeAttributeDescriptors(typeName);
            geometryDescriptorSet = _terralibService.getTypeGeometryDescriptors(typeName);
        }
        catch (IllegalStateException e)
        {
           throw new IOException("The database is in an illegal state",e);
        }
        catch (TypeNotFoundException e)
        { 
            throw new IOException("Could not find type \""+typeName+"\"",e);
        }
//        catch (DataLoadFailedException e)
//        {
//            throw new IOException("Failed to load data for the typename \"" + typeName + "\".", e);
//        }
        
        GeometryDescriptor defaultGeometry = null;
        for(GeometryDescriptor geometryDescriptor : geometryDescriptorSet)
        {
            if(geometryDescriptor.getType().getBinding().equals(MultiPolygon.class))
            {
                defaultGeometry = geometryDescriptor;
                break;
            }
            if(geometryDescriptor.getType().getBinding().equals(MultiLineString.class))
            {
                defaultGeometry = geometryDescriptor;
                break;
            }
            if(geometryDescriptor.getType().getBinding().equals(MultiPoint.class))
            {
                defaultGeometry = geometryDescriptor;
                break;
            }
            if(geometryDescriptor.getType().getBinding().equals(MultiTextGeometry.class))
            {
            	defaultGeometry = geometryDescriptor;
                break;
            }
        }
        
        final String defaultGeometryTable = DefaultTerralibService.getDEFAULT_GEOMETRY_TABLE_NAME();
        
        if (schema.get(defaultGeometryTable) == null)
        {
            schema.put(defaultGeometryTable, new ArrayList<AttributeDescriptor>()); 
        }
        
        schema.get(defaultGeometryTable).add(defaultGeometry);
        
        Name featureTypeName = new NameImpl(typeName);
        TerralibFeatureType terralibFeatureType = null;
        try
        {
            terralibFeatureType = new TerralibFeatureType(featureTypeName, schema, defaultGeometry, false, null, BasicFeatureTypes.FEATURE, null);
        }
        catch (AttributeNameColisionException e)
        {
            throw new IOException("Error creating the TerralibFeatureType '"+featureTypeName+"'.",e);
            
        }
        return terralibFeatureType ;
    }

    /**
     * Fetches the available view IDs from the DataStore. The TerralibDataStore
     * uses view to group themes into a single view (for instance, to have a view
     * that shows only a region of a country). This method get all the view ids 
     * registered at the TerralibDataStore.
     * Note: This method only returns the view ids that are used in at least one
     * one theme. Existing views that are not associated with any themes won't be
     * at the returned array.
     * @return List of view IDs at the TerraLib DataStore.
     * @throws IOException if data cannot be retrieved.
     */
    public String[] getViewIDs() throws IOException
    {
    	return _terralibService.getViewIDs();
    }

    /**
     * Gets the view name for a given id
     * @param viewID The id to retrieve the view name
     * @return The view name
     * @throws IOException if data cannot be retrieved.
     */
    public String getViewName(String viewID) throws IOException
    {
        try
        {
            return _terralibService.getViewName(viewID);
        }
        catch (TypeNotFoundException e)
        {
            throw new IOException("Could not find the requested view.",e);
        }
    }
    
    /**
     * Fetches the available theme names from a given view at the DataStore. 
     * @return List of theme names for the given viewID
     * @throws IOException if data cannot be retrieved.
     */    
    public String[] getViewTypeNames(String viewID) throws IOException
    {
    	if (viewID == null)
    		throw new NullArgumentException("viewID");
    	return _terralibService.getViewTypesNames(viewID);
    }  
    
    /**
     * 
     * @param viewID
     * @return
     * @throws IOException
     */
    public ViewElement[] getViewStructure(String viewID) throws IOException
    {
    	if (viewID == null)
    		throw new NullArgumentException("viewID");
    	return _terralibService.getViewStructure(viewID);
    }
    
    /**
     * Fetches the CoordinateReferenceSystem for the given view ID at tge DataStore.
     * @param viewID The ID of the view to get the CRS.
     * @return the view CRS.
     * @throws IOException if data cannot be retrieved.
     */
    public CoordinateReferenceSystem getViewCRS(String viewID) throws IOException
    {
    	try{
    		return _terralibService.getViewCRS(viewID);
    	}
    	catch(TypeNotFoundException e)
    	{
    		throw new IOException("Coudl not find the requested view.",e);
    	}
    	
    }
    
    /**
     * Fetches the available type names from the DataStore. The TerralibDataStore
     * uses <a href="http://www.terralib.org/">TerraLib</a> layers and themes as 
     * it's types, but the type names are not the same as the layers' and themes' 
     * names. This is needed since their (layer and theme) names aren't guaranteed to 
     * be unique. The returned type names are derived identifiers for these types 
     * and are valid only for this DataStore implementation.
     * @see org.geotools.data.AbstractDataStore#getTypeNames()
     * @return List of type names available for the TerraLib {@link DataStore}. 
     */
    @Override
    public String[] getTypeNames() throws IOException
    {
    	try {
			return _terralibService.getTypeNames();
		} catch (IllegalStateException e) {
			throw new IOException("Terralib database is in a invalid state",e);
			
		}
    }
    
    /* (non-Javadoc)
     * @see org.geotools.data.AbstractDataStore#getBounds(org.geotools.data.Query)
     */
    @Override
    protected ReferencedEnvelope getBounds(Query query) throws IOException
    {
    	if (query.getFilter() == Filter.EXCLUDE)
    	{
    		return new ReferencedEnvelope(getSchema(query.getTypeName()).getCoordinateReferenceSystem()); 
    	}
    	else if (query.getFilter() == Filter.INCLUDE)
    	{
	        try
	        {
	            return _terralibService.getTypeBoundingBox(query.getTypeName());
	        }
	        catch(TypeNotFoundException e)
	        {
	            throw new IOException(e.getMessage(),e);
	        }
	        catch (MismatchedDimensionException e)
	        {
	            throw new IOException(e.getMessage(),e);
	        }
	        catch (IllegalStateException e)
	        {
	            throw new IOException(e.getMessage(),e);
	        }
    	}
    	else
    	{
    		ReferencedEnvelope envelope = new ReferencedEnvelope(getSchema(query.getTypeName()).getCoordinateReferenceSystem());
    		FeatureCollection<SimpleFeatureType, SimpleFeature> features = getFeatureSource(query.getTypeName()).getFeatures(query);
    		Iterator<SimpleFeature> it = features.iterator();
    		while (it.hasNext())
    		{
    			SimpleFeature feature = it.next();
    			BoundingBox bounds = feature.getBounds();
    			Envelope env = new Envelope(bounds.getMinX()
    					,bounds.getMaxX(),bounds.getMinY(), bounds.getMaxY());
    			envelope.expandToInclude(env);
    		}
    		return envelope;
    	}
    }
    
    /* (non-Javadoc)
     * @see org.geotools.data.AbstractDataStore#getSupportedHints()
     */
    @Override
    protected Set<Key> getSupportedHints()
    {
        return _supportedHints;
    }

    @Override
    public void dispose()
    {
        super.dispose();
        
        if (_terralibService != null)
        {
            _terralibService.dispose();
        }
        
        if (_metadata != null)
        {
            _metadata.dispose();
        }
        
        try
        {
            _con.close();
        }
        catch (SQLException e)
        {
            _logger.error("Unable to close jdbc datastore connection", e);
        }
    }
    /**
     * 
     * @param persistenceHandler
     * @since TDK 3.0.0
     */
    public void setAttributePersistenceHandler(TerralibAttributePersistenceHandler persistenceHandler)
    {
        if(persistenceHandler == null)
        {
            throw new NullArgumentException("persistenceHandler");
        }
        
        _persistenceHandler = persistenceHandler;
    }

    /**
     * @return
     * @since
     */
    public TerralibAttributePersistenceHandler getAttributePersistenceHandler()
    {
        return _persistenceHandler;
    }
    

    /*
     * (non-Javadoc)
     * @see org.geotools.data.DataStore#createSchema(org.opengis.feature.simple.SimpleFeatureType)
     */
    @Override
    public void createSchema(SimpleFeatureType featureType) throws IOException 
    {
    	validadeSchema(featureType);
    	
    	//creates the terralib layer
    	try 
    	{
			_terralibService.createType(featureType.getTypeName(), featureType.getGeometryDescriptor(), featureType.getCoordinateReferenceSystem());
		} 
    	catch (InvalidCrsWktException e) 
    	{
			throw new IOException("The feature type's Coordinate Reference System is not supported by Terralib Provider.",e);
		}
    	catch (Exception e)
    	{
    		throw new IOException(e.getMessage(),e);
    	}

    	Map<String,String> linkTables = null;
    	
    	//check if the provider should handle the attribute table
    	if (_persistenceHandler.canProcess(featureType.getTypeName(), TerralibAttributePersistenceHandlerTypeOperation.CREATE_FEATURE_TYPE))
    	{
    		try
    		{
    			linkTables = _persistenceHandler.createFeatureTypeAttributes(featureType);
    		}
    		catch (IOException e)
    		{
    			//undo the layer creation
				try 
				{
					_terralibService.dropFeatureType(featureType.getTypeName(),true);
				} 
				catch (TypeNotFoundException e1) 
				{
				}
				throw e;    			
    		}
    		
    		if (linkTables == null)
    			throw new IOException("Persistence handler shouldn't return null.");
    	}
    	else
    	{
    		linkTables = new HashMap<String, String>();
    		if (featureType.getAttributeCount() > 1)
    		{
    			try
    			{
    				createAttributeTable(featureType,linkTables);
    			}
    			catch (IOException e)
    			{
    				//attribute table failed, tries to undo the layer creation
    				try 
    				{
						_terralibService.dropFeatureType(featureType.getTypeName(),true);
					} 
    				catch (TypeNotFoundException e1) {}
    				throw e;
    			}
    	    	catch (Exception e)
    	    	{
    	    		throw new IOException(e.getMessage(),e);
    	    	}
    			
    		}
    	}
    	
    	validateLinkTable(featureType,linkTables);
    	
    	//link the attribute table to the terralib layer
    	for (Entry<String,String> linkTableEntry: linkTables.entrySet())
    	{
    		try
    		{
    			_terralibService.linkAttributeTable(featureType.getTypeName(), linkTableEntry.getKey(), linkTableEntry.getValue());
    		}
    		catch (Exception e)
    		{
    			//tries to undo the attribute table
    		    undoLayerAttributeCreation(featureType,linkTables);
    			throw new IOException("Error linking the attribute table to the layer",e);
    		}
    	}
    }
    
    
    protected void validateLinkTable(SimpleFeatureType featureType, Map<String,String> linkTables) throws IOException
    {
        //check all link names (will be empty if the link table was handle by terralib)
        for (Entry<String,String> linkTableEntry: linkTables.entrySet())
        {
            if (nameIsInvalid(linkTableEntry.getKey()))
            {
                undoLayerAttributeCreation(featureType,linkTables);
                throw new IOException("The link table name (" +  linkTableEntry.getKey() + ") is invalid.");
            }
            if (nameIsInvalid(linkTableEntry.getValue()))
            {
                undoLayerAttributeCreation(featureType,linkTables);
                throw new IOException("The link column name (" +  linkTableEntry.getValue() + ") is invalid.");
            }
        }        
    }
    
    protected void undoLayerAttributeCreation(SimpleFeatureType featureType, Map<String,String> linkTables) throws IOException
    {
        //tries to undo the attribute table
        if (_persistenceHandler.canProcess(featureType.getTypeName(), TerralibAttributePersistenceHandlerTypeOperation.CREATE_FEATURE_TYPE))
        {
            _persistenceHandler.undoCreateFeatureTypeAttributes(featureType, linkTables);
        	try 
        	{
    			_terralibService.dropFeatureType(featureType.getTypeName(), true);
    		}
        	catch (TypeNotFoundException e) 
        	{
    			throw new IOException(e);
    		}
        }
        else
        {
        	try 
        	{
    			_terralibService.dropFeatureType(featureType.getTypeName());
    		}
        	catch (TypeNotFoundException e) 
        	{
    			throw new IOException(e);
    		}
        }
    }
    
    
    /**
     * Checks the schema, to see if it's compatible with terralib provider
     * @param feature The feature to verify
     * @throws IOException if any error if found.
     */
    protected void validadeSchema(SimpleFeatureType featureType) throws IOException
    {
    	if (featureType == null)
    		throw new NullArgumentException("featureType");

    	//check if the feature type's name is valid
		if (nameIsInvalid(featureType.getTypeName()))
			throw new IOException("The feature type's name " + featureType.getTypeName() + " is invalid.");
    	
    	GeometryDescriptor geometry = featureType.getGeometryDescriptor();
    	if (geometry == null)
    		throw new IOException("The feature type must have a default Geometry");

    	//checks if the default geometry is actually a geometry
    	TypeAttributeMap geomType = TypeAttributeMap.fromBindingClass(geometry.getType().getBinding());
    	if (!geomType.isGeometry())
    		throw new IOException("The default geometry attribute is not a geometry.");
    	
    	//checks if all attributes have valid types
    	//checks if there are more than one geometry attribute
    	//checks also if there are attributes with the wrong name
    	int numOfGeometries = 0;
    	List<AttributeDescriptor> attributes = featureType.getAttributeDescriptors();
    	for (AttributeDescriptor desc: attributes)
    	{
    		TypeAttributeMap type = TypeAttributeMap.fromBindingClass(desc.getType().getBinding());
    		if (type == null)
    			throw new IOException("Attribute " + desc.getLocalName() + " has a binding type (" + desc.getType().getBinding() + ") that is not supported by terralib.");
    		
    		if (type == TypeAttributeMap.TA_LONG)
    			_logger.warn("Schema to create has an attribute " + desc.getLocalName() + " with type=LONG. Terralib does not support, so we'll convert it to INT.");
    		
    		if (type.isGeometry())
    			numOfGeometries++;
    		
    		if (isReservedName(desc.getLocalName()))
    		    throw new IOException("One of the schema attributes is called " + desc.getLocalName() + ". That name is invalid because it's a reserved name at terralib.");
    		
    		if (nameIsInvalid(desc.getLocalName()))
    			throw new IOException("The attribute name " + desc.getLocalName() + " is invalid.");
    	}
    	if (numOfGeometries != 1)
    		throw new IOException("The feature type can have only one geometry attribute, the default Geometry.");
    	
//    	//tests if there is a persistance handler and there are non-geometry attributes (terralib provider does not support that yet)
//    	if ((_persistenceHandler.canProcess(featureType.getTypeName(), TerralibAttributePersistenceHandlerTypeOperation.CREATE_FEATURE_TYPE))
//    		&& (featureType.getAttributeCount() > 1))
//    		throw new IOException("You can't define non-geometry attributes and use a persistence handler to handle CREATE_FEATURE_TYPE. Terralib Provider does not support that.");
    }
    
    protected boolean isReservedName(String localName) 
    {
		for (String s: RESERVED_NAMES)
		{
			if (s.equalsIgnoreCase(localName))
				return true;
		}
		return false;
	}

	protected String getAttributeTableName(SimpleFeatureType featureType)
    {
    	return featureType.getTypeName() + "_attr";
    }
    
    /**
     * Creates the terralib attribute table
     * @throws IOException 
     */
    protected void createAttributeTable(SimpleFeatureType featureType, Map<String,String> linkTables) throws IOException
    {
		//remove the feature type's geometry attribute from the list of attributes to create the attribute table
    	List<AttributeDescriptor> attributes = new ArrayList<AttributeDescriptor>();
    	for (AttributeDescriptor attrDesc: featureType.getAttributeDescriptors())
    	{
    		if (!TypeAttributeMap.fromBindingClass(attrDesc.getType().getBinding()).isGeometry())
    			attributes.add(attrDesc);
    	}

		//find the terralib link attribute name
    	String defaultColumnName = DefaultTerralibService.getFEATURE_ID_COLUMN_NAME();
    	String linkColumnName = defaultColumnName +"0";
    	int i = 0;
    	while (columnExists(linkColumnName,attributes))
    	{
    		i++;
    		linkColumnName = defaultColumnName + i;  
    	}
    	
    	//creates a new attribute descriptor, with the link attribute name
    	attributes.add(TerralibAttributeHelper.buildType(linkColumnName, TypeAttributeMap.TA_STRING, false, false, 255));
    	
    	try
    	{
    		_terralibService.createAttributeTable(getAttributeTableName(featureType), attributes);
    	}
    	catch (Exception e)
    	{
    		try {
				_terralibService.dropFeatureType(featureType.getTypeName(),true);
			} 
    		catch (TypeNotFoundException e1) 
    		{
    		}
    		throw new IOException("Error creating the attribute table: " + e.getMessage(),e);
    	}
    	
    	linkTables.put(getAttributeTableName(featureType), linkColumnName);
    }

	private boolean columnExists(String linkColumnName, List<AttributeDescriptor> attributes) 
	{
		for (AttributeDescriptor attrDesc: attributes)
		{
			if (attrDesc.getLocalName().equalsIgnoreCase(linkColumnName))
				return true;
		}
		return false;
	}
	
	/* (non-Javadoc)
	 * @see org.geotools.data.AbstractDataStore#getCount(org.geotools.data.Query)
	 */
	@Override
	protected int getCount(Query query) throws IOException
	{
	    if (query == null)
	        throw new NullArgumentException("query");
	    
	    String typeName = query.getTypeName();
	    if ((typeName != null) && (!typeName.equals("")))
	    {
	        if (nameIsInvalid(typeName))
	            throw new IllegalArgumentException("The typename " + typeName + " is invalid.");
	        
	        
	        if(!query.getFilter().equals(Filter.INCLUDE))
	        {
	        	return -1;
	        }
	        
	        try
            {
                return _terralibService.getGeometryCount(typeName);
            }
            catch (TypeNotFoundException e)
            {
                throw new IOException(e);
            }
	    }
	    else
	        throw new IllegalArgumentException("The typename " + typeName + " is invalid.");
	}
	
	/**
     * Checks if the given name is invalid for table or field names.
     * @param name
     * @return
     */
    protected boolean nameIsInvalid(String name) 
    {
        name = name.toUpperCase();
        
        for (String illegalWord: ILLEGAL_WORDS)
        {
        	if (name.contains(illegalWord))
        		return true;
        }
        
        if ((name.length() > 0) && (name.substring(0, 1).equals(" ")))
        	return true;
        else
        	return false;
    }	

    public void dropSchema(String typeName) throws IOException
    {
    	if (typeName == null)
    		throw new NullArgumentException("typeName");
    	
    	
    	if (_persistenceHandler.canProcess(typeName, TerralibAttributePersistenceHandlerTypeOperation.DROP_FEATURE_TYPE))
    	{
    		FeatureType feature = getSchema(typeName);
    		
    		try
    		{
    			_terralibService.dropFeatureType(typeName, true);
    		}
    		catch (TypeNotFoundException e)
    		{
    			throw new IOException(e);
    		}
    		
			_persistenceHandler.dropFeatureTypeAttributes(feature);
    	}
    	else
    	{
    		try
    		{
    			_terralibService.dropFeatureType(typeName);
    		}
    		catch (TypeNotFoundException e)
    		{
    			throw new IOException(e);
    		}
    	}
    }
    
    /**
     * Clear the terralib memory metadata. Terralib stores all the featureTypes at memory
     * when it's started. When you use terralibDataStore to change anything, we update 
     * the cache automatically. But if a different program changes the database metadata
     * you won't see the changes (for ie, if a typeName changes).
     * This method asks terralib to reload all its metadata, making these changes visible.
     */
    public void reloadMetadata()
    {
        _terralibService.reloadMetadata();
    }
    
    /**
     * Return the list of Terralib reserved names, that can't be used as attribute names. 
     * @return List of Terralib reserved names
     */
    public static String[] getReservedNames()
    {
        String[] reservedNames = new String[RESERVED_NAMES.length];
        for (int i=0; i<RESERVED_NAMES.length; i++)
            reservedNames[i] = RESERVED_NAMES[i];
        
    	return reservedNames;
    }
    
    /**
     * Return the list of Terralib illegal words, that can't be part of attribute names. 
     * @return List of Terralib illegal words
     */
    public static String[] getIllegalWords()
    {
        String[] illegalWords = new String[ILLEGAL_WORDS.length];
        for (int i=0; i<ILLEGAL_WORDS.length; i++)
            illegalWords[i] = ILLEGAL_WORDS[i];
        
        return illegalWords;
    }

    /* (non-Javadoc)
     * @see org.geotools.data.AbstractDataStore#getFeatureSource(java.lang.String)
     */
    @Override
    public FeatureSource<SimpleFeatureType, SimpleFeature> getFeatureSource(String typeName) throws IOException
    {
        if (typeName == null)
            throw new NullArgumentException("typeName");
        return new TerralibFeatureStore(this,_supportedHints,getSchema(typeName));
    }

    /**
     * Removes all features of a given feature type.
     * @param typeName The feature type name.
     * @throws IOException
     */
	public void removeAllFeatures(String typeName) throws IOException
	{
		try {
			if(_persistenceHandler.canProcess(typeName,TerralibAttributePersistenceHandlerTypeOperation.REMOVE))
			{
				
				_terralibService.removeAllFeatures(typeName, true);
				_persistenceHandler.removeAll(typeName);
			}
			else
			{
				_terralibService.removeAllFeatures(typeName, false);
			}
		} catch (IllegalStateException e) {
			throw new IOException(e);
		} catch (TypeNotFoundException e) {
			throw new IOException(e);
		}
	}

	/**
	 * Gets the data type for a given layer. Terralib accepts a type with both Coverage and Vectorial datas,
	 * but we haven't got any real example with this.
	 * @param typeName The type name to check
	 * @return the {@link TerralibLayerDataType}
	 * @throws IOException if any error occurs while retrieving this info.
	 */
	public TerralibLayerDataType getLayerType(String typeName) throws IOException
	{
		try {
			return _terralibService.getLayerType(typeName);
		} catch (IllegalStateException e) {
			throw new IOException(e);
		} catch (TypeNotFoundException e) 
		{
			throw new IOException(e);
		}
	}

	
	/**
	 * Retrieves the list of coverage files associated with the given type
	 * @param typeName
	 * @return A {@link List} of {@link URL} which locates the coverage files associated with the given type.
	 * @throws IOException if any error occurs while retrieving this info.
	 */
	public List<URL> getCoverageFilePath(String typeName) throws IOException
	{	
		if(getLayerType(typeName) != TerralibLayerDataType.COVERAGE)
		{
			return new ArrayList<URL>();
		}
		
		try {
			return _terralibService.getRasterFileURL(typeName);
		} catch (TypeNotFoundException e) {
			throw new IOException(e);
		} catch (IllegalStateException e) {
			throw new IOException(e);
		} catch (IOException e) {
			throw new IOException(e);
		}
	}
    
}
