/**
 * Tecgraf - GIS development team
 * 
 * Tdk Framework
 * Copyright TecGraf 2010(c).
 * 
 * file: JavaQuerier.java
 * created: Apr 16, 2010
 */
package org.geotools.data.terralib.query;

import java.io.IOException;
import java.sql.Connection;
import java.sql.ResultSet;
import java.sql.SQLException;
import java.sql.Statement;
import java.util.List;

import org.apache.log4j.Logger;
import org.geotools.data.Query;
import org.geotools.data.terralib.TerralibMetadata;
import org.geotools.data.terralib.TerralibService;
import org.geotools.data.terralib.TerralibMetadata.AttributeTableInfo;
import org.geotools.data.terralib.exception.NullArgumentException;
import org.geotools.data.terralib.exception.TerralibProviderRuntimeException;
import org.geotools.data.terralib.exception.TypeNotFoundException;
import org.geotools.data.terralib.query.check.QueryChecker;
import org.geotools.data.terralib.query.filter.TerralibFilterToSQL;
import org.geotools.data.terralib.query.portal.JavaPortal;
import org.geotools.data.terralib.util.TerralibResultSet;
import org.geotools.data.terralib.util.TypeAttributeMap;
import org.opengis.filter.And;
import org.opengis.filter.Filter;
import org.opengis.filter.sort.SortBy;
import org.opengis.filter.sort.SortOrder;
import org.opengis.filter.spatial.BinarySpatialOperator;

/**
 * Java version for the Querier, does not call Terralib to query the database.
 * Creates all queries in Java using JDBC.
 * @author fabiomano
 * @since TDK 3.1
 */
public class JavaQuerier
{
    private static final String COL_GEOM_ID = "geom_id";    //terralib column that holds the geometry ID
    private static final String COL_OBJECT_ID = "object_id";    //terralib column that holds the object ID

    private static final String ROW_NUMBER_COL = "rowNumber";   //column name for row_number() function
    private final String START_INDEX_TABLE = "StartIndexFrom";  //table name for the FROM that creates the row_number values
    private final String START_INDEX_OBJECT_ID = "StartIndexObjectId";  //column name for the object_id at row_number table
    
    private static Logger _logger = Logger.getLogger(JavaQuerier.class);
    
    private TerralibService _service;
    private Connection _con;
    private QueryChecker _checker;
    
    private TerralibMetadata _metadata;
    
    
    
    public JavaQuerier(TerralibService service, Connection jdbcConnection, TerralibMetadata metadata, QueryChecker checker)
    {
        if (service == null)
            throw new NullArgumentException("service");
        if (metadata == null)
            throw new NullArgumentException("metadata");
        if (checker == null)
            throw new NullArgumentException("checker");
    
        _metadata = metadata;
        _service = service;
        _con = jdbcConnection;
        _checker = checker;
    }
    
    public QueryData execute(QuerierParams params) throws IOException, TypeNotFoundException
    {
        if (_con == null)
            throw new IOException("The connection with the database could not be estabilished");
        if (params == null)
            throw new NullArgumentException("params");
        
        _checker.checkQuery(params.getQuery()); //checks if the current query is supported
        
        String query = buildQuery(params);
        
        JavaPortal portal = new JavaPortal(executeQuery(query));
        
        return new TerralibQueryData(_service,portal,params.getFeatureTypeInfo());
    }

    /**
     * @param query
     * @return
     * @throws SQLException 
     */
    private ResultSet executeQuery(String query) throws IOException
    {
        try
        {
            Statement stmt = _con.createStatement();
            return new TerralibResultSet(stmt.executeQuery(query));
        }
        catch (SQLException e)
        {
            throw new IOException("Error executing query to retrieve terralib geometries.",e);
        }
    }

    /**
     * @param params
     * @return
     */
    protected String buildQuery(QuerierParams params) throws IOException
    {
        String query = "";
        
        Class<?> binding = params.getFeatureTypeInfo().getDefaultGeometryType().getBinding();

        //RETRIEVE THE GEOMETRY TYPE 
        TypeAttributeMap type = TypeAttributeMap.fromBindingClass(binding);
        if (type == null)
            throw new TerralibProviderRuntimeException("Type " + binding + " not supported");
        else if (!type.isGeometry())
            throw new TerralibProviderRuntimeException("Type " + binding + " is not a geometry");


        Filter filter = params.getFilter();
        
        //CHECKS IF TerralibFilterToSQL SUPPORTS THE GIVEN FILTER. IF NOT, WILL IGNORE THE FILTERS AND RETRIEVE EVERYONE (FILTERS IN MEMORY)
        TerralibFilterToSQL filterToSQL = new TerralibFilterToSQL(params.getFeatureTypeInfo().getFeatureType(),_metadata.getGeometryTable(params.getFeatureTypeInfo().getTypeName()));
        if (!filterToSQL.supports(filter))
        {
            _logger.warn("Filter " + params.getFilter() + " is not supported by JavaQuerier. All features will be retrieved and it will be filtered in memory (slower)");
            return buildSelectSimple(params,"",false,"");
        }

        //IF IT HAS STARTINDEX AND/OR MAXFEATURES, QUERY WILL BE MORE COMPLEX AND USE A DIFFERENT METHOD TO BUILD IT
        if (hasMaxFeatures(params) || hasStartIndex(params))
        {
            query = buildStartIndexMaxFeaturesSelect(params,filterToSQL.encodeToString(filter));
        }
        //CHECKS IF THE FILTER IS A SPATIAL FILTER. IF THE FILTER CONTAINS ONLY ONE SPATIAL FILTER, WILL OPTIMIZE THE QUERY
        else if (BinarySpatialOperator.class.isAssignableFrom(filter.getClass()))
        {
            TerralibFilterToSQL optimizedFilterToSQL = new TerralibFilterToSQL(params.getFeatureTypeInfo().getFeatureType());    //constructor without the geometry table name optimizes the filter
            String spatialWhere = optimizedFilterToSQL.encodeToString(filter);
            query = buildSelectSimple(params,"",true,spatialWhere);
        }
        //CHECKS IF THE FILTER IS AN AND FILTER WITH 2 FILTERS, ONE SPATIAL FILTER AND THE OTHER NOT. IF IT IS, WILL OPTIMIZE THE QUERY
        else if (isAndSpatialFilter(filter)) 
        {
            TerralibFilterToSQL optimizedFilterToSQL = new TerralibFilterToSQL(params.getFeatureTypeInfo().getFeatureType());    //constructor without the geometry table name optimizes the filter
            Filter spatial = getSpatialAndFilter((And) filter);
            Filter nonSpatial = getNonSpatialAndFilter((And)filter);
            String where = filterToSQL.encodeToString(nonSpatial);;
            String spatialWhere = optimizedFilterToSQL.encodeToString(spatial); 
            
            query = buildSelectSimple(params,where,true,spatialWhere);
        }
        //USES NON-OPTIMIZED FILTER TO SQL
        else    
        {
            String where = filterToSQL.encodeToString(filter);
            query = buildSelectSimple(params,where,false,"");
        }
        
        return query;
    }

    /**
     * Returns the spatial part of the And filter
     * @param filter
     * @return
     */
    private Filter getSpatialAndFilter(And filter)
    {
        Filter left = filter.getChildren().get(0);
        if (BinarySpatialOperator.class.isAssignableFrom(left.getClass()))
            return left;
        else
            return filter.getChildren().get(1);
    }
    
    /**
     * Returns the non spatial part of the And filter
     * @param filter
     * @return
     */
    private Filter getNonSpatialAndFilter(And filter)
    {
        Filter left = filter.getChildren().get(0);
        if (!BinarySpatialOperator.class.isAssignableFrom(left.getClass()))
            return left;
        else
            return filter.getChildren().get(1);
    }
    

    /**
     * Tests if the filter is an AND filter that contains a spatial filter and something else. In this case it can be optimized 
     * @param filter The filter to check
     * @return true if its this case, false otherwise
     */
    private boolean isAndSpatialFilter(Filter filter)
    {
        if (    (And.class.isAssignableFrom(filter.getClass()))
                &&( ((And)filter).getChildren().size() == 2)    //and between only two values
            )
        {
            Filter left = ((And)filter).getChildren().get(0);
            Filter right = ((And)filter).getChildren().get(1);
            
            //if left is spatial and right isn't
            if ((BinarySpatialOperator.class.isAssignableFrom(left.getClass()))
                && (!BinarySpatialOperator.class.isAssignableFrom(right.getClass()))
                )
                return true;
            
            //if right is spatial and left isn't
            if ((BinarySpatialOperator.class.isAssignableFrom(right.getClass()))
                && (!BinarySpatialOperator.class.isAssignableFrom(left.getClass()))
                )
                return true;
        }
            
       return false; 
    }
    
    /**
     * Builds the query to retrieve the geometry and the attribute tables when StartIndex
     * and MaxFeatures are not defined.
     * @throws IOException 
     */
    private String buildSelectSimple(QuerierParams params, final String where, boolean optimize, String optimizedSpatialWhere) throws IOException
    
    {
        String typeName = params.getFeatureTypeInfo().getTypeName();
        
        String geomTable = _metadata.getGeometryTable(typeName);
        List<AttributeTableInfo> attrTables = _metadata.getAttributeTables(typeName);
        
        /*
         * Important: This query here is only compatible with SQL-Server and Access, 
         * for different datastores we might need to implement a different Querier.
         */

        String query = "SELECT * FROM " + geomTable + " AS P ";

        String whereQuery;
        
        /*
         * If optimize == true, will not use IN clause at the WHERE clause
         * for spatial filters. Instead, will do two selects at the geometry  
         * table and join them. This works faster sometimes.
         */
        if (optimize)
        {
            //Adds the extra FROM to the query 
            query = query + ", (select distinct " + COL_OBJECT_ID + " as spatial_object_id from " + geomTable + " " + optimizedSpatialWhere + ") as spatialAux";
            
            if (where.isEmpty())
                whereQuery = " WHERE ";
            else
                whereQuery = where + " AND ";

            //adds the JOIN between the extra FROM (for the Geometry WHERE) and the normal geometry table 
            whereQuery = whereQuery + " p." + COL_OBJECT_ID + " = spatialAux.spatial_object_id ";
        }
        /*
         * If optimize == false, the spatial filter will be fully resolved at the WHERE 
         * clause using OBJECT_ID IN (select object_id from GEOMTABLE where <spatial_filter>)
         * This must be done because terralib contains multiple lines with geometries 
         * for a single feature.   
         */
        else
            whereQuery = where;
        
        /*
         * Builds the rest of the FROM clause
         */
        query = query + buildAttributeFrom(attrTables) + " ";
        
        //now will build the where
        if (whereQuery.isEmpty())
            whereQuery = " WHERE " + buildAttributeJoinWhere(attrTables,"P");
        else
            whereQuery = whereQuery + " AND " + buildAttributeJoinWhere(attrTables,"P");
        
        query = query + whereQuery + buildOrderBy(params,true,attrTables);
        
        return query;
    }
    
    /**
     * Builds the FROM clause including all attribute tables.
     * Will return something like
     * , ATTR_TABLE1 as t1, ATTR_TABLE2 as t2
     */
    private String buildAttributeFrom(List<AttributeTableInfo> attrTables)
    {
        String from = "";
        int i = 1;
        for (AttributeTableInfo info: attrTables)
        {
            from = from + ", " + info.getTableName() + " t" + i;
            i++;
        }
        return from;
    }
    
    /**
     * Builds the WHERE clause the joins for each attribute tables.
     * Will return something like
     * t1.object_id = p.object_id AND t2.object_id = p.object_id
     */
    private String buildAttributeJoinWhere(List<AttributeTableInfo> attrTables, String attrTableAlias)
    {
        String where = "";
        int i = 1;
        for (AttributeTableInfo info: attrTables)
        {
            if (!where.isEmpty())
                where = where + " AND ";
            
            where = where + attrTableAlias + "." + COL_OBJECT_ID + " = t" + i + "." + info.getUniqueId();
            i++;
        }
        return where;
    }
    
    
    /**
     * @param params
     * @return
     */
    private String buildMaxFeaturesStartIndexWhereQuery(QuerierParams params)
    {
        String rowNumberWhere = "";
        if (hasMaxFeatures(params) || (hasStartIndex(params)))
        {
            rowNumberWhere = " AND " + COL_OBJECT_ID + " = " + START_INDEX_OBJECT_ID;
            
            int startIndex = 1;
            
            if (hasStartIndex(params))
            {
                startIndex = params.getQuery().getStartIndex();
                rowNumberWhere = rowNumberWhere + " AND " 
                    + ROW_NUMBER_COL + " >= " + startIndex 
                    ;
            }
            
            if (hasMaxFeatures(params))
                rowNumberWhere = rowNumberWhere 
                    + " AND " + ROW_NUMBER_COL + " < " 
                    + (startIndex + params.getQuery().getMaxFeatures())
                    ;
        }
        return rowNumberWhere;
    }

    /**
     * Creates the FROM clause to get row numbers.
     * Will return something like this:
     * , 
     * SELECT ROW_NUMBER() OVER (ORDER BY CLI_S_CITY, cli_s_id) AS rowNumber
     * , object_id AS StartIndexObjectId 
     * FROM (
     * SELECT DISTINCT object_id FROM Points7 AS P 
     * , Clientes_attr t1 WHERE cli_s_city = 'Rio de Janeiro' 
     * AND P.object_id = t1.cli_s_id
     * ) AS TEMP, Clientes_attr t1 
     * WHERE t1.cli_s_id = TEMP.object_id  
     * 
     * @param params
     * @return
     */
    private String buildRowNumberFromQuery(QuerierParams params, String geomTable, String where, List<AttributeTableInfo> attrTables) throws IOException
    {
        if (hasMaxFeatures(params) || (hasStartIndex(params)))
        {
            /*
             * Builds the internal query that will get all distinct values for 
             * OBJECT_ID, so we can create the row numbers at select TOP X and/or
             * OFFSET.
             */
            String internalQuery = "SELECT DISTINCT P." 
                + COL_OBJECT_ID + " FROM " + geomTable + " AS P "
                + buildAttributeFrom(attrTables) 
                ;

            String whereQuery = where;
            if (whereQuery.isEmpty())
                whereQuery = " WHERE " + buildAttributeJoinWhere(attrTables,"P");
            else
                whereQuery = " " + whereQuery + " AND " + buildAttributeJoinWhere(attrTables,"P");
            
            internalQuery = internalQuery + whereQuery;
            
            /*
             * Surrounds the internal query at a query that counts the row numbers.
             */
            return 
                ", ("
                + " SELECT ROW_NUMBER() OVER (" + buildOrderBy(params,false,attrTables) + ") AS " + ROW_NUMBER_COL
                + ", " + COL_OBJECT_ID + " AS " + START_INDEX_OBJECT_ID + " FROM (" 
                + internalQuery
                + ") AS TEMP "
                + buildAttributeFrom(attrTables) //FROM ATTR_TABLE1, ...
                + " WHERE "
                + buildAttributeJoinWhere(attrTables,"TEMP")
                + ") AS " + START_INDEX_TABLE
                ;
        }
        else
            return "";
    }

    /**
     * Creates the order by clause (something like ORDER BY ATTRIBUTE1, OBJECT_ID, GEOM_ID)
     * includeGeo says if it needs to include the fields from the geometry table 
     * (object_id and geom_id)
     * or only the fields from the attribute table.
     */
    private String buildOrderBy(QuerierParams params, boolean includeGeom, List<AttributeTableInfo> attrTables)
    {
        String orderBy = "";
        if (params.getQuery().getSortBy() != null)
        {
            for (SortBy sort: params.getQuery().getSortBy())
            {
                orderBy = orderBySeparator(orderBy);
                
                orderBy = orderBy + sort.getPropertyName();
                if (sort.getSortOrder() == SortOrder.DESCENDING)
                    orderBy = orderBy + " DESC ";
            }
        }
        
        if (includeGeom)
        {
            orderBy = orderBySeparator(orderBy);
            orderBy += " P." + COL_OBJECT_ID + ", P." + COL_GEOM_ID;
        }
        else
        {
            /*
             * includes only the object_id from the first attribute table at the order by
             * (its not necessary to add all) 
             */
            if (attrTables.size() > 0)
            {
                orderBy = orderBySeparator(orderBy);
                orderBy += attrTables.get(0).getUniqueId();
            }
        }
            
        return orderBy;
    }
    
    private String orderBySeparator(String orderBy)
    {
        if (orderBy.isEmpty())
            return " ORDER BY ";
        else
            return orderBy + ",";
    }
    
    private boolean hasStartIndex(QuerierParams params)
    {
        return (params.getQuery().getStartIndex() != null);
    }
    
    private boolean hasMaxFeatures(QuerierParams params)
    {
        return (params.getQuery().getMaxFeatures() != Query.DEFAULT_MAX);
    }    
    
    /**
     * Builds the query to retrieve the geometry and the attribute tables when the
     * user selects the StartIndex query parameter.
     * When using start index we won't try to optimize the query (since it's already
     * slower due to start index).
     * @param params
     * @param where
     * @return
     * @throws IOException 
     */
    private String buildStartIndexMaxFeaturesSelect(QuerierParams params, String where) throws IOException
    {
        String typeName = params.getFeatureTypeInfo().getTypeName();
        
        String geomTable = _metadata.getGeometryTable(typeName);
        List<AttributeTableInfo> attrTables = _metadata.getAttributeTables(typeName);
        
        /*
         * Important: This query here is only compatible with SQL-Server, 
         * for different databases we might need to implement a different Querier.
         */

        /*
         * Creates the rowNumber FROM clause (in case the user requested StartIndex).
         * This FROM subquery guarantees that each OBJECT_ID count as one object 
         * (Geometry tables can have more than one row for a single object) 
         */
        String rowNumberFromQuery = buildRowNumberFromQuery(params,geomTable,where,attrTables);
        String rowNumberWhere = buildMaxFeaturesStartIndexWhereQuery(params);
        
        String query;
        
        query = "SELECT * FROM " + geomTable + " AS P " + rowNumberFromQuery;
        
        String whereQuery = ""; //does not include the filter WHERE here, it's already included at the rowNumberFromQuery
        
        int i = 1;
        for (AttributeTableInfo info: attrTables)
        {
            query = query + ", " + info.getTableName() + " t" + i;
            
            if (whereQuery.isEmpty())
                whereQuery = " WHERE ";
            else
                whereQuery = " " + whereQuery + " AND ";
            
            whereQuery = whereQuery + "P." + COL_OBJECT_ID + " = t" + i + "." + info.getUniqueId();
            i++;
        }
        query = query + whereQuery + rowNumberWhere + buildOrderBy(params,true,attrTables);
        
        return query;        
    }    
}
