/**
 * Tecgraf - GIS development team
 *
 * Tdk Framework
 * Copyright TecGraf 2008(c).
 *
 * file: NativeLibraryLoader.java
 * created: 2009/03/11
 */
package org.geotools.data.terralib;

import java.io.BufferedOutputStream;
import java.io.File;
import java.io.FileOutputStream;
import java.io.IOException;
import java.io.InputStream;
import java.util.Properties;

import org.apache.log4j.Logger;

/**
 * Loads the native libraries needed for the terralib datastore provider.
 * @author fabiomano
 * @since TDK3.0.0
 */
public class NativeLibraryLoader
{
    /**
     * 
     */
    private static final String PROPERTIES_FILENAME = "loader.properties";

    /**
     * Release native library directory name.
     */
    private static final String RELEASE_DIR = "release";

    /**
     * Debug native library directory name.
     */
    private static final String DEBUG_DIRECTORY = "debug";
    
    private static final String DEBUG_MODE = "debug";
    private static final String FORCE_OVERWRITE = "forceoverwrite";

    private static final Logger _logger = Logger.getLogger(NativeLibraryLoader.class);
    
    //The default is release mode.
    private static String _library_dir = RELEASE_DIR;
    
    private static final boolean isWindows = System.getProperty("os.name").toLowerCase().indexOf("windows") != -1;
    private static final boolean isLinux = System.getProperty("os.name").toLowerCase().indexOf("linux") != -1;

    private static final String PROVIDER_TEMP_DIR_NAME = "TerralibProviderLibraries";    
    private static String _libraryExtension = null;
    private static String _systemTemporaryDirName = null;
    
    private static boolean _loaded = false;
    
    private static boolean _debugModeOn = false;
    private static boolean _forceOverwrite = true;
    
    static 
    {
        Properties properties = new Properties();
        InputStream stream = null;
        
        stream = NativeLibraryLoader.class.getResourceAsStream("/"+PROPERTIES_FILENAME);
 
        try 
        {
        	properties.load(stream);
		} 
        catch (IOException e) 
        {
			_logger.error("Could not open loader.properties file. Starting loader with release dlls", e);
		}
        finally
        {
        	if(stream != null)
        	{
        		try {
					stream.close();
				} catch (IOException e) {
					_logger.error("Error closing properties file input stream.", e);
				}
        	}
        }
        
        String value = properties.getProperty(DEBUG_MODE);
        if((value != null) && (value.equalsIgnoreCase("true")))
        {
            _debugModeOn = true;
            _library_dir = DEBUG_DIRECTORY;
            
        }
        
        String overwrite = properties.getProperty(FORCE_OVERWRITE);
        if ((overwrite != null) && (overwrite.equalsIgnoreCase("false")))
        {
            _forceOverwrite = false;
        }
  
    }
    
    synchronized public static boolean loadLibraries()
    {
        if (!_loaded)
        {
            if(_debugModeOn)
            {
                _logger.info("Using libraries in debug mode.");
                System.out.println("Using libraries in debug mode.");
            }
            else
            {
                _logger.info("Using libraries in release mode.");
                System.out.println("Using libraries in release mode.");
            }
            
            if (_forceOverwrite)
            {
                _logger.info("Forcing library overwrite.");
                System.out.println("Forcing library overwrite.");
            }
            else
            {
                _logger.info("Not forcing library overwrite.");
                System.out.println("Not forcing library overwrite.");
            }
            
            try{
            	initializeTemporaryDir();
            }
            catch(Throwable t)
            {
            	_logger.warn("Error preparing temp dir.",t);
            }
            
            _logger.info("Using '"+_systemTemporaryDirName+"' as temporary dir");
            
            try
            {   
            	
                loadLibraryFromJar("zlib1");
                if(_debugModeOn)
                {
                    loadLibraryFromJar("tiff_d");
                }
                else
                {
                    loadLibraryFromJar("tiff");
                }
                loadLibraryFromJar("terralib");
                loadLibraryFromJar("teadodriver");
                loadLibraryFromJar("terralibprovider");
                _loaded = true;
            }
            catch(Throwable t)
            {
                _logger.error("Error loading terralib provider dlls.",t);
                t.printStackTrace();
            }            
        }
        return _loaded;
    }
    
    /**
     * 
     * @throws IOException 
     * @since
     */
    private static void initializeTemporaryDir() throws IOException
    {
        _systemTemporaryDirName = getTemporaryDirPath();
        
        File file = new File(_systemTemporaryDirName);
        
        if(file.exists())
        {
            if(file.isFile())
            {
                if(!file.delete())
                {
                    throw new IOException("Error trying to delete file "+_systemTemporaryDirName);
                }
            }
            else if (file.isDirectory())
            {
               if(!file.canRead() || !file.canWrite())
               {
                   throw new IOException("No read/write permission on dir "+_systemTemporaryDirName);
               }
            }
        }
        else
        {
            if(!file.mkdirs())
            {
                throw new IOException("Error trying to create dir "+_systemTemporaryDirName);
            }
        }

    }

    /**
     * 
     * @since
     */
    private static String getTemporaryDirPath()
    {
        String currentTempDir = System.getProperty("java.io.tmpdir");
        if(!currentTempDir.endsWith(File.separator))
        {
            currentTempDir += File.separator;
        }
        return currentTempDir+PROVIDER_TEMP_DIR_NAME+File.separator+_library_dir+File.separator;
    }
    
    private static void copyFileToTempDir(final String libraryName, final File tempLibraryFile) throws IOException
    {
    	
        try
        {
            if (!tempLibraryFile.createNewFile())
                _logger.warn("Temporary file used to load libraries already exists.");
        
        }
        catch (IOException e1)
        {
            throw new IOException("unable to create temporary file for " + libraryName);
        }
        
        InputStream in = NativeLibraryLoader.class.getResourceAsStream(getLibraryResourcePath(libraryName));
        
        if(in == null)
        {
            throw new IOException("Could not find '"+libraryName+"'");
        }
        
        byte[] buffer = new byte[16777216]; //new byte[32768];
        BufferedOutputStream out = null;
        try
        {
            out = new BufferedOutputStream(new FileOutputStream(tempLibraryFile));
            while (true)
            {
                int readed = in.read(buffer);
                if (readed > -1)
                {
                    out.write(buffer, 0, readed);
                }
                else
                {
                    break;
                }
            }
            _logger.debug("Wrote " + tempLibraryFile);
        }
        catch (Exception e)
        {
            _logger.warn("Can't write to " + libraryName + ". It's probably being used by another process. Will try to load it from the existing file.",e);
        }
        finally
        {
            if (out != null)
            {
                out.close();
            }
        }      
        
    }

    private static void loadLibraryFromJar(final String libraryName)
    {
    	File tempLibraryFile = null;
    	tempLibraryFile = new File(_systemTemporaryDirName+libraryName+getLibraryExtension());
    	
    	if ((!_forceOverwrite) && (tempLibraryFile.exists()))
    	{
    	    _logger.info("Library " + libraryName + " already exists. It won't be copied.");
    	}
    	else
    	{
        	try 
        	{
    			copyFileToTempDir(libraryName,tempLibraryFile);
    		} 
        	catch (IOException e) {
    			_logger.warn("Error trying to copy '"+tempLibraryFile.getName()+"' to temp dir", e);
    		}
    	}

    	System.load(tempLibraryFile.getAbsolutePath());
    }
    
    /**
     * Get the library's resource path
     * @param library the library name (without an extension)
     * @return resource path
     */
    private static String getLibraryResourcePath(String library)
    {
        return "/" + _library_dir + "/" + library + getLibraryExtension(); 
    }
    
    /**
     * Get the library extension for the user's O.S.
     * @return The library extension for the user's O.S.
     */
    private static String getLibraryExtension()
    {
        if (_libraryExtension == null)
        {
            if (isWindows)
                _libraryExtension = ".dll";
            else if (isLinux)
                _libraryExtension = ".so";
            else
                throw new UnsatisfiedLinkError("unsupported O.S.");
        }
        return _libraryExtension;
    }
    
    /**
     * To test if the Loader works.. If no exception is throw
     * , it has worked!
     * @param args
     */
    public static void main(String[] args)
    {
        if (loadLibraries())
            System.out.println("Done...");
        else
            System.out.println("Error loading dlls...");
    }
}
