package tecgraf.openbus.browser;

import java.util.ArrayList;
import java.util.List;
import java.util.concurrent.ArrayBlockingQueue;
import java.util.concurrent.CopyOnWriteArrayList;
import java.util.concurrent.atomic.AtomicInteger;

/**
 * Reengenharia de {@link java.util.Observable}, com otimizaes:
 * <p>
 * O disparo dos observadores  feito em uma thread separada, para no bloquear
 * o fluxo de execuo de quem est notificando.
 * 
 * <p>
 * Caso haja a necessidade de que a notificao dos observadores seja sncrona,
 *  possvel utilizar o {@link #notifyObserversSync()}.
 * 
 * @author Daltro Gama (daltro@tecgraf.puc-rio.br)
 */
public abstract class AsyncObservable {

	/**
	 * Interface a ser observada por quem pretende receber notificaes de objetos
	 * {@link AsyncObservable}. As instncias desta devem ser registradas atravs
	 * de {@link AsyncObservable#addObserver(AsyncObserver)}.
	 */
	public interface AsyncObserver {
		/**
		 * Cdigo que ser executado na notificao.
		 * 
		 * @param src Instncia da extenso de {@link AsyncObservable} que gerou a
		 *          notificao.
		 */
		public void event(AsyncObservable src);
	}

	/**
	 * Fila de eventos a ser executado numa thread dedicada.
	 */
	private static final ArrayBlockingQueue<Runnable> queue = new ArrayBlockingQueue<Runnable>(128);

	/**
	 * Conjunto dos observadores, em um modelo de coleo no bloqueante, porm thread-safe.
	 */
	private final List<AsyncObserver> observers = new CopyOnWriteArrayList<AsyncObserver>();

	/**
	 * Indicador de que houveram modificaes. Flag usada para seguir o padro usado
	 * em {@link java.util.Observable}.
	 */
	private volatile boolean changed;

	/**
	 * Thread de execuo de notificaes.
	 */
	private static final Thread asyncEventThread = new Thread() {
		@Override
		public void run() {
			try {
				while (!isInterrupted()) {
					Runnable r = queue.take();
					try {
						r.run();
					}
					catch (Throwable e) {
						if (e instanceof InterruptedException) {
							throw (InterruptedException) e;
						}
						e.printStackTrace(System.err);
					}
				}
			}
			catch (InterruptedException ie) {
			}
		};
	};

	static {
		asyncEventThread.setDaemon(false);
		asyncEventThread.setName("AsyncObserver event thread");
		asyncEventThread.start();
		Runtime.getRuntime().addShutdownHook(new Thread() {
			@Override
			public void run() {
				asyncEventThread.interrupt();
				try {
					asyncEventThread.join();
				}
				catch (InterruptedException e) {
					e.printStackTrace(System.err);
				}
			}
		});
		Runtime.getRuntime().addShutdownHook(new Thread("AsyncObserver terminator thread") {
			@Override
			public void run() {
				asyncEventThread.interrupt();
				Runnable r;
				while ((r = queue.poll()) != null) {
					try {
						r.run();
					}
					catch (Throwable e) {
						e.printStackTrace(System.err);
					}
				}
			}
		});
	}

	/**
	 * Mesma semntica do {@link java.util.Observable#setChanged()}
	 */
	protected final void setChanged() {
		this.changed = true;
	}

	/**
	 * Associar um novo observador assncrono neste objeto.
	 * @param observer Instncia do observador.
	 */
	public void addObserver(AsyncObserver observer) {
		observers.add(observer);
	}

	/**
	 * Remover uma instncia de observador da coleo de observadores
	 * associados  instncia de {@link AsyncObservable}.
	 * @param observer Instncia a ser removida.
	 * @return true se ela foi removida. False se ela no estava previamente associada.
	 */
	public boolean removeObserver(AsyncObserver observer) {
		return observers.remove(observer);
	}

	/**
	 * Realizar a notificao dos observadores, porm de forma bloqueante.
	 * Este mtodo no retorna enquanto todos os observadores associados no
	 * forem executados.
	 * 
	 * <p>
	 * A notificao s ocorre de fato se {@link #setChanged()} tiver sido chamado
	 * internamente depois da ltima notificao.
	 */
	public void notifyObserversSync() {
		if (!changed)
			return;
		changed = false;
		try {
			List<AsyncObserver> snapshot = new ArrayList<AsyncObserver>(observers);
			final AtomicInteger latch = new AtomicInteger(snapshot.size());
			for (final AsyncObserver o : snapshot) {
				queue.put(new Runnable() {
					@Override
					public void run() {
						try {
							o.event(AsyncObservable.this);
						}
						finally {
							latch.decrementAndGet();
						}
					}
				});
			}
			while (latch.get() > 0) {
				Thread.yield();
				if (asyncEventThread == null || asyncEventThread.isInterrupted() || !asyncEventThread.isAlive()) {
					Runnable x;
					while ((x = queue.poll()) != null) {
						try {
							x.run();
						}
						catch (Throwable e) {
							e.printStackTrace(System.err);
						}
					}
					return;
				}
			}
		}
		catch (InterruptedException ie) {
			throw new IllegalStateException("Thread interrompida durante a notificao do AsyncObserver", ie);
		}
	}

	/**
	 * Executa os eventos {@link AsyncObserver#event(AsyncObservable)} de todos os
	 * observadores associados, em uma thread separada.
	 * 
	 * <p>
	 * Este mtodo no  bloqueante.
	 * 
	 * <p>
	 * A notificao s ocorre de fato se {@link #setChanged()} tiver sido chamado
	 * internamente depois da ltima notificao.
	 */
	public void notifyObservers() {
		if (!changed)
			return;
		changed = false;
		try {
			for (final AsyncObserver o : observers) {
				queue.put(new Runnable() {
					@Override
					public void run() {
						o.event(AsyncObservable.this);
					}
				});
			}
		}
		catch (InterruptedException ie) {
			throw new IllegalStateException("Thread interrompida durante a notificao do AsyncObserver", ie);
		}
	}

}