package tecgraf.openbus.browser;

import java.awt.event.ActionEvent;
import java.awt.event.ActionListener;
import java.awt.event.MouseAdapter;
import java.awt.event.MouseEvent;
import java.util.regex.Pattern;

import javax.swing.DefaultComboBoxModel;
import javax.swing.JComboBox;
import javax.swing.JMenuItem;
import javax.swing.JPopupMenu;
import javax.swing.SwingUtilities;

/**
 * {@link JComboBox} decorada para trabalhar como uma lista de "coisas"
 * recentemente usadas. As funcionalidades aqui encapsuladas so:
 * <ul>
 * <li>Serializao/Desserializao do conjunto LRU ({@link #getLRUJoinedText()}/{@link #loadLRUJoinedText(String)};
 * <li>Gerenciamento da ordem da lista, no permitindo que ela fique maior do
 * que {@link #setMaxSize(int)};
 * <li>Permite ao usurio limpar a lista, clicando com o boto direito sobre a
 * combobox.
 * </ul>
 * 
 * @author Daltro Gama (daltro@tecgraf.puc-rio.br)
 */
@SuppressWarnings("serial")
public class LRUComboBox extends JComboBox {

	private final DefaultComboBoxModel model = new DefaultComboBoxModel();

	private static final char DIV_CHAR = '';

	private int maxSize = 30;

	private Runnable saveHandler;

	public LRUComboBox() {
		super.setModel(model);
		addMouseListener(new MouseAdapter() {
			@Override
			public void mousePressed(MouseEvent e) {
				if (SwingUtilities.isRightMouseButton(e) && e.getClickCount() == 1) {
					JPopupMenu lruMenu = new JPopupMenu("Lista de itens recentes");
					JMenuItem clearLRUItem = new JMenuItem("Limpar lista");
					clearLRUItem.addActionListener(new ActionListener() {
						@Override
						public void actionPerformed(ActionEvent e) {
							model.removeAllElements();
							if (saveHandler != null)
								saveHandler.run();
						}
					});
					lruMenu.add(clearLRUItem);
					lruMenu.show(LRUComboBox.this, e.getX(), e.getY());
				}
			}
		});
	}

	public String[] getLRUList() {
		String[] res = new String[model.getSize()];
		for (int i = 0; i < model.getSize(); i += 1) {
			res[i] = model.getElementAt(i).toString();
		}

		return res;
	}

	/**
	 * Adiciona uma nova entrada no topo da lista LRU, ou move a entrada para o
	 * topo da lista caso ela j exista.
	 * 
	 * @param value Texto a ser includo ou movido para o topo
	 */
	public void addLRUItem(String value) {
		model.removeElement(value);
		model.insertElementAt(value, 0);
		while (model.getSize() > maxSize)
			model.removeElementAt(maxSize);
	}

	/**
	 * Tamanho mximo da lista. tens mais antigos sero descartados.
	 * 
	 * @return Nmero de itens
	 */
	public int getMaxSize() {
		return maxSize;
	}

	/**
	 * Tamanho mximo da lista. tens mais antigos sero descartados.
	 * 
	 * @param maxSize Nmero de itens
	 */
	public void setMaxSize(int maxSize) {
		this.maxSize = maxSize;
	}

	/**
	 * Recuperar a lista LRU consolidada em uma s String, que poder ser usada
	 * para persistncia.
	 * 
	 * @return Toda a lista LRU da combo
	 */
	public String getLRUJoinedText() {
		StringBuilder res = new StringBuilder();
		for (int i = 0; i < model.getSize(); i += 1) {
			if (res.length() > 0)
				res.append(DIV_CHAR);
			res.append(model.getElementAt(i).toString());
		}
		return res.toString();
	}

	/**
	 * Carregar na combo todas as entradas vindas da lista consolidada. O formato
	 * do parmetro  o mesmo que  retornado por {@link #getLRUJoinedText()}.
	 * Este mtodo apaga todas as atuais entradas da combo antes de inserir o novo
	 * contedo.
	 * 
	 * @param joinedText Texto consolidado para carregar a combo.
	 */
	public void loadLRUJoinedText(String joinedText) {
		model.removeAllElements();
		if (joinedText == null || joinedText.isEmpty())
			return;

		String parts[] = joinedText.split(Pattern.quote(new String(new char[] { DIV_CHAR })));
		for (String p : parts)
			model.addElement(p);
	}

	/**
	 * Quando o usurio clica com o boto direito e clica em "Limpar Lista", a
	 * combobox  esvaziada, porm o seu novo estado no  persistido.
	 * 
	 * <p>
	 * O "SaveHandler" contm o cdigo que persiste o contedo da LRU.
	 * 
	 * @return Instncia de Runnable que persiste o contedo da LRU.
	 */
	public Runnable getSaveHandler() {
		return saveHandler;
	}

	/**
	 * Quando o usurio clica com o boto direito e clica em "Limpar Lista", a
	 * combobox  esvaziada, porm o seu novo estado no  persistido.
	 * 
	 * <p>
	 * O "SaveHandler" contm o cdigo que persiste o contedo da LRU.
	 * 
	 * @param saveHandler Cdigo que persiste o contedo da LRU
	 */
	public void setSaveHandler(Runnable saveHandler) {
		this.saveHandler = saveHandler;
	}

}
