/**
 * 
 */
package tecgraf.openbus.browser;

import java.awt.BorderLayout;
import java.awt.Component;
import java.awt.Dimension;
import java.awt.event.WindowAdapter;
import java.awt.event.WindowEvent;
import java.io.IOException;
import java.net.InetAddress;
import java.net.UnknownHostException;
import java.util.Properties;
import java.util.logging.LogManager;
import java.util.prefs.BackingStoreException;
import java.util.prefs.Preferences;

import javax.swing.JFrame;
import javax.swing.JSplitPane;
import javax.swing.JTabbedPane;

import org.eclipse.wb.swing.FocusTraversalOnArray;
import org.omg.CORBA.ORB;
import org.omg.CORBA.ORBPackage.InvalidName;

import tecgraf.openbus.OpenBusContext;
import tecgraf.openbus.browser.scs_offers.OfferRegistryPanel;
import tecgraf.openbus.core.ORBInitializer;

/**
 * <h1>Tela inicial, principal. Aonde tudo comea.</h1>
 * 
 * Aqui est o main da aplicao standalone.
 * 
 * Aqui o Openbus  inicializado para que todas as demais clases do sistema
 * possa usufruir.
 * 
 * @author Daltro Gama (daltro@tecgraf.puc-rio.br)
 * 
 */
@SuppressWarnings("serial")
public class OpenbusBrowser extends JFrame {

	/**
	 * Painel de autenticao, que fica na parte superior da tela.
	 */
	private final AuthPanel authPanel;

	/**
	 * Esta aplicao trabalha com UM s ORB. No caso de vrias conexes, todas
	 * usaram o mesmo ORB.
	 */
	private final ORB orb;

	/**
	 * Referncia singleton para esta instncia da tela principal.
	 */
	private static OpenbusBrowser singletonInstance;

	/**
	 * Referncia para a implementao do pool de conexes, que gerencia recursos.
	 */
	private final ManagedConnectionPool managedConnectionPool;

	/**
	 * Preferncias persistidas na estao de trabalho do usurio referentes 
	 * esta tela.
	 */
	private final Preferences persistedPrefs = Preferences.userNodeForPackage(getClass());

	/**
	 * Construtor principal (da tela principal!)
	 * 
	 * @param args Parmetros aqui sero passados diretamente para o
	 *          {@link ORBInitializer#initORB(String[])}
	 */
	private OpenbusBrowser(String[] args) {

		Properties jacorbProps = new Properties();
		
		jacorbProps.put("jacorb.codeset", "on");
		jacorbProps.put("jacorb.retries", "1");
		jacorbProps.put("jacorb.dns.enable", "on");
		jacorbProps.put("jacorb.iiop.alternate_addresses", "auto");
		try {
			jacorbProps.put("jacorb.ior_proxy_host", InetAddress.getLocalHost().getHostName());
		}
		catch (UnknownHostException e1) {
			System.err.println("No foi possvel configurar jacorb.ior_proxy_host=java.net.InetAddress.getLocalHost().getHostName(): "+e1.getMessage());
			e1.printStackTrace(System.err);
		}
		
		orb = ORBInitializer.initORB(args, jacorbProps);
		
		final OpenBusContext ctx;
		try {
			ctx = (OpenBusContext) orb.resolve_initial_references("OpenBusContext");
		}
		catch (InvalidName e) {
			e.printStackTrace(System.err);
			throw new IllegalStateException(e);
		}

		setDefaultCloseOperation(EXIT_ON_CLOSE);
		setMinimumSize(new Dimension(640, 480));

		int width = persistedPrefs.getInt("width", 800);
		int height = persistedPrefs.getInt("height", 600);

		setSize(new Dimension(width, height));

		getContentPane().setLayout(new BorderLayout(0, 0));

		managedConnectionPool = new ManagedConnectionPool(ctx);

		JSplitPane splitPane = new JSplitPane();
		splitPane.setDividerLocation(160);
		splitPane.setOrientation(JSplitPane.VERTICAL_SPLIT);
		splitPane.setOneTouchExpandable(true);
		splitPane.setContinuousLayout(true);
		getContentPane().add(splitPane);

		authPanel = new AuthPanel(managedConnectionPool);
		splitPane.setLeftComponent(authPanel);

		JTabbedPane tabbedPane = new JTabbedPane(JTabbedPane.TOP);
		splitPane.setRightComponent(tabbedPane);
		tabbedPane.addTab("OfferRegistry", new OfferRegistryPanel(managedConnectionPool));
		//tabbedPane.addTab("Logins", new LoginsPanel(managedConnectionPool));
		//tabbedPane.setEnabledAt(1, false);

		getContentPane().setFocusTraversalPolicy(new FocusTraversalOnArray(new Component[] { splitPane }));

		Thread orbThread = new Thread("ORB Thread") {
			@Override
			public void run() {
				orb.run();
			};
		};
		orbThread.setDaemon(true);
		orbThread.start();

		Properties versionProps = new Properties();
		try {
			versionProps.load(getClass().getResourceAsStream("version.properties"));
		}
		catch (IOException e) {
			e.printStackTrace(System.err);
		}

		String version = versionProps.getProperty("version", "v. ?");
		String buildDate = versionProps.getProperty("revision", "SNAPSHOT");
		String titleAppend = version;
		if (version.contains("SNAPSHOT")) {
			titleAppend = titleAppend.replaceAll("\\-SNAPSHOT", " revision " + buildDate.replaceAll("\\$", "\\\\\\$"));
		}
		setTitle("Openbus Browser - " + titleAppend 
			+ " - SDK " + versionProps.getProperty("openbus-sdk-version", "2.?"));

		addWindowListener(new WindowAdapter() {
			@Override
			public void windowClosing(WindowEvent e) {
				persistedPrefs.putInt("height", getHeight());
				persistedPrefs.putInt("width", getWidth());
				setVisible(false);
				managedConnectionPool.shutdown();
			}
		});

		Runtime.getRuntime().addShutdownHook(new Thread() {
			@Override
			public void run() {
				managedConnectionPool.shutdown();
			}
		});

		setLocationRelativeTo(null);

	}

	/**
	 * Recuperar instncia singleton da tela principal da aplicao.
	 * 
	 * @return Instncia da tela principal da aplicao.
	 */
	public static OpenbusBrowser getSingletonInstance() {
		return singletonInstance;
	}

	/**
	 * MAIN!! :-)
	 * 
	 * @param args Parmetros aqui sero passados diretamente para o
	 *          {@link ORBInitializer#initORB(String[])}
	 * @throws SecurityException
	 * @throws IOException
	 */
	public static void main(String[] args) throws SecurityException, IOException {

		LogManager.getLogManager().readConfiguration(OpenbusBrowser.class.getResource("/logging.properties").openStream());

		// Para migrao transparente de usurios que j usaram a verso anterior, e
		// possuem preferncias salvas
		//TODO: Apagar este procedimento em releases futuras.
		migratePrefs("ep/petrobras/openbusmonitor", "tecgraf/openbus/browser");
		migratePrefs("ep/petrobras/openbusmonitor/scs_offers", "tecgraf/openbus/browser/scs_offers");

		singletonInstance = new OpenbusBrowser(args);
		Thread.setDefaultUncaughtExceptionHandler(new SwingUncaughtExceptionHandler(singletonInstance));

		singletonInstance.setVisible(true);
	}

	/**
	 * Migrar preferncias do pacote antigo (poca do antigo OpenbusMonitor) para
	 * o pacote novo, de forma que o usurio no perca suas preferncias.
	 * 
	 * @todo Apagar este mtodo em uma verso futura.
	 * @param fromNode
	 * @param toNode
	 */
	private static final void migratePrefs(String fromNode, String toNode) {
		try {
			if (Preferences.userRoot().nodeExists(fromNode)) {
				Preferences fromPrefs = Preferences.userRoot().node(fromNode);
				Preferences toPrefs = Preferences.userRoot().node(toNode);
				for (String key : fromPrefs.keys()) {
					String val = fromPrefs.get(key, null);
					if (val != null)
						toPrefs.put(key, val);
				}
				fromPrefs.removeNode();
				toPrefs.flush();
			}
		}
		catch (BackingStoreException e) {
			e.printStackTrace(System.err);
		}
	}

}
