package tecgraf.openbus.browser;

import java.awt.Dimension;
import java.awt.Font;
import java.awt.event.ActionEvent;
import java.awt.event.ActionListener;
import java.awt.event.WindowAdapter;
import java.awt.event.WindowEvent;
import java.lang.reflect.InvocationTargetException;
import java.text.NumberFormat;

import javax.swing.ImageIcon;
import javax.swing.JButton;
import javax.swing.JDialog;
import javax.swing.JLabel;
import javax.swing.SwingUtilities;

import tecgraf.openbus.browser.scs_offers.OfferRegistryPanel;

import net.miginfocom.swing.MigLayout;

/**
 * Utilitrio para execuo de cdigos demorados, que fornece feedback visual
 * para o usurio, com possibilidade de cancelamento.
 * 
 * <p>
 * O cancelamento  implementado como o simples envio de um
 * {@link Thread#interrupt()}.
 * 
 * @author Daltro Gama (daltro@tecgraf.puc-rio.br)
 */
@SuppressWarnings("serial")
public final class SlowRequester extends JDialog {

	private static final int DIALOG_SHOW_DELAY = 2000;

	public static abstract class SlowRunnable {
		public abstract void run() throws Exception;

		private volatile String progress = "Executando processamento...";
		private volatile String title = "Aguarde";
		private volatile Throwable exception = null;

		protected void setProgress(String progress) {
			if (progress == null)
				progress = "";
			this.progress = progress;
		}

		protected void setTitle(String title) {
			if (title == null)
				title = "";
			this.title = title;
		}

		public String getProgress() {
			return progress;
		}

		public String getTitle() {
			return title;
		}

		protected Throwable getException() {
			return exception;
		}

		protected void setException(Throwable exception) {
			this.exception = exception;
		}

		public void finish() {
			if (getException() != null)
				Thread.getDefaultUncaughtExceptionHandler()
				  .uncaughtException(Thread.currentThread(), exception);
		}

	}

	private static final ImageIcon IMG_TIME = new ImageIcon(
	  OfferRegistryPanel.class.getResource("time-32.png")
	  );

	private static final long SECOND = 1000;
	private static final long MINUTE = SECOND * 60;
	private static final long HOUR = MINUTE * 60;
	private static final NumberFormat fmtNum;

	static {
		fmtNum = NumberFormat.getIntegerInstance();
		fmtNum.setMinimumFractionDigits(0);
		fmtNum.setMaximumFractionDigits(0);
		fmtNum.setMinimumIntegerDigits(2);
		fmtNum.setMaximumIntegerDigits(3);
	}

	private final long initTimer;
	private final Thread executorThread;
	private Thread animatorThread = null;
	private final SlowRunnable runnable;
	private final JLabel lblTime;
	private final JLabel lblProgress;

	private class AnimatorThread extends Thread {

		@Override
		public void run() {
			try {
				while (!Thread.currentThread().isInterrupted()) {
					Thread.sleep(100);
					if (!isVisible())
						continue;

					final String time = getTime();
					final String title = runnable.getTitle();
					final String progress = runnable.getProgress();
					if (!getTitle().equals(title)
					  || !lblProgress.getText().equals(progress)
					  || !lblTime.getText().equals(time)) {
						try {
							SwingUtilities.invokeAndWait(new Runnable() {
								@Override
								public void run() {
									if (!getTitle().equals(title))
										setTitle(title);
									if (!lblProgress.getText().equals(progress))
										lblProgress.setText(progress);
									if (!lblTime.getText().equals(time))
										lblTime.setText(time);
								}
							});
						}
						catch (InvocationTargetException e) {
							e.printStackTrace(System.err);
							break;
						}
					}
				}
			}
			catch (InterruptedException ie) {
			}
		}

	}

	/**
	 * Instancia o dilogo com o cronmetro e o boto de cancelar.
	 * 
	 * O dilogo s  instanciado e exibido ao usurio caso o cdigo asncrono
	 * demore mais do que #DIALOG_SHOW_DELAY milisegundos, evitando overhead
	 * desnecessrio no Swing para execues rpidas.
	 * 
	 * @param initTimer Timestamp marcador de incio da execuo, para o
	 *          cronmetro.
	 * @param executorThread Instncia da thread que executa o cdigo.
	 * @param runnable Cdigo a ser executado.
	 */
	private SlowRequester(long initTimer, Thread executorThread, SlowRunnable runnable) {
		super(OpenbusBrowser.getSingletonInstance(), ModalityType.MODELESS);
		setTitle("Aguarde");
		setLocationRelativeTo(getOwner());
		this.initTimer = initTimer;
		this.executorThread = executorThread;
		this.runnable = runnable;

		setMinimumSize(new Dimension(300, 150));
		setResizable(true);
		getContentPane().setLayout(new MigLayout("", "[grow]", "[grow][][][][grow]"));

		lblProgress = new JLabel("Executando processamento lento");
		lblProgress.setMinimumSize(new Dimension(0, 0));
		getContentPane().add(lblProgress, "cell 0 1,alignx center");

		lblTime = new JLabel("00:00:00.0", IMG_TIME, JLabel.CENTER);
		lblTime.setMinimumSize(new Dimension(0, 0));
		lblTime.setFont(new Font(Font.MONOSPACED, Font.BOLD, 20));
		getContentPane().add(lblTime, "cell 0 2,alignx center");

		JButton btnCancell = new JButton("Interromper");
		btnCancell.setToolTipText("Dar sinal interrupt() na thread do processamento.");
		getContentPane().add(btnCancell, "cell 0 3,alignx center");
		btnCancell.addActionListener(new ActionListener() {
			@Override
			public void actionPerformed(ActionEvent e) {
				SlowRequester.this.executorThread.interrupt();
			}
		});

		setDefaultCloseOperation(DO_NOTHING_ON_CLOSE);

		addWindowListener(new WindowAdapter() {
			@Override
			public void windowOpened(WindowEvent e) {
				closeAnimatorThread();
				animatorThread = new AnimatorThread();
				animatorThread.setDaemon(true);
				animatorThread.setPriority(Thread.MIN_PRIORITY);
				animatorThread.start();
			}

			private void closeAnimatorThread() {
				if (animatorThread != null) {
					if (animatorThread.isAlive())
						animatorThread.interrupt();
				}
				animatorThread = null;
			}

			@Override
			public void windowClosed(WindowEvent e) {
				closeAnimatorThread();
			}
		});

		pack();

	}

	/**
	 * Construir texto a ser exibido no cronmetro, usando o {@link #initTimer} e
	 * o {@link System#currentTimeMillis()} como referncia.
	 * 
	 * @return Texto a ser exibido no cronmetro.
	 */
	private String getTime() {
		StringBuilder res = new StringBuilder();

		long time = System.currentTimeMillis() - initTimer;
		int hour = (int) (time / HOUR);
		time -= (hour * HOUR);

		int minute = (int) (time / MINUTE);
		time -= (minute * MINUTE);

		int second = (int) (time / SECOND);
		time -= (second * SECOND);

		int deciSecond = (int) (time / 100);

		res.append(fmtNum.format(hour));
		res.append(":");
		res.append(fmtNum.format(minute));
		res.append(":");
		res.append(fmtNum.format(second));
		res.append(".");
		res.append(deciSecond);

		return res.toString();
	}

	/**
	 * Executar cdigo possivelmente demorado, que caso demore, um dilogo ser
	 * exibido para o usurio saber que se trata de algo lento e com a
	 * possibilidade de cancelamento.
	 * 
	 * O cancelamento  feito atravs de {@link Thread#interrupt()}. Portanto,
	 * caso o cdigo enviado no trate adequadamente a interrupo, o cancelamento
	 * da tarefa poder ser mal sucedido.
	 * 
	 * @param r Cdigo a ser executado.
	 * @param async true para que no seja bloqueante.
	 */
	public static final void run(final SlowRunnable r, final boolean async) {

		final Thread executorThread = new Thread() {
			@Override
			public void run() {
				try {
					r.setException(null);
					r.run();
				}
				catch (Throwable e) {
					if (e instanceof InterruptedException) {
						return;
					}
					r.setException(e);
				}
			}
		};

		final Runnable auxRunnable = new Runnable() {
			@Override
			public void run() {

				executorThread.setDaemon(true);
				long timer = System.currentTimeMillis();
				final SlowRequester dialog[] = new SlowRequester[1];

				executorThread.start();
				try {
					while (executorThread.isAlive()) {
						try {
							Thread.sleep(5);
						}
						catch (InterruptedException e) {
							e.printStackTrace(System.err);
							break;
						}
						long passedTime = System.currentTimeMillis() - timer;
						if (passedTime > DIALOG_SHOW_DELAY && dialog[0] == null) {
							dialog[0] = new SlowRequester(timer, executorThread, r);
							SwingUtilities.invokeLater(new Runnable() {
								@Override
								public void run() {
									dialog[0].setVisible(true);
								}
							});
						}
					}
				}
				finally {
					if (dialog[0] != null) {
						SwingUtilities.invokeLater(new Runnable() {
							@Override
							public void run() {
								dialog[0].setVisible(false);
								dialog[0].dispose();
							}
						});
					}
					r.finish();
				}

			}
		};

		if (async)
			new Thread(auxRunnable).start();
		else
			auxRunnable.run();

	}

}
