package tecgraf.openbus.browser.scs_offers;

import java.awt.Dimension;
import java.awt.Font;
import java.lang.reflect.InvocationTargetException;
import java.text.NumberFormat;

import javax.swing.ImageIcon;
import javax.swing.JLabel;
import javax.swing.JPanel;
import javax.swing.SwingUtilities;
import javax.swing.event.AncestorEvent;
import javax.swing.event.AncestorListener;

import net.miginfocom.swing.MigLayout;

/**
 * Painel de detalhes que ser exibido para o usurio do lado direito
 * da rvore de resultados quando ele clicar sobre um n do tipo
 * {@link AsyncExpandableLoadingNodeBean}, que representa um carregamento
 * de filhos ainda em andamento.
 * <p>
 * O painel exibe basicamente um cronmetro para o tempo do carregamento
 * e a mensagem que o n informou atravs de 
 * {@link AsyncExpandableTreeNode#getLoadingDescription()}.
 * 
 * @author Daltro Gama (daltro@tecgraf.puc-rio.br)
 */
@SuppressWarnings("serial")
final class AsyncExpandableLoadingDetailsPanel extends JPanel {

	private static final ImageIcon IMG_TIME = new ImageIcon(
	  AsyncExpandableLoadingDetailsPanel.class.getResource("time-32.png")
	  );

	private static final long SECOND = 1000;
	private static final long MINUTE = SECOND * 60;
	private static final long HOUR = MINUTE * 60;
	private static final NumberFormat fmtNum;

	private static volatile AsyncExpandableLoadingDetailsPanel animedInstance = null;

	private static final class CounterThread extends Thread {
		@Override
		public void run() {
			try {
				while (!isInterrupted()) {
					while (true) {
						sleep(100);
						final AsyncExpandableLoadingDetailsPanel panel = animedInstance;
						if (panel == null)
							continue;
						if (!panel.isVisible())
							continue;

						try {
							SwingUtilities.invokeAndWait(new Runnable() {
								@Override
								public void run() {
									panel.updateTime();
								}
							});
						}
						catch (InvocationTargetException e) {
							e.printStackTrace(System.err);
						}
					}
				}
			}
			catch (InterruptedException ie) {
			}
		}
	};

	static {
		fmtNum = NumberFormat.getIntegerInstance();
		fmtNum.setMinimumFractionDigits(0);
		fmtNum.setMaximumFractionDigits(0);
		fmtNum.setMinimumIntegerDigits(2);
		fmtNum.setMaximumIntegerDigits(3);
		CounterThread thread = new CounterThread();
		thread.setDaemon(true);
		thread.setPriority(Thread.MIN_PRIORITY);
		thread.start();
	}

	private final AsyncExpandableLoadingNodeBean bean;

	private final JLabel lblTempo;

	public AsyncExpandableLoadingDetailsPanel(AsyncExpandableLoadingNodeBean argBean) {
		this.bean = argBean;
		setLayout(new MigLayout("", "[grow]", "[grow][][][grow]"));

		JLabel lblSatus = new JLabel(bean.getDescription());
		lblSatus.setMinimumSize(new Dimension(0, 0));
		add(lblSatus, "cell 0 1,alignx center");

		lblTempo = new JLabel("00:00:00.0", IMG_TIME, JLabel.CENTER);
		lblTempo.setMinimumSize(new Dimension(0, 0));
		lblTempo.setFont(new Font(Font.MONOSPACED, Font.BOLD, 20));
		add(lblTempo, "cell 0 2,alignx center");

		addAncestorListener(new AncestorListener() {

			@Override
			public void ancestorRemoved(AncestorEvent event) {
				animedInstance = null;
			}

			@Override
			public void ancestorMoved(AncestorEvent event) {
			}

			@Override
			public void ancestorAdded(AncestorEvent event) {
				animedInstance = AsyncExpandableLoadingDetailsPanel.this;
			}
		});

	}

	private void updateTime() {
		StringBuilder res = new StringBuilder();

		long time = System.currentTimeMillis() - bean.getStartTimestamp();
		int hour = (int) (time / HOUR);
		time -= (hour * HOUR);

		int minute = (int) (time / MINUTE);
		time -= (minute * MINUTE);

		int second = (int) (time / SECOND);
		time -= (second * SECOND);

		int deciSecond = (int) (time / 100);

		res.append(fmtNum.format(hour));
		res.append(":");
		res.append(fmtNum.format(minute));
		res.append(":");
		res.append(fmtNum.format(second));
		res.append(".");
		res.append(deciSecond);

		String newVal = res.toString();
		if (!lblTempo.getText().equals(newVal))
			lblTempo.setText(newVal);
	}

}
