package tecgraf.openbus.browser.scs_offers;

import java.util.Enumeration;
import java.util.List;

import javax.swing.SwingUtilities;
import javax.swing.tree.DefaultMutableTreeNode;
import javax.swing.tree.DefaultTreeModel;
import javax.swing.tree.TreeNode;

import tecgraf.openbus.browser.scs_offers.basic_nodes.ErrorNodeBean;


/**
 * Abstrao para um tipo especial de n da rvore cujos filhos s so carregados
 * na primeira vez em que o n  expandido.
 * <p>
 * O carregamento  feito de forma assncrona, incluindo no lugar do n um 
 * n temporrio que ilustra a espera pelo carregamento em questo.
 * <p>
 * Ns que estenderem deste devero implementar apenas o cdigo que carrega
 * os filhos ({@link #loadChildren()}), e um mtodo para dar um "ttulo" para
 * a tela de "aguarde" para o usurio, para o caso de ele clicar no n enquanto
 * o carregamento  feito ({@link #getLoadingDescription()}.
 * <p>
 * Ns deste tipo implicitamente aceitam a tecla de atalho F5 para recarregar
 * os ns filhos.
 * 
 * @author Daltro Gama (daltro@tecgraf.puc-rio.br)
 */
@SuppressWarnings("serial")
public abstract class AsyncExpandableTreeNode extends NodeWithTreeReference implements RefreshableNode {

	/**
	 * Estados admissveis pelo comportamento de carregamento assncrono.
	 */
	public enum LoadingState {
		UNLOADED,
		LOADING,
		LOADED,
		ERROR
	}

	private LoadingState loadingState = LoadingState.UNLOADED;
	protected final DefaultTreeModel myModel;

	/**
	 * Carregar os ns filhos deste. Este mtodo  executado em uma thread
	 * prpria, no interferindo com a performance do sistema. A demora na
	 * execuo deste mtodo ser informada para o usurio em interface prpria
	 * para tal.
	 * @return Lista dos filhos do n.
	 * @throws Exception
	 */
	protected abstract List<DefaultMutableTreeNode> loadChildren() throws Exception;

	/**
	 * Caso o usurio clique no n "aguardando...", o cronmetro do carregamento
	 * ser exibido junto com a mensagem retornada por este mtodo.
	 * @return Mensagem a ser exibida para o usurio.
	 */
	protected abstract String getLoadingDescription();

	private final synchronized void assertLoaded() {
		if (loadingState == LoadingState.UNLOADED) {
			doLoad();
		}
	}

	public AsyncExpandableTreeNode(SCSTree tree) {
		super(tree);
		this.myModel = (DefaultTreeModel) tree.getModel();
	}

	@Override
	public void refreshNode() {
		Runnable refresh = new Runnable() {
			@Override
			public void run() {
				assertLoaded();
				if (loadingState != LoadingState.LOADING) {
					removeAllChildren();
					loadingState = LoadingState.UNLOADED;
					doLoad();
				}
			}
		};

		if (SwingUtilities.isEventDispatchThread())
			refresh.run();
		else
			SwingUtilities.invokeLater(refresh);

	}

	private synchronized void doLoad() {
		loadingState = LoadingState.LOADING;
		final AsyncExpandableLoadingNodeBean loadingBean = new AsyncExpandableLoadingNodeBean(
		  System.currentTimeMillis(),
		  getLoadingDescription());
		SwingUtilities.invokeLater(new Runnable() {
			@Override
			public void run() {
				setOnlyChild(new DefaultMutableTreeNode(loadingBean));
			}
		});
		new Thread() {
			@Override
			public void run() {
				try {
					loadingBean.setStartTimestamp(System.currentTimeMillis());
					final List<DefaultMutableTreeNode> res = loadChildren();
					SwingUtilities.invokeLater(new Runnable() {
						@Override
						public void run() {
							try{
								removeAllChildren();
								for (DefaultMutableTreeNode node : res)
									add(node);
								myModel.nodeStructureChanged(AsyncExpandableTreeNode.this);
							}catch(Throwable ex){
								ex.printStackTrace(System.err);
							}
						}
					});
					loadingState = LoadingState.LOADED;
				}
				catch (final Throwable e) {
					SwingUtilities.invokeLater(new Runnable() {
						@Override
						public void run() {
							try{
								setOnlyChild(new DefaultMutableTreeNode(new ErrorNodeBean(e)));
							}catch(Throwable ex){
								ex.printStackTrace(System.err);
							}
						}
					});
					loadingState = LoadingState.ERROR;
				}
			};
		}.start();
	}

	private void setOnlyChild(DefaultMutableTreeNode child) {
		removeAllChildren();
		add(child);
		myModel.nodeStructureChanged(this);
	}

	@Override
	public TreeNode getChildAt(int childIndex) {
		assertLoaded();
		return super.getChildAt(childIndex);
	}

	@Override
	public int getChildCount() {
		assertLoaded();
		return super.getChildCount();
	}

	@Override
	public int getIndex(TreeNode node) {
		assertLoaded();
		return super.getIndex(node);
	}

	@Override
	public boolean getAllowsChildren() {
		return true;
	}

	@Override
	public boolean isLeaf() {
		return loadingState == LoadingState.LOADED && children.isEmpty();
	}

	@Override
	public Enumeration<?> children() {
		assertLoaded();
		return super.children();
	}

}
