package tecgraf.openbus.browser.scs_offers;

import java.awt.BorderLayout;
import java.awt.Cursor;
import java.awt.Dimension;
import java.awt.event.ActionEvent;
import java.awt.event.ActionListener;
import java.awt.event.ItemEvent;
import java.awt.event.ItemListener;
import java.awt.event.KeyAdapter;
import java.awt.event.KeyEvent;
import java.awt.event.MouseAdapter;
import java.awt.event.MouseEvent;
import java.util.ArrayList;
import java.util.Collections;
import java.util.Comparator;
import java.util.Observable;
import java.util.Observer;
import java.util.prefs.Preferences;
import java.util.regex.Pattern;

import javax.swing.ButtonGroup;
import javax.swing.DefaultCellEditor;
import javax.swing.DefaultComboBoxModel;
import javax.swing.ImageIcon;
import javax.swing.JButton;
import javax.swing.JCheckBoxMenuItem;
import javax.swing.JLabel;
import javax.swing.JOptionPane;
import javax.swing.JPanel;
import javax.swing.JPopupMenu;
import javax.swing.JRadioButtonMenuItem;
import javax.swing.JScrollPane;
import javax.swing.JSplitPane;
import javax.swing.JTable;
import javax.swing.JTextField;
import javax.swing.ListSelectionModel;
import javax.swing.SwingUtilities;
import javax.swing.border.TitledBorder;
import javax.swing.event.TableModelEvent;
import javax.swing.event.TableModelListener;
import javax.swing.event.TreeSelectionEvent;
import javax.swing.event.TreeSelectionListener;
import javax.swing.table.DefaultTableModel;
import javax.swing.tree.DefaultMutableTreeNode;
import javax.swing.tree.TreePath;

import net.miginfocom.swing.MigLayout;
import tecgraf.openbus.browser.ConnectionProvider;
import tecgraf.openbus.browser.LRUComboBox;
import tecgraf.openbus.browser.ManagedConnection;
import tecgraf.openbus.browser.OpenbusBrowser;
import tecgraf.openbus.browser.TipPanel;
import tecgraf.openbus.browser.TipPanelInterface;
import tecgraf.openbus.browser.ManagedConnection.ConnectionState;
import tecgraf.openbus.core.v2_0.services.offer_registry.ServiceProperty;

/**
 * Painel pai para a interface que faz a busca no servio de registro.
 * 
 * Este vai englobar o painel onde o usurio define o critrio da busca, e
 * tambm o painel com a rvore dos componentes, verses e detalhes.
 * 
 * @author Daltro Gama (daltro@tecgraf.puc-rio.br)
 */
@SuppressWarnings("serial")
public final class OfferRegistryPanel extends JPanel implements FindServicesQueryControlInterface {

	private static final char LRU_PROP_DIV_CHAR = '';
	private static final ImageIcon IMAGE_TIP_HIDE = new ImageIcon(OpenbusBrowser.class.getResource("tip-hide.png"));
	private static final ImageIcon IMAGE_TIP_SHOW = new ImageIcon(OpenbusBrowser.class.getResource("tip-open.png"));

	private static final ImageIcon IMAGE_TREE_EXPAND = new ImageIcon(
	  OfferRegistryPanel.class.getResource("tree_expand.png"));
	private static final ImageIcon IMAGE_TREE_CONTRACT = new ImageIcon(
	  OfferRegistryPanel.class.getResource("tree_contract.png"));
	private static final ImageIcon IMAGE_TREE_SEARCH = new ImageIcon(
	  OfferRegistryPanel.class.getResource("tree_search.png"));

	private final ConnectionProvider connectionProvider;
	private final JTable propertyTable;
	private static final String columnNames[] = new String[] {
	    "Propriedade", "Valor"
	};

	private final DefaultTableModel propertyTableModel = new DefaultTableModel(
	  new Object[][] {
	      new Object[] { "", "" }
	  },
	  columnNames
	  ) {
		  Class<?>[] columnTypes = new Class[] {
		      String.class, String.class
		  };

		  @Override
		  public Class<?> getColumnClass(int columnIndex) {
			  return columnTypes[columnIndex];
		  }

		  @Override
		  public boolean isCellEditable(int row, int column) {
			  return true;
		  };
	  };

	private volatile boolean cmbPropertyNameResizing = false;
	private final Preferences persistedPrefs = Preferences.userNodeForPackage(getClass());

	private final LRUComboBox cmbPropertyName;
	private final JButton btnFindOffers;
	private final LRUComboBox cmbLRU;
	private JLabel lblFilter;
	private ManagedConnection currentConnectionUsed = null;
	private boolean filterIsShowingFilterText = true;
	private final SCSTreeAndDetailsComponent scsTreeAndDetails;

	private TipPanel tipPanel;
	private JButton btnClearFilter;

	private JLabel tipHiderLabel;
	private DefaultCellEditor propCellEditor;

	public OfferRegistryPanel(ConnectionProvider argConnectionProvider) {
		super();
		this.connectionProvider = argConnectionProvider;
		setLayout(new MigLayout("", "0[grow]0", "0[grow]0"));
		ServiceOfferRegistryObserversPool.getSingleton().setConnectionProvider(argConnectionProvider);

		final JSplitPane mainSplit = new JSplitPane(JSplitPane.VERTICAL_SPLIT);
		mainSplit.setContinuousLayout(true);
		mainSplit.setOneTouchExpandable(true);
		add(mainSplit, "cell 0 0, grow");

		JPanel pnlFindOffers = new JPanel();
		pnlFindOffers.setBorder(new TitledBorder(null, "FindOffers:", TitledBorder.LEADING, TitledBorder.TOP, null, null));
		pnlFindOffers.setLayout(new MigLayout("", "0[grow]0", "0[grow][]0"));

		JScrollPane scrPropertyTable = new JScrollPane();
		pnlFindOffers.add(scrPropertyTable, "cell 0 0,alignx left,aligny top, grow");

		mainSplit.setTopComponent(pnlFindOffers);
		mainSplit.setDividerLocation(100);

		cmbPropertyName = new LRUComboBox();
		cmbPropertyName.setEditable(true);
		cmbPropertyName.addItemListener(new ItemListener() {
			@Override
			public void itemStateChanged(ItemEvent e) {
				if (cmbPropertyName.getSelectedItem() == null)
					return;
				if (cmbPropertyName.getSelectedItem().toString().trim().isEmpty())
					return;

				DefaultComboBoxModel model = (DefaultComboBoxModel) cmbPropertyName.getModel();
				int pos = model.getIndexOf(cmbPropertyName.getSelectedItem());
				if (pos == -1) {
					cmbPropertyName.addLRUItem(cmbPropertyName.getSelectedItem().toString());
					persistedPrefs.put("propLRU", cmbPropertyName.getLRUJoinedText());
				}
			}
		});
		cmbPropertyName.setSaveHandler(new Runnable() {
			@Override
			public void run() {
				persistedPrefs.put("propLRU", cmbPropertyName.getLRUJoinedText());
			}
		});

		cmbPropertyName.loadLRUJoinedText(persistedPrefs.get("propLRU", ""));
		for (String v : new String[] {
		    "openbus.offer.id",
		    "openbus.offer.login",
		    "openbus.offer.entity",
		    "openbus.offer.timestamp",
		    "openbus.offer.year",
		    "openbus.offer.month",
		    "openbus.offer.day",
		    "openbus.offer.hour",
		    "openbus.offer.minute",
		    "openbus.offer.second",
		    "openbus.component.name",
		    "openbus.component.version.major",
		    "openbus.component.version.minor",
		    "openbus.component.version.patch",
		    "openbus.component.facet",
		    "openbus.component.interface"
		}) {
			cmbPropertyName.addLRUItem(v);
		}

		propertyTable = new JTable();
		propertyTable.setModel(propertyTableModel);
		propertyTable.setSurrendersFocusOnKeystroke(true);

		propertyTableModel.addTableModelListener(new TableModelListener() {
			@Override
			public void tableChanged(TableModelEvent e) {
				reinferRowNumbers();
				String lruText = getLRUText();
				int posInLRU;
				if (lruText != null && !lruText.trim().isEmpty())
					posInLRU = ((DefaultComboBoxModel) cmbLRU.getModel()).getIndexOf(lruText);
				else
					posInLRU = -1;
				if (posInLRU != cmbLRU.getSelectedIndex())
					cmbLRU.setSelectedIndex(posInLRU);
			}
		});
		propertyTable.addKeyListener(new KeyAdapter() {
			@Override
			public void keyPressed(KeyEvent e) {
				if (!propertyTable.isEditing() && e.getKeyCode() == KeyEvent.VK_DELETE) {
					e.consume();
					cmbPropertyNameResizing = true;
					try {
						for (int col : propertyTable.getSelectedColumns()) {
							for (int row : propertyTable.getSelectedRows()) {
								propertyTableModel.setValueAt("", row, col);
							}
						}
					}
					finally {
						cmbPropertyNameResizing = false;
						reinferRowNumbers();
					}
				}
			}
		});

		propCellEditor = new DefaultCellEditor(cmbPropertyName);
		configurePropColumnModel();
		propertyTable.setCellSelectionEnabled(true);
		propertyTable.setSelectionMode(ListSelectionModel.SINGLE_SELECTION);

		scrPropertyTable.setViewportView(propertyTable);

		JLabel lblLRU = new JLabel("LRU:");
		pnlFindOffers.add(lblLRU, "cell 0 1,alignx left");

		cmbLRU = new LRUComboBox();
		cmbLRU.setMaximumSize(new Dimension(Integer.MAX_VALUE, 20));
		cmbLRU.setMinimumSize(new Dimension(0, 20));
		lblLRU.setLabelFor(cmbLRU);
		pnlFindOffers.add(cmbLRU, "cell 0 1,alignx left, grow");
		cmbLRU.addItemListener(new ItemListener() {
			@Override
			public void itemStateChanged(ItemEvent e) {
				if (cmbLRU.getSelectedItem() == null)
					return;
				if (!getLRUText().equals(cmbLRU.getSelectedItem().toString())) {
					configureByLRUText(cmbLRU.getSelectedItem().toString());
				}
			}
		});

		btnClearFilter = new JButton("Limpar");
		btnClearFilter.setMaximumSize(new Dimension(Integer.MAX_VALUE, 20));
		pnlFindOffers.add(btnClearFilter, "cell 0 1,alignx");

		btnFindOffers = new JButton("Consultar");
		btnFindOffers.setMaximumSize(new Dimension(Integer.MAX_VALUE, 20));
		pnlFindOffers.add(btnFindOffers, "cell 0 1,alignx");

		scsTreeAndDetails = new SCSTreeAndDetailsComponent();
		scsTreeAndDetails.setFindServicesQueryControlInterface(this);

		JPanel resultPanel = new JPanel();
		resultPanel.setLayout(new BorderLayout());
		resultPanel.add(scsTreeAndDetails, BorderLayout.CENTER);

		scsTreeAndDetails.getTree().addTreeSelectionListener(new TreeSelectionListener() {
			@Override
			public void valueChanged(TreeSelectionEvent e) {
				TreePath sel = scsTreeAndDetails.getTree().getSelectionPath();
				NodeWithHelpTipInterface help = null;
				if (sel != null) {
					if (sel.getLastPathComponent() instanceof NodeWithHelpTipInterface) {
						help = (NodeWithHelpTipInterface) sel.getLastPathComponent();
					}
					else if (sel.getLastPathComponent() instanceof DefaultMutableTreeNode) {
						DefaultMutableTreeNode n = (DefaultMutableTreeNode) sel.getLastPathComponent();
						if (n.getUserObject() != null && n.getUserObject() instanceof NodeWithHelpTipInterface) {
							help = (NodeWithHelpTipInterface) n.getUserObject();
						}
					}
				}

				if (help != null)
					help.configureHelpTip(tipPanel);
				else
					configureDefaultTip(tipPanel);
			}
		});

		JPanel resultsFilterPanel = new JPanel();
		resultPanel.add(resultsFilterPanel, BorderLayout.NORTH);
		resultsFilterPanel.setLayout(new MigLayout("", "1[][]1", "1[]1"));

		JButton btnContract = new JButton(IMAGE_TREE_CONTRACT);
		btnContract.setToolTipText("Contrair todos os resultados da rvore abaixo.");
		btnContract.setMinimumSize(new Dimension(16, 16));
		btnContract.setMaximumSize(new Dimension(16, 16));
		btnContract.setPreferredSize(new Dimension(16, 16));
		btnContract.addActionListener(new ActionListener() {
			@Override
			public void actionPerformed(ActionEvent e) {
				for (int i = 0; i < scsTreeAndDetails.getTree().getRowCount(); i += 1)
					scsTreeAndDetails.getTree().collapseRow(i);
			}
		});
		resultsFilterPanel.add(btnContract, "cell 0 0");

		JButton btnExpand = new JButton(IMAGE_TREE_EXPAND);
		btnExpand.setToolTipText("Expandir todo o resultado at as ofertas.");
		btnExpand.setMinimumSize(new Dimension(16, 16));
		btnExpand.setMaximumSize(new Dimension(16, 16));
		btnExpand.setPreferredSize(new Dimension(16, 16));
		btnExpand.addActionListener(new ActionListener() {
			@Override
			public void actionPerformed(ActionEvent e) {
				for (int i = 0; i < scsTreeAndDetails.getTree().getRowCount(); i += 1) {
					if (scsTreeAndDetails.getTree().getPathForRow(i).getPathCount() < 4)
						scsTreeAndDetails.getTree().expandRow(i);
				}
			}
		});
		resultsFilterPanel.add(btnExpand, "cell 0 0");

		JButton btnTreeSearch = new JButton(IMAGE_TREE_SEARCH);
		btnTreeSearch
		  .setToolTipText("Localizar texto nas ofertas " +
		    "(voc pode clicar na rvore e comear a digitar, tambm). " +
		    "No busca em ns do DataService.");
		btnTreeSearch.setMinimumSize(new Dimension(16, 16));
		btnTreeSearch.setMaximumSize(new Dimension(16, 16));
		btnTreeSearch.setPreferredSize(new Dimension(16, 16));
		btnTreeSearch.addActionListener(new ActionListener() {
			@Override
			public void actionPerformed(ActionEvent e) {
				scsTreeAndDetails.showSearchField();
			}
		});
		resultsFilterPanel.add(btnTreeSearch, "cell 0 0");

		lblFilter = new JLabel("Faa uma consulta");
		lblFilter.setMinimumSize(new Dimension(0, 0));
		lblFilter.addMouseListener(new MouseAdapter() {
			@Override
			public void mouseClicked(MouseEvent e) {
				filterIsShowingFilterText = !filterIsShowingFilterText;
				setFilterTextAndTooltip();
			}
		});
		resultsFilterPanel.add(lblFilter, "cell 1 0");

		mainSplit.setBottomComponent(resultPanel);

		tipHiderLabel = new JLabel(IMAGE_TIP_SHOW);
		tipHiderLabel.setOpaque(true);
		tipHiderLabel.setAlignmentX(JLabel.LEFT_ALIGNMENT);
		tipHiderLabel.setVerticalAlignment(JLabel.CENTER);
		tipHiderLabel.setHorizontalAlignment(JLabel.LEFT);
		tipHiderLabel.setAlignmentY(JLabel.CENTER_ALIGNMENT);
		tipHiderLabel.setBackground(getBackground().darker());
		tipHiderLabel.setMinimumSize(new Dimension(0, 12));
		tipHiderLabel.setMaximumSize(new Dimension(Integer.MAX_VALUE, 12));
		tipHiderLabel.setCursor(new Cursor(Cursor.HAND_CURSOR));
		tipHiderLabel.addMouseListener(new MouseAdapter() {
			@Override
			public void mouseClicked(MouseEvent e) {
				if (SwingUtilities.isLeftMouseButton(e) && e.getClickCount() == 1) {
					e.consume();
					tipPanel.setVisible(!tipPanel.isVisible());
					configureTipVisibility();
					persistedPrefs.putBoolean("tipPanelVisible", tipPanel.isVisible());
					validate();
					repaint();
				}
			}
		});

		tipPanel = new TipPanel();
		tipPanel.setVisible(persistedPrefs.getBoolean("tipPanelVisible", true));
		scsTreeAndDetails.addPanelToBottomDetails(tipHiderLabel);
		scsTreeAndDetails.addPanelToBottomDetails(tipPanel);

		configureTipVisibility();

		btnClearFilter.addActionListener(new ActionListener() {
			@Override
			public void actionPerformed(ActionEvent e) {
				propertyTableModel.setDataVector(new Object[][] {
				  new Object[] { "", "" }
				}, columnNames);
				configurePropColumnModel();
			}
		});

		btnFindOffers.addActionListener(new ActionListener() {
			@Override
			public void actionPerformed(ActionEvent e) {

				if (OfferRegistryPanel.this.connectionProvider.getConnectionToUse() == null) {
					JOptionPane.showMessageDialog(OfferRegistryPanel.this, "Selecione um login!", "Sem login",
					  JOptionPane.ERROR_MESSAGE);
					return;
				}

				currentConnectionUsed = OfferRegistryPanel.this.connectionProvider.getConnectionToUse();

				ServiceProperty[] propertyFilter = getPropertiesFromTable();
				scsTreeAndDetails.setCurrentPropertyFilter(propertyFilter);
				setFilterTextAndTooltip();

				scsTreeAndDetails.startSearch(currentConnectionUsed, propertyFilter);

				String lru = getLRUText(propertyFilter);
				if (lru != null && !lru.trim().isEmpty()) {
					cmbLRU.addLRUItem(lru);
					cmbLRU.setSelectedItem(lru);
					persistedPrefs.put("lru", cmbLRU.getLRUJoinedText());
				}
			}
		});

		btnFindOffers.addMouseListener(new MouseAdapter() {
			@Override
			public void mousePressed(MouseEvent e) {
				if (e.isPopupTrigger()) {
					e.consume();
					JPopupMenu opt = new JPopupMenu();

					final ButtonGroup autoLoadTypeGroup = new ButtonGroup();

					final JRadioButtonMenuItem mnuNoAutoLoad
					= new JRadioButtonMenuItem("No fazer busca automaticamente na primeira conexo.");

					autoLoadTypeGroup.add(mnuNoAutoLoad);

					final JRadioButtonMenuItem mnuAllAutoLoad
					= new JRadioButtonMenuItem("Fazer busca sem filtro automaticamente na primeira conexo.");

					autoLoadTypeGroup.add(mnuAllAutoLoad);

					final JRadioButtonMenuItem mnuLastAutoLoad
					= new JRadioButtonMenuItem("Fazer busca com ltimo filtro automaticamente na primeira conexo.");

					autoLoadTypeGroup.add(mnuLastAutoLoad);

					String prefVal = persistedPrefs.get("autoLoad", "last");

					if ("off".equals(prefVal))
						mnuNoAutoLoad.setSelected(true);
					else if ("all".equals(prefVal))
						mnuAllAutoLoad.setSelected(true);
					else if ("last".equals(prefVal))
						mnuLastAutoLoad.setSelected(true);

					opt.add(mnuNoAutoLoad);
					opt.add(mnuAllAutoLoad);
					opt.add(mnuLastAutoLoad);

					mnuNoAutoLoad.addActionListener(new ActionListener() {
						@Override
						public void actionPerformed(ActionEvent e) {
							persistedPrefs.put("autoLoad", "off");
						}
					});
					mnuAllAutoLoad.addActionListener(new ActionListener() {
						@Override
						public void actionPerformed(ActionEvent e) {
							persistedPrefs.put("autoLoad", "all");
						}
					});
					mnuLastAutoLoad.addActionListener(new ActionListener() {
						@Override
						public void actionPerformed(ActionEvent e) {
							persistedPrefs.put("autoLoad", "last");
						}
					});

					JCheckBoxMenuItem mnuDefaultExpanded = new JCheckBoxMenuItem("Expandir os ns das ofertas na rvore");
					mnuDefaultExpanded.setSelected(persistedPrefs.getBoolean("isOffersDefaultExpanded", false));
					mnuDefaultExpanded.addActionListener(new ActionListener() {
						@Override
						public void actionPerformed(ActionEvent arg0) {
							persistedPrefs.putBoolean("isOffersDefaultExpanded", ((JCheckBoxMenuItem) arg0.getSource()).isSelected());
						}
					});

					opt.add(mnuDefaultExpanded);

					opt.show(btnFindOffers, 0, btnFindOffers.getHeight() + 2);
				}
			}
		});

		btnFindOffers.setToolTipText("Executar consulta. Clique com o boto direito para opes.");

		cmbLRU.setSaveHandler(new Runnable() {
			@Override
			public void run() {
				persistedPrefs.put("lru", cmbLRU.getLRUJoinedText());
			}
		});
		cmbLRU.loadLRUJoinedText(persistedPrefs.get("lru", ""));

		configureDefaultTip(tipPanel);

		//configureByLRUText("");

		argConnectionProvider.addObserver(new Observer() {
			@Override
			public void update(Observable o, Object arg) {
				String prefVal = persistedPrefs.get("autoLoad", "last");
				if (!"off".equals(prefVal)) {
					if (connectionProvider.getConnectionToUse() != null
					  && connectionProvider.getConnectionToUse().getState().equals(ConnectionState.AUTENTICATED)
					  && currentConnectionUsed == null) {

						if ("all".equals(prefVal)) {
							btnClearFilter.doClick();
						}
						else if ("last".equals(prefVal)) {
							if (cmbLRU.getModel().getSize() > 0) {
								cmbLRU.setSelectedIndex(0);
							}
						}

						btnFindOffers.doClick();

					}
				}
			}
		});
	}

	private void configurePropColumnModel() {
		propertyTable.getColumnModel().getColumn(0).setCellEditor(propCellEditor);
		propertyTable.getColumnModel().getColumn(1).setCellEditor(new DefaultCellEditor(new JTextField()));
		((DefaultCellEditor) propertyTable.getColumnModel().getColumn(0).getCellEditor()).setClickCountToStart(1);
		((DefaultCellEditor) propertyTable.getColumnModel().getColumn(1).getCellEditor()).setClickCountToStart(1);
	}

	private void configureTipVisibility() {
		if (tipPanel.isVisible() && tipHiderLabel.getIcon() != IMAGE_TIP_HIDE) {
			tipHiderLabel.setIcon(IMAGE_TIP_HIDE);
		}
		else if (tipHiderLabel.getIcon() != IMAGE_TIP_SHOW) {
			tipHiderLabel.setIcon(IMAGE_TIP_SHOW);
		}
	}

	private void configureDefaultTip(TipPanelInterface tipPanel) {

		tipPanel.setTips(new String[] {
		    "Feito um login, faa uma busca no servio de registro de ofertas preenchendo " +
		      "a tabela acima e clicando em \"Consultar\"",

		    "As propriedades padro do OpenBus j esto pr-definidas na caixa de seleo " +
		      "acessvel pela tabela de critrios de busca.",

		    "Aqui iro aparecer dicas de uso referentes ao n da rvore de resultados que " +
		      "voc selecionou. Use '<' e '>' daqui da esquerda para ver as demais dicas " +
		      "disponveis no contexto.",

		    "Use a barra cinza logo acima para ocultar esta ajuda ou faz-la reaparecer.",

		    "As listas LRU vo se lembrar do que voc fez recentemente e facilitar para voc repetir " +
		      "algo. Mas se voc quiser que as listas LRU \"se esqueam\" das suas ltimas aes, basta " +
		      "clicar com o boto direito sobre elas.",

		    "Se voc fizer mais de um login simultaneamente, voc pode escolher qual deles usar para realizar " +
		      "a sua consulta no servio de ofertas selecionando pelo radiobutton associado, l em cima.",

		    "Apenas depois de voc fazer alguma consulta, o servio de ofertas ser monitorado para lhe indicar " +
		      "em tempo real quando novas ofertas forem feitas, assim como quando as ofertas forem removidas, sem " +
		      "que voc precise ficar clicando em 'consultar' repetidamente para dar 'refresh' na rvore.",

		    "Se voc no quiser que esta aplicao faa login e uma busca automaticamente ao iniciar, clique com o boto " +
		      "direito do mouse sobre os botes [Login] e [Consultar]",

		    "Poupe tempo! Se seu ltimo login foi feito com chave privada, ao reiniciar a aplicao o login ser feito " +
		      "automaticamente, assim como uma primeira consulta. Para desativar este recurso, clique com o boto direito "
		      +
		      "sobre os botes [Login] e [Consultar]"
		}, new String[] {
		  "Problem?!"
		});

	}

	private void setFilterTextAndTooltip() {
		String filterText;
		if (scsTreeAndDetails.getCurrentPropertyFilter() == null
		  || scsTreeAndDetails.getCurrentPropertyFilter().length == 0)
			filterText = "Todas as ofertas disponveis";
		else
			filterText = "Filtro: " + getLRUText(scsTreeAndDetails.getCurrentPropertyFilter())
			  .replaceAll(Pattern.quote(String.valueOf(LRU_PROP_DIV_CHAR)), ", ");

		String cnnText;
		if (currentConnectionUsed == null) {
			cnnText = "Faa uma consulta";
		}
		else {
			cnnText = currentConnectionUsed.getEntity()
			  + "@" + currentConnectionUsed.getHost()
			  + ":" + currentConnectionUsed.getPort();
		}

		String tooltip;
		String text;
		if (filterIsShowingFilterText) {
			tooltip = cnnText;
			text = filterText;
		}
		else {
			tooltip = filterText;
			text = cnnText;
		}

		if (ServiceOfferRegistryObserversPool.getServiceOfferObserversSystemPropertyValue()) {
			tooltip = "(Usando observadores do servio de registro) " + tooltip;
			text = "(Ao Vivo) " + text;
		}

		if (!text.equals(lblFilter.getText()))
			lblFilter.setText(text);
		if (!tooltip.equals(lblFilter.getToolTipText()))
			lblFilter.setToolTipText(tooltip);
	}

	private ServiceProperty[] getPropertiesFromTable() {
		ArrayList<ServiceProperty> res = new ArrayList<ServiceProperty>(propertyTableModel.getRowCount());
		for (int i = 0; i < propertyTableModel.getRowCount(); i += 1) {
			boolean rowEmpty = isEmpty(propertyTableModel.getValueAt(i, 0))
			  && isEmpty(propertyTableModel.getValueAt(i, 1));
			if (!rowEmpty) {
				res.add(new ServiceProperty(
				  toStringNZ(propertyTableModel.getValueAt(i, 0)),
				  toStringNZ(propertyTableModel.getValueAt(i, 1))));
			}
		}
		Collections.sort(res, new Comparator<ServiceProperty>() {
			@Override
			public int compare(ServiceProperty o1, ServiceProperty o2) {
				return o1.name.compareTo(o2.name);
			}
		});
		return res.toArray(new ServiceProperty[] {});
	}

	private static final String toStringNZ(Object o) {
		if (o == null)
			return "";
		return o.toString().trim();
	}

	private void reinferRowNumbers() {
		if (cmbPropertyNameResizing)
			return;
		cmbPropertyNameResizing = true;
		try {
			recalculate: while (true) {
				int maxRowFilled = 0;
				for (int i = 0; i < propertyTableModel.getRowCount(); i += 1) {
					boolean rowEmpty = isEmpty(propertyTableModel.getValueAt(i, 0))
					  && isEmpty(propertyTableModel.getValueAt(i, 1));

					if (rowEmpty && i < propertyTableModel.getRowCount() - 1) {
						propertyTableModel.removeRow(i);
						continue recalculate;
					}

					if (!rowEmpty)
						maxRowFilled = i;
				}

				while (propertyTableModel.getRowCount() < (maxRowFilled + 2)) {
					propertyTableModel.addRow(new Object[] { "", "" });
				}

				while (propertyTableModel.getRowCount() > (maxRowFilled + 2)) {
					propertyTableModel.removeRow(propertyTableModel.getRowCount() - 1);
				}

				break;
			}
		}
		finally {
			cmbPropertyNameResizing = false;
		}
	}

	private static final boolean isEmpty(Object val) {
		if (val == null)
			return true;
		if (val.toString().isEmpty())
			return true;
		if (val.toString().trim().isEmpty())
			return true;
		return false;
	}

	private final String getLRUText() {
		return getLRUText(getPropertiesFromTable());
	}

	private final String getLRUText(ServiceProperty props[]) {
		StringBuilder res = new StringBuilder();
		for (ServiceProperty p : props) {
			if (res.length() > 0)
				res.append(LRU_PROP_DIV_CHAR);
			res.append(p.name);
			res.append("=");
			res.append(p.value);
		}
		return res.toString();
	}

	private final void configureByLRUText(String lruText) {
		String props[] = lruText.split(Pattern.quote(String.valueOf(LRU_PROP_DIV_CHAR)));

		Object[][] newData = new Object[props.length][];
		for (int i = 0; i < props.length; i += 1) {
			newData[i] = props[i].split("=", 2);
		}

		propertyTableModel.setDataVector(newData, columnNames);
		configurePropColumnModel();

	}

	@Override
	public void addCriteria(final String property, final String value) {
		if (property == null)
			throw new NullPointerException("property  nulo");
		if (value == null)
			throw new NullPointerException("value  nulo");
		if (property.trim().isEmpty() && value.trim().isEmpty())
			return;
		for (ServiceProperty prop : getPropertiesFromTable()) {
			if (prop.name.equals(property) && prop.value.equals(value))
				return;
		}
		SwingUtilities.invokeLater(new Runnable() {
			@Override
			public void run() {
				propertyTableModel.addRow(new Object[] { property, value });
			}
		});
	}

	@Override
	public boolean isOffersDefaultExpanded() {
		return persistedPrefs.getBoolean("isOffersDefaultExpanded", false);
	}

}
