package tecgraf.openbus.browser.scs_offers;

import java.awt.BorderLayout;
import java.awt.Color;
import java.awt.Component;
import java.awt.Container;
import java.awt.Dimension;
import java.awt.Font;
import java.awt.LayoutManager;
import java.awt.Rectangle;
import java.awt.event.ActionEvent;
import java.awt.event.ActionListener;
import java.awt.event.FocusEvent;
import java.awt.event.FocusListener;
import java.awt.event.KeyAdapter;
import java.awt.event.KeyEvent;
import java.awt.event.KeyListener;
import java.util.concurrent.atomic.AtomicInteger;

import javax.swing.BorderFactory;
import javax.swing.JComponent;
import javax.swing.JLabel;
import javax.swing.JLayeredPane;
import javax.swing.JOptionPane;
import javax.swing.JPanel;
import javax.swing.JProgressBar;
import javax.swing.JScrollPane;
import javax.swing.JSplitPane;
import javax.swing.JTextField;
import javax.swing.SwingUtilities;
import javax.swing.Timer;
import javax.swing.event.AncestorEvent;
import javax.swing.event.AncestorListener;
import javax.swing.event.DocumentEvent;
import javax.swing.event.DocumentListener;
import javax.swing.tree.DefaultMutableTreeNode;
import javax.swing.tree.DefaultTreeModel;
import javax.swing.tree.TreePath;

import net.miginfocom.swing.MigLayout;
import tecgraf.openbus.Connection;
import tecgraf.openbus.browser.ManagedConnection;
import tecgraf.openbus.browser.scs_offers.ServiceOfferRegistryObserversPool.ServiceOfferRegistryObserversPoolListener;
import tecgraf.openbus.browser.scs_offers.TreeStructureBuilder.TreeLoadStatusListener;
import tecgraf.openbus.browser.scs_offers.basic_nodes.ServiceOfferNode;
import tecgraf.openbus.core.v2_0.services.offer_registry.ServiceOfferDesc;
import tecgraf.openbus.core.v2_0.services.offer_registry.ServiceProperty;

public class SCSTreeAndDetailsComponent extends JComponent {
	private static final long serialVersionUID = 182205186035813454L;

	private final JSplitPane resultSplit;
	private final SCSTree resultTree;
	private final JProgressBar loadProgressBar;
	private final JPanel detailsPanel;
	private final JPanel detailsSplit;
	private ServiceProperty currentPropertyFilter[] = null;
	private final TreeStructureBuilder treeStructureBuilder;

	private final JTextField txtSearch;
	private final JLabel lblOutOfQueryEntrance;
	private final JLabel lblOutOfQueryRemoval;

	private final AtomicInteger outOfQueryEntranceCounter = new AtomicInteger(0);
	private final AtomicInteger outOfQueryEntranceAnimCountdown = new AtomicInteger(0);
	private final AtomicInteger outOfQueryRemovalCounter = new AtomicInteger(0);
	private final AtomicInteger outOfQueryRemovalAnimCountdown = new AtomicInteger(0);

	private FindServicesQueryControlInterface findServicesQueryControlInterfaceImpl = null;

	private final ServiceOfferRegistryObserversPoolListener serviceOfferRegistryObserversPoolListener =
	  new ServiceOfferRegistryObserversPoolListener() {
		  @Override
		  public void offerRegistered(ManagedConnection cnn, ServiceOfferDesc offer) {
			  if (currentPropertyFilter != null && currentPropertyFilter.length > 0) {
				  for (ServiceProperty filter : currentPropertyFilter) {
					  boolean found = false;
					  for (ServiceProperty prop : offer.properties) {
						  if (prop.name.equals(filter.name) && prop.value.equals(filter.value)) {
							  found = true;
							  break;
						  }
					  }
					  if (!found) {
						  return;
					  }
				  }
			  }
			  resultTree.addServiceOffer(cnn, offer, (NodeWithTreeReference) resultTree.getModel().getRoot());
		  }
	  };

	private final TreeLoadStatusListener myTreeLoadStatusListener = new TreeLoadStatusListener() {
		@Override
		public void startedQuery() {
			SwingUtilities.invokeLater(new Runnable() {
				@Override
				public void run() {
					loadProgressBar.setIndeterminate(true);
					loadProgressBar.setVisible(true);
					loadProgressBar.setMaximum(0);
					loadProgressBar.setMinimum(0);
					loadProgressBar.setValue(0);
					loadProgressBar.setVisible(true);
					loadProgressBar.setString("Interrogando Openbus...");
				};
			});
		}

		@Override
		public void startedProcessing(final int total) {
			SwingUtilities.invokeLater(new Runnable() {
				@Override
				public void run() {
					if (total == 0) {
						loadProgressBar.setVisible(false);
						JOptionPane.showMessageDialog(SCSTreeAndDetailsComponent.this, "Nada foi encontrado com este critrio",
						  "OfferRegistry", JOptionPane.INFORMATION_MESSAGE);
					}
					else {
						loadProgressBar.setIndeterminate(false);
						loadProgressBar.setMaximum(total);
						loadProgressBar.setValue(0);
						loadProgressBar.setString("Interrogando oferta 0/" + total + "...");
					}
				}
			});
		}

		@Override
		public void setProgress(final int progress) {
			SwingUtilities.invokeLater(new Runnable() {
				@Override
				public void run() {
					if (progress == loadProgressBar.getMaximum()) {
						loadProgressBar.setVisible(false);
					}
					else {
						loadProgressBar.setValue(progress);
						loadProgressBar.setString("Interrogando oferta " + progress + "/" + loadProgressBar.getMaximum() + "...");
					}
				}
			});
		}

		@Override
		public void error(final Throwable error) {
			error.printStackTrace(System.err);
			SwingUtilities.invokeLater(new Runnable() {
				@Override
				public void run() {
					loadProgressBar.setVisible(false);
					JOptionPane.showMessageDialog(SCSTreeAndDetailsComponent.this, "Erro consultando Openbus: "
					  + error.getClass().getName() + ": " + error.getMessage(), "Erro", JOptionPane.ERROR_MESSAGE);
				}
			});
		}

		@Override
		public void outOfQueryNewOffer() {
			synchronized (outOfQueryEntranceAnimCountdown) {
				outOfQueryEntranceCounter.incrementAndGet();
				outOfQueryEntranceAnimCountdown.set(50);
			}
			if (!animTimer.isRunning())
				animTimer.start();
		}

		@Override
		public void outOfQueryRemovedOffer() {
			synchronized (outOfQueryRemovalAnimCountdown) {
				outOfQueryRemovalCounter.incrementAndGet();
				outOfQueryRemovalAnimCountdown.set(50);
			}
			if (!animTimer.isRunning())
				animTimer.start();
		}

	};

	private final Timer animTimer = new Timer(100, new ActionListener() {
		@Override
		public void actionPerformed(ActionEvent e) {
			if (getParent() == null || !isVisible())
				return;

			boolean entranceVisible;
			synchronized (outOfQueryEntranceAnimCountdown) {
				entranceVisible = outOfQueryEntranceAnimCountdown.get() > 0;
				if (lblOutOfQueryEntrance.isVisible() != entranceVisible)
					lblOutOfQueryEntrance.setVisible(entranceVisible);

				if (entranceVisible) {
					String outOfQueryEntranceString = "+" + outOfQueryEntranceCounter.get();
					if (!outOfQueryEntranceString.equals(lblOutOfQueryEntrance.getText()))
						lblOutOfQueryEntrance.setText(outOfQueryEntranceString);

					Color entranceColor = new Color(0f, 0f, 0f, outOfQueryEntranceAnimCountdown.getAndDecrement() / 50f);
					if (!lblOutOfQueryEntrance.getForeground().equals(entranceColor))
						lblOutOfQueryEntrance.setForeground(entranceColor);
				}
				else
					outOfQueryEntranceCounter.set(0);
			}

			boolean removalVisible;
			synchronized (outOfQueryRemovalAnimCountdown) {
				removalVisible = outOfQueryRemovalAnimCountdown.get() > 0;
				if (lblOutOfQueryRemoval.isVisible() != removalVisible)
					lblOutOfQueryRemoval.setVisible(removalVisible);

				if (removalVisible) {
					String outOfQueryremovalString = "-" + outOfQueryRemovalCounter.get();
					if (!outOfQueryremovalString.equals(lblOutOfQueryRemoval.getText()))
						lblOutOfQueryRemoval.setText(outOfQueryremovalString);

					Color removalColor = new Color(0f, 0f, 0f, outOfQueryRemovalAnimCountdown.getAndDecrement() / 50f);
					if (!lblOutOfQueryRemoval.getForeground().equals(removalColor))
						lblOutOfQueryRemoval.setForeground(removalColor);
				}
				else
					outOfQueryRemovalCounter.set(0);
			}

			if (!entranceVisible && !removalVisible) {
				animTimer.stop();
			}

		}
	});

	private final FindServicesQueryControlInterface queryControlDelegate = new FindServicesQueryControlInterface() {
		@Override
		public void addCriteria(String property, String value) {
			if (getQueryControlImpl() != null)
				getQueryControlImpl().addCriteria(property, value);
		}

		@Override
		public boolean isOffersDefaultExpanded() {
			if (getQueryControlImpl() != null)
				return getQueryControlImpl().isOffersDefaultExpanded();
			return false;
		}
	};

	public SCSTreeAndDetailsComponent() {
		super();

		resultTree = new SCSTree(this);

		treeStructureBuilder = new TreeStructureBuilder(queryControlDelegate, resultTree, myTreeLoadStatusListener);

		addAncestorListener(new AncestorListener() {
			@Override
			public void ancestorRemoved(AncestorEvent event) {
				ServiceOfferRegistryObserversPool.getSingleton().removeListener(serviceOfferRegistryObserversPoolListener);
			}

			@Override
			public void ancestorMoved(AncestorEvent event) {
			}

			@Override
			public void ancestorAdded(AncestorEvent event) {
				ServiceOfferRegistryObserversPool.getSingleton().addListener(serviceOfferRegistryObserversPoolListener);
			}
		});

		final JScrollPane scrTree = new JScrollPane(resultTree);
		loadProgressBar = new JProgressBar(JProgressBar.HORIZONTAL);
		final JLayeredPane resultLayers = new JLayeredPane();
		resultLayers.setLayout(new LayoutManager() {
			@Override
			public void removeLayoutComponent(Component comp) {
			}

			@Override
			public void addLayoutComponent(String name, Component comp) {
			}

			@Override
			public Dimension preferredLayoutSize(Container parent) {
				return scrTree.getPreferredSize();
			}

			@Override
			public Dimension minimumLayoutSize(Container parent) {
				return new Dimension(350, 0);
			}

			@Override
			public void layoutContainer(Container parent) {
				scrTree.setBounds(parent.getBounds());
				txtSearch.setBounds(new Rectangle(
				  parent.getWidth() - 170,
				  3,
				  150,
				  20
				  ));

				loadProgressBar.setBounds(new Rectangle(
				  10, parent.getBounds().height - 25,
				  parent.getWidth() - 20,
				  15
				  ));
				lblOutOfQueryEntrance.setBounds(new Rectangle(
				  5,
				  parent.getBounds().height - 50,
				  60, 32
				  ));
				lblOutOfQueryRemoval.setBounds(new Rectangle(
				  parent.getBounds().width - 80,
				  parent.getBounds().height - 50,
				  60, 32
				  ));
			}
		});

		txtSearch = new JTextField();
		txtSearch
		  .setToolTipText("Digite o que quer encontrar para que seja localizado na rvore para voc. Ns de DataService no sero buscados.");
		txtSearch.setVisible(false);
		txtSearch.setBorder(BorderFactory.createLineBorder(Color.gray, 1));
		txtSearch.addFocusListener(new FocusListener() {
			@Override
			public void focusLost(FocusEvent e) {
				txtSearch.setVisible(false);
			}

			@Override
			public void focusGained(FocusEvent e) {
				contractAllResultTree();
			}
		});
		txtSearch.addKeyListener(getSearchKeyListener());
		txtSearch.getDocument().addDocumentListener(getSearchDocumentListener());
		resultTree.addKeyListener(new KeyAdapter() {
			@Override
			public void keyTyped(KeyEvent e) {
				if (Character.isLetterOrDigit(e.getKeyChar())) {
					showSearchField();
					txtSearch.dispatchEvent(e);
					e.consume();
				}
			}
		});

		lblOutOfQueryEntrance = new JLabel("+0");
		lblOutOfQueryEntrance.setFont(new Font(Font.MONOSPACED, Font.BOLD, 16));
		lblOutOfQueryEntrance.setOpaque(false);
		lblOutOfQueryEntrance.setVisible(false);
		lblOutOfQueryEntrance.setForeground(new Color(0f, 0f, 0f, 0.5f));

		lblOutOfQueryRemoval = new JLabel("-0");
		lblOutOfQueryRemoval.setFont(new Font(Font.MONOSPACED, Font.BOLD, 16));
		lblOutOfQueryRemoval.setOpaque(false);
		lblOutOfQueryRemoval.setVisible(false);
		lblOutOfQueryRemoval.setForeground(new Color(0f, 0f, 0f, 0.5f));
		lblOutOfQueryRemoval.setAlignmentX(JLabel.RIGHT_ALIGNMENT);
		lblOutOfQueryRemoval.setHorizontalAlignment(JLabel.RIGHT);

		resultLayers.add(scrTree);
		resultLayers.setLayer(scrTree, 0);
		resultLayers.add(loadProgressBar, 0);
		resultLayers.setLayer(loadProgressBar, 1);
		resultLayers.add(lblOutOfQueryEntrance, 0);
		resultLayers.setLayer(lblOutOfQueryEntrance, 2);
		resultLayers.add(lblOutOfQueryRemoval, 0);
		resultLayers.setLayer(lblOutOfQueryRemoval, 2);
		resultLayers.add(txtSearch, 0);
		resultLayers.setLayer(txtSearch, 3);

		loadProgressBar.setVisible(false);
		loadProgressBar.setIndeterminate(true);
		loadProgressBar.setStringPainted(true);
		loadProgressBar.setFont(new Font(Font.MONOSPACED, Font.PLAIN, 9));
		loadProgressBar.setToolTipText("Executando consulta...");

		detailsSplit = new JPanel();
		detailsSplit.setLayout(new MigLayout("", "0[grow]0", "0[grow]0"));

		detailsPanel = new JPanel();
		detailsPanel.setLayout(new MigLayout("", "0[grow, fill]0", "0[grow, fill]0"));
		detailsSplit.add(detailsPanel, "flowy, cell 0 0, grow, gap 0");

		resultSplit = new JSplitPane(JSplitPane.HORIZONTAL_SPLIT);

		resultSplit.setOneTouchExpandable(true);
		resultSplit.setContinuousLayout(true);
		resultSplit.setLeftComponent(resultLayers);
		resultSplit.setRightComponent(detailsSplit);
		resultSplit.setDividerLocation(350);

		setLayout(new BorderLayout());
		add(resultSplit, BorderLayout.CENTER);

	}

	private DocumentListener getSearchDocumentListener() {
		return new DocumentListener() {
			@Override
			public void removeUpdate(DocumentEvent e) {
				doSeachInTree();
			}

			@Override
			public void insertUpdate(DocumentEvent e) {
				doSeachInTree();
			}

			@Override
			public void changedUpdate(DocumentEvent e) {
				doSeachInTree();
			}
		};
	}

	private void doSeachInTree() {
		SwingUtilities.invokeLater(new Runnable() {
			DefaultTreeModel model = ((DefaultTreeModel) resultTree.getModel());
			String textToSearch = null;

			@Override
			public void run() {
				textToSearch = txtSearch.getText().trim().toLowerCase();
				if (textToSearch.isEmpty()) {
					contractAllResultTree();
					return;
				}

				if (resultTree.getSelectionRows() != null && resultTree.getSelectionRows().length > 0) {
					if (!nodeHasText(resultTree.getPathForRow(resultTree.getSelectionRows()[0]), textToSearch)) {
						resultTree.clearSelection();
					}
				}

				doSearchInTree(new TreePath(model.getRoot()));
			}

			private boolean doSearchInTree(TreePath path) {

				boolean myRes = false;
				DefaultMutableTreeNode node = (DefaultMutableTreeNode) path.getLastPathComponent();
				if (node != model.getRoot()) {
					if (nodeHasText(path, textToSearch)) {
						doSelectOnSearch(path);
						myRes = true;
					}
				}

				boolean childRes = false;
				if (!(node instanceof AsyncExpandableTreeNode)) {
					for (int i = 0; i < node.getChildCount(); i += 1) {
						childRes |= doSearchInTree(path.pathByAddingChild(node.getChildAt(i)));
					}
				}

				if (!childRes && node != model.getRoot())
					resultTree.collapsePath(path);

				return myRes || childRes;
			}

		});
	}

	private boolean nodeHasText(TreePath path, String textToSearch) {
		DefaultMutableTreeNode node = (DefaultMutableTreeNode) path.getLastPathComponent();
		if (node.toString().toLowerCase().contains(textToSearch)) {
			return true;
		}
		else if (node instanceof NodeWithSearchCapability
		  && ((NodeWithSearchCapability) node).hasString(textToSearch)) {
			return true;
		}
		else if (node.getUserObject() != null
		  && node.getUserObject() instanceof NodeWithSearchCapability
		  && ((NodeWithSearchCapability) node.getUserObject()).hasString(textToSearch)) {
			return true;
		}
		return false;
	}

	private void doSelectOnSearch(TreePath path) {
		resultTree.expandPath(path.getParentPath());
		if (resultTree.getSelectionCount() == 0) {
			resultTree.setSelectionPath(path);
			Rectangle bounds = resultTree.getPathBounds(path);
			bounds.x = 0;
			resultTree.scrollRectToVisible(bounds);
		}
	}

	private KeyListener getSearchKeyListener() {
		return new KeyAdapter() {
			@Override
			public void keyPressed(KeyEvent e) {
				if (e.getKeyCode() == KeyEvent.VK_ESCAPE) {
					e.consume();
					txtSearch.setVisible(false);
				}
				if (e.getKeyCode() == KeyEvent.VK_DOWN || e.getKeyCode() == KeyEvent.VK_ENTER) {
					e.consume();
					doSearchForNext(1);
				}
				if (e.getKeyCode() == KeyEvent.VK_UP) {
					e.consume();
					doSearchForNext(-1);
				}
			}
		};
	}

	protected void doSearchForNext(int interval) {
		if (resultTree.getSelectionRows() == null || resultTree.getSelectionRows().length == 0) {
			doSeachInTree();
			return;
		}
		int selectedRow = resultTree.getSelectionRows()[0];
		int currentRow = selectedRow + interval;
		if (currentRow < 0)
			currentRow = resultTree.getRowCount() + currentRow;
		else if (currentRow >= resultTree.getRowCount())
			currentRow = currentRow - resultTree.getRowCount();

		String textToSearch = txtSearch.getText().trim().toLowerCase();
		while (currentRow != selectedRow) {
			TreePath path = resultTree.getPathForRow(currentRow);
			if (nodeHasText(path, textToSearch)) {
				resultTree.setSelectionPath(path);
				Rectangle bounds = resultTree.getPathBounds(path);
				bounds.x = 0;
				resultTree.scrollRectToVisible(bounds);
				return;
			}
			currentRow = currentRow + interval;
			if (currentRow < 0)
				currentRow = resultTree.getRowCount() + currentRow;
			else if (currentRow >= resultTree.getRowCount())
				currentRow = currentRow - resultTree.getRowCount();
		}
	}

	public void showSearchField() {
		txtSearch.setVisible(true);
		txtSearch.requestFocus();
		doSeachInTree();
	}

	void setDetails(final JPanel details) {
		detailsPanel.removeAll();
		if (details != null)
			detailsPanel.add(details, "cell 0 0, grow");

		detailsPanel.validate();
		detailsPanel.repaint();
	}

	public FindServicesQueryControlInterface getQueryControlImpl() {
		return findServicesQueryControlInterfaceImpl;
	}

	public void setFindServicesQueryControlInterface(FindServicesQueryControlInterface impl) {
		this.findServicesQueryControlInterfaceImpl = impl;
	}

	public SCSTree getTree() {
		return resultTree;
	}

	protected void addPanelToBottomDetails(Component component) {
		detailsSplit.add(component, "cell 0 0,grow, gapy 0, hidemode 3");
	}

	/**
	 * @return the currentPropertyFilter
	 */
	public ServiceProperty[] getCurrentPropertyFilter() {
		return currentPropertyFilter;
	}

	/**
	 * @param currentPropertyFilter the currentPropertyFilter to set
	 */
	public void setCurrentPropertyFilter(ServiceProperty[] currentPropertyFilter) {
		this.currentPropertyFilter = currentPropertyFilter;
	}

	/**
	 * Remover o n da rvore informado, notificando a remoo de uma oferta.
	 * 
	 * @param serviceOfferNode Oferta a ser removida da rvore.
	 */
	public void removeOffer(ServiceOfferNode serviceOfferNode) {
		treeStructureBuilder.removeOffer(serviceOfferNode);
	}

	protected void addServiceOffer(ManagedConnection cnn, ServiceOfferDesc service, NodeWithTreeReference root) {
		treeStructureBuilder.addServiceOffer(cnn, service, root);
	}

	public void startSearch(Connection cnn, ServiceProperty[] filter) {

		((DefaultTreeModel) resultTree.getModel()).setRoot(new NodeWithTreeReference(resultTree, "SCS"));
		treeStructureBuilder.startLoadingTree(cnn, filter);

	}

	private void contractAllResultTree() {
		for (int i = 0; i < resultTree.getRowCount(); i += 1) {
			resultTree.collapseRow(i);
		}
	}
}
